"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateObservable = exports.deleteObservable = exports.bulkAddObservables = exports.addObservable = void 0;
var _uuid = require("uuid");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _constants = require("../../../common/constants");
var _domain = require("../../../common/types/domain");
var _api = require("../../../common/types/api");
var _runtime_types = require("../../common/runtime_types");
var _authorization = require("../../authorization");
var _utils = require("../../common/utils");
var _constants2 = require("../../common/constants");
var _validators = require("../validators");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ensureUpdateAuthorized = async (authorization, theCase) => {
  return authorization.ensureAuthorized({
    operation: _authorization.Operations.updateCase,
    entities: [{
      id: theCase.id,
      owner: theCase.attributes.owner
    }]
  });
};
const addObservable = async (caseId, params, clientArgs, casesClient) => {
  const {
    services: {
      caseService,
      licensingService,
      userActionService
    },
    authorization,
    user
  } = clientArgs;
  const hasPlatinumLicenseOrGreater = await licensingService.isAtLeastPlatinum();
  if (!hasPlatinumLicenseOrGreater) {
    throw _boom.default.forbidden('In order to assign observables to cases, you must be subscribed to an Elastic Platinum license');
  }
  licensingService.notifyUsage(_constants2.LICENSING_CASE_OBSERVABLES_FEATURE);
  try {
    var _retrievedCase$attrib;
    const paramArgs = (0, _runtime_types.decodeWithExcessOrThrow)(_api.AddObservableRequestRt)(params);
    const retrievedCase = await caseService.getCase({
      id: caseId
    });
    await ensureUpdateAuthorized(authorization, retrievedCase);
    await (0, _validators.validateObservableTypeKeyExists)(casesClient, {
      caseOwner: retrievedCase.attributes.owner,
      observableTypeKey: params.observable.typeKey
    });
    (0, _validators.validateObservableValue)(paramArgs.observable.typeKey, paramArgs.observable.value);
    const currentObservables = (_retrievedCase$attrib = retrievedCase.attributes.observables) !== null && _retrievedCase$attrib !== void 0 ? _retrievedCase$attrib : [];
    if (currentObservables.length === _constants.MAX_OBSERVABLES_PER_CASE) {
      throw _boom.default.forbidden(`Max ${_constants.MAX_OBSERVABLES_PER_CASE} observables per case is allowed.`);
    }
    const updatedObservables = [...currentObservables, {
      ...paramArgs.observable,
      id: (0, _uuid.v4)(),
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    }];
    (0, _validators.validateDuplicatedObservablesInRequest)({
      requestFields: updatedObservables
    });
    const updatedCase = await caseService.patchCase({
      caseId: retrievedCase.id,
      originalCase: retrievedCase,
      updatedAttributes: {
        observables: updatedObservables
      }
    });
    await userActionService.creator.createUserAction({
      userAction: {
        type: _domain.UserActionTypes.observables,
        caseId: retrievedCase.id,
        owner: retrievedCase.attributes.owner,
        user,
        payload: {
          observables: {
            count: 1,
            actionType: 'add'
          }
        }
      }
    });
    const res = (0, _utils.flattenCaseSavedObject)({
      savedObject: {
        ...retrievedCase,
        ...updatedCase,
        attributes: {
          ...retrievedCase.attributes,
          ...(updatedCase === null || updatedCase === void 0 ? void 0 : updatedCase.attributes)
        },
        references: retrievedCase.references
      }
    });
    return (0, _runtime_types.decodeOrThrow)(_domain.CaseRt)(res);
  } catch (error) {
    throw _boom.default.badRequest(`Failed to add observable: ${error}`);
  }
};
exports.addObservable = addObservable;
const updateObservable = async (caseId, observableId, params, clientArgs, casesClient) => {
  const {
    services: {
      caseService,
      licensingService,
      userActionService
    },
    authorization,
    user
  } = clientArgs;
  const hasPlatinumLicenseOrGreater = await licensingService.isAtLeastPlatinum();
  if (!hasPlatinumLicenseOrGreater) {
    throw _boom.default.forbidden('In order to update observables in cases, you must be subscribed to an Elastic Platinum license');
  }
  licensingService.notifyUsage(_constants2.LICENSING_CASE_OBSERVABLES_FEATURE);
  try {
    var _retrievedCase$attrib2;
    const paramArgs = (0, _runtime_types.decodeWithExcessOrThrow)(_api.UpdateObservableRequestRt)(params);
    const retrievedCase = await caseService.getCase({
      id: caseId
    });
    await ensureUpdateAuthorized(authorization, retrievedCase);
    const currentObservables = (_retrievedCase$attrib2 = retrievedCase.attributes.observables) !== null && _retrievedCase$attrib2 !== void 0 ? _retrievedCase$attrib2 : [];
    const observableIndex = currentObservables.findIndex(observable => observable.id === observableId);
    if (observableIndex === -1) {
      throw _boom.default.notFound(`Failed to update observable: observable id ${observableId} not found`);
    }
    (0, _validators.validateObservableValue)(currentObservables[observableIndex].typeKey, paramArgs.observable.value);
    const updatedObservables = [...currentObservables];
    updatedObservables[observableIndex] = {
      ...updatedObservables[observableIndex],
      ...paramArgs.observable,
      updatedAt: new Date().toISOString()
    };
    (0, _validators.validateDuplicatedObservablesInRequest)({
      requestFields: updatedObservables
    });
    const updatedCase = await caseService.patchCase({
      caseId: retrievedCase.id,
      originalCase: retrievedCase,
      updatedAttributes: {
        observables: updatedObservables
      }
    });
    await userActionService.creator.createUserAction({
      userAction: {
        type: _domain.UserActionTypes.observables,
        caseId: retrievedCase.id,
        owner: retrievedCase.attributes.owner,
        user,
        payload: {
          observables: {
            count: 1,
            actionType: 'update'
          }
        }
      }
    });
    const res = (0, _utils.flattenCaseSavedObject)({
      savedObject: {
        ...retrievedCase,
        ...updatedCase,
        attributes: {
          ...retrievedCase.attributes,
          ...(updatedCase === null || updatedCase === void 0 ? void 0 : updatedCase.attributes)
        },
        references: retrievedCase.references
      }
    });
    return (0, _runtime_types.decodeOrThrow)(_domain.CaseRt)(res);
  } catch (error) {
    throw _boom.default.badRequest(`Failed to update observable: ${error}`);
  }
};
exports.updateObservable = updateObservable;
const deleteObservable = async (caseId, observableId, clientArgs, casesClient) => {
  const {
    services: {
      caseService,
      licensingService,
      userActionService
    },
    authorization,
    user
  } = clientArgs;
  const hasPlatinumLicenseOrGreater = await licensingService.isAtLeastPlatinum();
  if (!hasPlatinumLicenseOrGreater) {
    throw _boom.default.forbidden('In order to delete observables from cases, you must be subscribed to an Elastic Platinum license');
  }
  licensingService.notifyUsage(_constants2.LICENSING_CASE_OBSERVABLES_FEATURE);
  try {
    const retrievedCase = await caseService.getCase({
      id: caseId
    });
    await ensureUpdateAuthorized(authorization, retrievedCase);
    const updatedObservables = retrievedCase.attributes.observables.filter(observable => observable.id !== observableId);

    // NOTE: same length of observables pre and post filter means that the observable id has not been found
    if (updatedObservables.length === retrievedCase.attributes.observables.length) {
      throw _boom.default.notFound(`Failed to delete observable: observable id ${observableId} not found`);
    }
    await caseService.patchCase({
      caseId: retrievedCase.id,
      originalCase: retrievedCase,
      updatedAttributes: {
        observables: updatedObservables
      }
    });
    await userActionService.creator.createUserAction({
      userAction: {
        type: _domain.UserActionTypes.observables,
        caseId: retrievedCase.id,
        owner: retrievedCase.attributes.owner,
        user,
        payload: {
          observables: {
            count: 1,
            actionType: 'delete'
          }
        }
      }
    });
  } catch (error) {
    throw _boom.default.badRequest(`Failed to delete observable id: ${observableId}: ${error}`);
  }
};
exports.deleteObservable = deleteObservable;
const bulkAddObservables = async (params, clientArgs, casesClient) => {
  const {
    services: {
      caseService,
      licensingService,
      userActionService
    },
    authorization,
    user
  } = clientArgs;
  const hasPlatinumLicenseOrGreater = await licensingService.isAtLeastPlatinum();
  if (!hasPlatinumLicenseOrGreater) {
    throw _boom.default.forbidden('In order to assign observables to cases, you must be subscribed to an Elastic Platinum license');
  }
  licensingService.notifyUsage(_constants2.LICENSING_CASE_OBSERVABLES_FEATURE);
  try {
    var _retrievedCase$attrib3;
    const paramArgs = (0, _runtime_types.decodeWithExcessOrThrow)(_api.BulkAddObservablesRequestRt)(params);
    const retrievedCase = await caseService.getCase({
      id: paramArgs.caseId
    });
    await ensureUpdateAuthorized(authorization, retrievedCase);
    await Promise.all(params.observables.map(observable => (0, _validators.validateObservableTypeKeyExists)(casesClient, {
      caseOwner: retrievedCase.attributes.owner,
      observableTypeKey: observable.typeKey
    })));
    const currentObservables = (_retrievedCase$attrib3 = retrievedCase.attributes.observables) !== null && _retrievedCase$attrib3 !== void 0 ? _retrievedCase$attrib3 : [];
    const updatedObservablesMap = new Map();
    currentObservables.forEach(observable => {
      (0, _utils2.processObservables)(updatedObservablesMap, observable);
    });
    paramArgs.observables.forEach(observable => (0, _utils2.processObservables)(updatedObservablesMap, observable));
    const finalObservables = Array.from(updatedObservablesMap.values()).slice(0, _constants.MAX_OBSERVABLES_PER_CASE);
    const updatedCase = await caseService.patchCase({
      caseId: retrievedCase.id,
      originalCase: retrievedCase,
      updatedAttributes: {
        observables: finalObservables
      }
    });
    const newObservablesCount = finalObservables.length - currentObservables.length;
    await userActionService.creator.createUserAction({
      userAction: {
        type: _domain.UserActionTypes.observables,
        caseId: retrievedCase.id,
        owner: retrievedCase.attributes.owner,
        user,
        payload: {
          observables: {
            count: newObservablesCount,
            actionType: 'add'
          }
        }
      }
    });
    const res = (0, _utils.flattenCaseSavedObject)({
      savedObject: {
        ...retrievedCase,
        ...updatedCase,
        attributes: {
          ...retrievedCase.attributes,
          ...(updatedCase === null || updatedCase === void 0 ? void 0 : updatedCase.attributes)
        },
        references: retrievedCase.references
      }
    });
    return (0, _runtime_types.decodeOrThrow)(_domain.CaseRt)(res);
  } catch (error) {
    throw _boom.default.badRequest(`Failed to add observable: ${error}`);
  }
};
exports.bulkAddObservables = bulkAddObservables;