"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configSchema = exports.EnabledActionTypes = exports.DEFAULT_QUEUED_MAX = exports.AllowedHosts = void 0;
exports.getValidatedConfig = getValidatedConfig;
var _configSchema = require("@kbn/config-schema");
var _common = require("../common");
var _parse_date = require("./lib/parse_date");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
let AllowedHosts = exports.AllowedHosts = /*#__PURE__*/function (AllowedHosts) {
  AllowedHosts["Any"] = "*";
  return AllowedHosts;
}({});
let EnabledActionTypes = exports.EnabledActionTypes = /*#__PURE__*/function (EnabledActionTypes) {
  EnabledActionTypes["Any"] = "*";
  return EnabledActionTypes;
}({});
const MAX_MAX_ATTEMPTS = 10;
const MIN_MAX_ATTEMPTS = 1;
const MIN_QUEUED_MAX = 1;
const DEFAULT_QUEUED_MAX = exports.DEFAULT_QUEUED_MAX = 1000000;
const validRateLimiterConnectorTypeIds = new Set(['email']);
const preconfiguredActionSchema = _configSchema.schema.object({
  name: _configSchema.schema.string({
    minLength: 1
  }),
  actionTypeId: _configSchema.schema.string({
    minLength: 1
  }),
  config: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any(), {
    defaultValue: {}
  }),
  secrets: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any(), {
    defaultValue: {}
  }),
  exposeConfig: _configSchema.schema.maybe(_configSchema.schema.boolean({
    defaultValue: false
  }))
});
const customHostSettingsSchema = _configSchema.schema.object({
  url: _configSchema.schema.string({
    minLength: 1
  }),
  smtp: _configSchema.schema.maybe(_configSchema.schema.object({
    ignoreTLS: _configSchema.schema.maybe(_configSchema.schema.boolean()),
    requireTLS: _configSchema.schema.maybe(_configSchema.schema.boolean())
  })),
  ssl: _configSchema.schema.maybe(_configSchema.schema.object({
    verificationMode: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('none'), _configSchema.schema.literal('certificate'), _configSchema.schema.literal('full')], {
      defaultValue: 'full'
    })),
    certificateAuthoritiesFiles: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string({
      minLength: 1
    }), _configSchema.schema.arrayOf(_configSchema.schema.string({
      minLength: 1
    }), {
      minSize: 1
    })])),
    certificateAuthoritiesData: _configSchema.schema.maybe(_configSchema.schema.string({
      minLength: 1
    }))
  }))
});
const connectorTypeSchema = _configSchema.schema.object({
  id: _configSchema.schema.string(),
  maxAttempts: _configSchema.schema.maybe(_configSchema.schema.number({
    min: MIN_MAX_ATTEMPTS,
    max: MAX_MAX_ATTEMPTS
  }))
});

// We leverage enabledActionTypes list by allowing the other plugins to overwrite it by using "setEnabledConnectorTypes" in the plugin setup.
// The list can be overwritten only if it's not already been set in the config.
const enabledConnectorTypesSchema = _configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.literal(EnabledActionTypes.Any)]), {
  defaultValue: [AllowedHosts.Any]
});
const rateLimiterSchema = _configSchema.schema.recordOf(_configSchema.schema.string({
  validate: value => {
    if (!validRateLimiterConnectorTypeIds.has(value)) {
      return `Rate limiter configuration for connector type "${value}" is not supported. Supported types: ${Array.from(validRateLimiterConnectorTypeIds).join(', ')}`;
    }
  }
}), _configSchema.schema.object({
  lookbackWindow: _configSchema.schema.string({
    defaultValue: '15m',
    validate: _parse_date.validateDuration
  }),
  limit: _configSchema.schema.number({
    defaultValue: 500,
    min: 1,
    max: 5000
  })
}));
const configSchema = exports.configSchema = _configSchema.schema.object({
  allowedHosts: _configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.string({
    hostname: true
  }), _configSchema.schema.literal(AllowedHosts.Any)]), {
    defaultValue: [AllowedHosts.Any]
  }),
  enabledActionTypes: enabledConnectorTypesSchema,
  preconfiguredAlertHistoryEsIndex: _configSchema.schema.boolean({
    defaultValue: false
  }),
  preconfigured: _configSchema.schema.recordOf(_configSchema.schema.string(), preconfiguredActionSchema, {
    defaultValue: {},
    validate: validatePreconfigured
  }),
  proxyUrl: _configSchema.schema.maybe(_configSchema.schema.string()),
  proxyHeaders: _configSchema.schema.maybe(_configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.string())),
  proxyBypassHosts: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string({
    hostname: true
  }))),
  proxyOnlyHosts: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string({
    hostname: true
  }))),
  ssl: _configSchema.schema.maybe(_configSchema.schema.object({
    verificationMode: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('none'), _configSchema.schema.literal('certificate'), _configSchema.schema.literal('full')], {
      defaultValue: 'full'
    })),
    proxyVerificationMode: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal('none'), _configSchema.schema.literal('certificate'), _configSchema.schema.literal('full')], {
      defaultValue: 'full'
    }))
  })),
  maxResponseContentLength: _configSchema.schema.byteSize({
    defaultValue: '1mb'
  }),
  responseTimeout: _configSchema.schema.duration({
    defaultValue: '60s'
  }),
  customHostSettings: _configSchema.schema.maybe(_configSchema.schema.arrayOf(customHostSettingsSchema)),
  microsoftGraphApiUrl: _configSchema.schema.string({
    defaultValue: _common.DEFAULT_MICROSOFT_GRAPH_API_URL
  }),
  microsoftGraphApiScope: _configSchema.schema.string({
    defaultValue: _common.DEFAULT_MICROSOFT_GRAPH_API_SCOPE
  }),
  microsoftExchangeUrl: _configSchema.schema.string({
    defaultValue: _common.DEFAULT_MICROSOFT_EXCHANGE_URL
  }),
  email: _configSchema.schema.maybe(_configSchema.schema.object({
    domain_allowlist: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
    recipient_allowlist: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string(), {
      minSize: 1
    })),
    services: _configSchema.schema.maybe(_configSchema.schema.object({
      enabled: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.literal('google-mail'), _configSchema.schema.literal('microsoft-exchange'), _configSchema.schema.literal('microsoft-outlook'), _configSchema.schema.literal('amazon-ses'), _configSchema.schema.literal('elastic-cloud'), _configSchema.schema.literal('other'), _configSchema.schema.literal('*')]), {
        minSize: 1
      })),
      ses: _configSchema.schema.maybe(_configSchema.schema.object({
        host: _configSchema.schema.string({
          minLength: 1
        }),
        port: _configSchema.schema.number({
          min: 1,
          max: 65535
        })
      }))
    }, {
      validate: obj => {
        if (obj && Object.keys(obj).length === 0) {
          return 'email.services.enabled or email.services.ses must be defined';
        }
      }
    }))
  }, {
    validate: obj => {
      if (obj && Object.keys(obj).length === 0) {
        return 'email.domain_allowlist, email.recipient_allowlist, or email.services must be defined';
      }
      if (obj !== null && obj !== void 0 && obj.domain_allowlist && obj !== null && obj !== void 0 && obj.recipient_allowlist) {
        return 'email.domain_allowlist and email.recipient_allowlist can not be used at the same time';
      }
    }
  })),
  run: _configSchema.schema.maybe(_configSchema.schema.object({
    maxAttempts: _configSchema.schema.maybe(_configSchema.schema.number({
      min: MIN_MAX_ATTEMPTS,
      max: MAX_MAX_ATTEMPTS
    })),
    connectorTypeOverrides: _configSchema.schema.maybe(_configSchema.schema.arrayOf(connectorTypeSchema))
  })),
  enableFooterInEmail: _configSchema.schema.boolean({
    defaultValue: true
  }),
  queued: _configSchema.schema.maybe(_configSchema.schema.object({
    max: _configSchema.schema.maybe(_configSchema.schema.number({
      min: MIN_QUEUED_MAX,
      defaultValue: DEFAULT_QUEUED_MAX
    }))
  })),
  usage: _configSchema.schema.maybe(_configSchema.schema.object({
    url: _configSchema.schema.maybe(_configSchema.schema.string()),
    enabled: _configSchema.schema.maybe(_configSchema.schema.boolean()),
    ca: _configSchema.schema.maybe(_configSchema.schema.object({
      path: _configSchema.schema.string()
    }))
  })),
  webhook: _configSchema.schema.maybe(_configSchema.schema.object({
    ssl: _configSchema.schema.object({
      pfx: _configSchema.schema.object({
        enabled: _configSchema.schema.boolean({
          defaultValue: true
        })
      })
    })
  })),
  rateLimiter: _configSchema.schema.maybe(rateLimiterSchema)
});
// It would be nicer to add the proxyBypassHosts / proxyOnlyHosts restriction on
// simultaneous usage in the config validator directly, but there's no good way to express
// this relationship in the cloud config constraints, so we're doing it "live".
function getValidatedConfig(logger, originalConfig) {
  const proxyBypassHosts = originalConfig.proxyBypassHosts;
  const proxyOnlyHosts = originalConfig.proxyOnlyHosts;
  const proxyUrl = originalConfig.proxyUrl;
  if (proxyUrl) {
    try {
      new URL(proxyUrl);
    } catch (err) {
      logger.warn(`The configuration xpack.actions.proxyUrl: ${proxyUrl} is invalid.`);
    }
  }
  if (proxyBypassHosts && proxyOnlyHosts) {
    logger.warn('The configurations xpack.actions.proxyBypassHosts and xpack.actions.proxyOnlyHosts can not be used at the same time. The configuration xpack.actions.proxyOnlyHosts will be ignored.');
    const tmp = originalConfig;
    delete tmp.proxyOnlyHosts;
    return tmp;
  }
  return originalConfig;
}
const invalidActionIds = new Set(['', '__proto__', 'constructor']);
function validatePreconfigured(preconfigured) {
  // check for ids that should not be used
  for (const id of Object.keys(preconfigured)) {
    if (invalidActionIds.has(id)) {
      return `invalid preconfigured action id "${id}"`;
    }
  }

  // in case __proto__ was used as a preconfigured action id ...
  if (Object.getPrototypeOf(preconfigured) !== Object.getPrototypeOf({})) {
    return `invalid preconfigured action id "__proto__"`;
  }
}