"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ScheduledReportsService = void 0;
var _reportingServer = require("@kbn/reporting-server");
var _lodash = require("lodash");
var _saved_objects = require("../../saved_objects");
var _audit_events = require("../audit_events/audit_events");
var _constants = require("./constants");
var _transforms = require("./transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SCHEDULED_REPORT_ID_FIELD = 'scheduled_report_id';
const CREATED_AT_FIELD = 'created_at';
class ScheduledReportsService {
  constructor(auditLogger, userCanManageReporting, esClient, logger, responseFactory, savedObjectsClient, taskManager) {
    this.auditLogger = auditLogger;
    this.userCanManageReporting = userCanManageReporting;
    this.esClient = esClient;
    this.logger = logger;
    this.responseFactory = responseFactory;
    this.savedObjectsClient = savedObjectsClient;
    this.taskManager = taskManager;
  }
  static async build({
    logger,
    reportingCore,
    responseFactory,
    request
  }) {
    const esClient = await reportingCore.getEsClient();
    const auditLogger = await reportingCore.getAuditLogger(request);
    const savedObjectsClient = await reportingCore.getScopedSoClient(request);
    const taskManager = await reportingCore.getTaskManager();
    const userCanManageReporting = await reportingCore.canManageReportingForSpace(request);
    return new ScheduledReportsService(auditLogger, userCanManageReporting, esClient, logger, responseFactory, savedObjectsClient, taskManager);
  }
  async list({
    user,
    page = 1,
    size = _constants.DEFAULT_SCHEDULED_REPORT_LIST_SIZE
  }) {
    try {
      const username = this.getUsername(user);
      const response = await this.savedObjectsClient.find({
        type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
        page,
        perPage: size,
        ...(!this.userCanManageReporting ? {
          filter: `scheduled_report.attributes.createdBy: "${username}"`
        } : {})
      });
      if (!response) {
        return this.getEmptyListApiResponse(page, size);
      }
      const scheduledReportIdsAndName = response === null || response === void 0 ? void 0 : response.saved_objects.map(so => ({
        id: so.id,
        name: so.attributes.title
      }));
      if (!scheduledReportIdsAndName || scheduledReportIdsAndName.length === 0) {
        return this.getEmptyListApiResponse(page, size);
      }
      scheduledReportIdsAndName.forEach(({
        id,
        name
      }) => this.auditLog({
        action: _audit_events.ScheduledReportAuditAction.LIST,
        id,
        name
      }));
      const scheduledReportIds = scheduledReportIdsAndName.map(({
        id
      }) => id);
      let lastRunResponse;
      try {
        lastRunResponse = await this.esClient.asInternalUser.search({
          index: _reportingServer.REPORTING_DATA_STREAM_WILDCARD_WITH_LEGACY,
          size,
          _source: [CREATED_AT_FIELD],
          sort: [{
            [CREATED_AT_FIELD]: {
              order: 'desc'
            }
          }],
          query: {
            bool: {
              filter: [{
                terms: {
                  [SCHEDULED_REPORT_ID_FIELD]: scheduledReportIds
                }
              }]
            }
          },
          collapse: {
            field: SCHEDULED_REPORT_ID_FIELD
          }
        });
      } catch (error) {
        // if no scheduled reports have run yet, we will get an error from the collapse query
        // ignore these and return an empty last run
        this.logger.warn(`Error getting last run for scheduled reports: ${error.message}`);
      }
      let nextRunResponse;
      try {
        nextRunResponse = await this.taskManager.bulkGet(scheduledReportIds);
      } catch (error) {
        // swallow this error
        this.logger.warn(`Error getting next run for scheduled reports: ${error.message}`);
      }
      return (0, _transforms.transformListResponse)(this.logger, response, lastRunResponse, nextRunResponse);
    } catch (error) {
      throw this.responseFactory.customError({
        statusCode: 500,
        body: `Error listing scheduled reports: ${error.message}`
      });
    }
  }
  async bulkDisable({
    ids,
    user
  }) {
    try {
      let taskIdsToDisable = [];
      const bulkErrors = [];
      const disabledScheduledReportIds = new Set();
      const username = this.getUsername(user);
      const bulkGetResult = await this.savedObjectsClient.bulkGet(ids.map(id => ({
        id,
        type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE
      })));
      const scheduledReportSavedObjectsToUpdate = [];
      for (const so of bulkGetResult.saved_objects) {
        if (so.error) {
          bulkErrors.push({
            message: so.error.message,
            status: so.error.statusCode,
            id: so.id
          });
        } else {
          // check if user is allowed to update this scheduled report
          if (so.attributes.createdBy !== username && !this.userCanManageReporting) {
            var _so$attributes;
            bulkErrors.push({
              message: `Not found.`,
              status: 404,
              id: so.id
            });
            this.logger.warn(`User "${username}" attempted to disable scheduled report "${so.id}" created by "${so.attributes.createdBy}" without sufficient privileges.`);
            this.auditLog({
              action: _audit_events.ScheduledReportAuditAction.DISABLE,
              id: so.id,
              name: so === null || so === void 0 ? void 0 : (_so$attributes = so.attributes) === null || _so$attributes === void 0 ? void 0 : _so$attributes.title,
              error: new Error('Not found.')
            });
          } else if (so.attributes.enabled === false) {
            this.logger.debug(`Scheduled report ${so.id} is already disabled`);
            disabledScheduledReportIds.add(so.id);
          } else {
            this.auditLog({
              action: _audit_events.ScheduledReportAuditAction.DISABLE,
              id: so.id,
              name: so.attributes.title,
              outcome: 'unknown'
            });
            scheduledReportSavedObjectsToUpdate.push(so);
          }
        }
      }

      // nothing to update, return early
      if (scheduledReportSavedObjectsToUpdate.length > 0) {
        const bulkUpdateResult = await this.savedObjectsClient.bulkUpdate(scheduledReportSavedObjectsToUpdate.map(so => ({
          id: so.id,
          type: so.type,
          attributes: {
            enabled: false
          }
        })));
        for (const so of bulkUpdateResult.saved_objects) {
          if (so.error) {
            var _so$attributes2;
            bulkErrors.push({
              message: so.error.message,
              status: so.error.statusCode,
              id: so.id
            });
            this.auditLog({
              action: _audit_events.ScheduledReportAuditAction.DISABLE,
              id: so.id,
              name: so === null || so === void 0 ? void 0 : (_so$attributes2 = so.attributes) === null || _so$attributes2 === void 0 ? void 0 : _so$attributes2.title,
              error: new Error(so.error.message)
            });
          } else {
            taskIdsToDisable.push(so.id);
          }
        }
      } else {
        return {
          scheduled_report_ids: [...disabledScheduledReportIds],
          errors: bulkErrors,
          total: disabledScheduledReportIds.size + bulkErrors.length
        };
      }

      // it's possible that the scheduled_report saved object was disabled but
      // task disabling failed so add the list of already disabled IDs
      // task manager filters out disabled tasks so this will not cause extra load
      taskIdsToDisable = taskIdsToDisable.concat([...disabledScheduledReportIds]);
      const resultFromDisablingTasks = await this.taskManager.bulkDisable(taskIdsToDisable);
      for (const error of resultFromDisablingTasks.errors) {
        bulkErrors.push({
          message: `Scheduled report disabled but task disabling failed due to: ${error.error.message}`,
          status: error.error.statusCode,
          id: error.id
        });
      }
      for (const result of resultFromDisablingTasks.tasks) {
        disabledScheduledReportIds.add(result.id);
      }
      return {
        scheduled_report_ids: [...disabledScheduledReportIds],
        errors: bulkErrors,
        total: disabledScheduledReportIds.size + bulkErrors.length
      };
    } catch (error) {
      throw this.responseFactory.customError({
        statusCode: 500,
        body: `Error disabling scheduled reports: ${error.message}`
      });
    }
  }
  async bulkDelete({
    ids,
    user
  }) {
    try {
      const username = this.getUsername(user);
      const bulkGetResult = await this.savedObjectsClient.bulkGet(ids.map(id => ({
        id,
        type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE
      })));
      const [validSchedules, bulkGetErrors] = (0, _lodash.partition)(bulkGetResult.saved_objects, so => so.error === undefined);
      const [authorizedSchedules, unauthorizedSchedules] = (0, _lodash.partition)(validSchedules, so => so.attributes.createdBy === username || this.userCanManageReporting);
      const authErrors = this.formatAndAuditBulkDeleteAuthErrors({
        bulkGetErrors,
        unauthorizedSchedules,
        username
      });
      this.auditBulkGetAuthorized({
        action: _audit_events.ScheduledReportAuditAction.DELETE,
        authorizedSchedules
      });
      if (authorizedSchedules.length === 0) {
        return (0, _transforms.transformBulkDeleteResponse)({
          deletedSchedulesIds: [],
          errors: authErrors
        });
      }
      const bulkDeleteResult = await this.savedObjectsClient.bulkDelete(authorizedSchedules.map(so => ({
        id: so.id,
        type: so.type
      })));
      const [deletedSchedules, bulkDeleteErrors] = (0, _lodash.partition)(bulkDeleteResult.statuses, status => status.error === undefined);
      const executionErrors = this.formatAndAuditBulkDeleteSchedulesErrors({
        errorStatuses: bulkDeleteErrors
      });
      const removeTasksResult = await this.taskManager.bulkRemove(deletedSchedules.map(so => so.id));
      const [removedTasks, erroredTasks] = (0, _lodash.partition)(removeTasksResult.statuses, status => Boolean(status.success));
      const taskErrors = this.formatBulkDeleteTaskErrors({
        errorStatuses: erroredTasks
      });
      return (0, _transforms.transformBulkDeleteResponse)({
        deletedSchedulesIds: removedTasks.map(task => task.id),
        errors: [...authErrors, ...executionErrors, ...taskErrors]
      });
    } catch (error) {
      throw this.responseFactory.customError({
        statusCode: 500,
        body: `Error deleting scheduled reports: ${error.message}`
      });
    }
  }
  auditBulkGetAuthorized({
    action,
    authorizedSchedules
  }) {
    authorizedSchedules.forEach(so => {
      this.auditLog({
        action,
        id: so.id,
        name: so.attributes.title,
        outcome: 'unknown'
      });
    });
  }
  formatAndAuditBulkDeleteAuthErrors({
    bulkGetErrors,
    unauthorizedSchedules,
    username
  }) {
    const bulkErrors = [];
    bulkGetErrors.forEach(so => {
      if (!so.error) {
        return;
      }
      bulkErrors.push({
        message: so.error.message,
        status: so.error.statusCode,
        id: so.id
      });
    });
    unauthorizedSchedules.forEach(so => {
      var _so$attributes3;
      bulkErrors.push({
        message: `Not found.`,
        status: 404,
        id: so.id
      });
      this.logger.warn(`User "${username}" attempted to delete scheduled report "${so.id}" created by "${so.attributes.createdBy}" without sufficient privileges.`);
      this.auditLog({
        action: _audit_events.ScheduledReportAuditAction.DELETE,
        id: so.id,
        name: so === null || so === void 0 ? void 0 : (_so$attributes3 = so.attributes) === null || _so$attributes3 === void 0 ? void 0 : _so$attributes3.title,
        outcome: 'failure',
        error: new Error(`Not found.`)
      });
    });
    return bulkErrors;
  }
  formatAndAuditBulkDeleteSchedulesErrors({
    errorStatuses
  }) {
    const bulkErrors = [];
    errorStatuses.forEach(status => {
      if (!status.error) {
        return;
      }
      bulkErrors.push({
        message: status.error.message,
        status: status.error.statusCode,
        id: status.id
      });
      this.auditLog({
        action: _audit_events.ScheduledReportAuditAction.DELETE,
        id: status.id,
        error: new Error(status.error.message)
      });
    });
    return bulkErrors;
  }
  formatBulkDeleteTaskErrors({
    errorStatuses
  }) {
    const bulkErrors = [];
    errorStatuses.forEach(error => {
      if (error.error == null) {
        return;
      }
      bulkErrors.push({
        message: `Scheduled report deleted but task deleting failed due to: ${error.error.message}`,
        status: error.error.statusCode,
        id: error.id
      });
    });
    return bulkErrors;
  }
  getUsername(user) {
    return user ? user.username : false;
  }
  getEmptyListApiResponse(page, perPage) {
    return {
      page,
      per_page: perPage,
      total: 0,
      data: []
    };
  }
  auditLog({
    action,
    id,
    name,
    outcome,
    error
  }) {
    this.auditLogger.log((0, _audit_events.scheduledReportAuditEvent)({
      action,
      savedObject: {
        type: _saved_objects.SCHEDULED_REPORT_SAVED_OBJECT_TYPE,
        id,
        name
      },
      outcome,
      error
    }));
  }
}
exports.ScheduledReportsService = ScheduledReportsService;