"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ContentStream = void 0;
exports.finishedWithNoPendingCallbacks = finishedWithNoPendingCallbacks;
exports.getContentStream = getContentStream;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _stream = require("stream");
var _uuid = require("uuid");
var _promises = require("stream/promises");
var _promises2 = require("timers/promises");
var _reportingServer = require("@kbn/reporting-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ONE_MB = 1024 * 1024;
class ContentStream extends _stream.Duplex {
  constructor(client, logger, document, {
    encoding = 'base64'
  } = {}) {
    super();
    (0, _defineProperty2.default)(this, "buffers", []);
    (0, _defineProperty2.default)(this, "bytesBuffered", 0);
    (0, _defineProperty2.default)(this, "bytesRead", 0);
    (0, _defineProperty2.default)(this, "chunksRead", 0);
    (0, _defineProperty2.default)(this, "chunksWritten", 0);
    (0, _defineProperty2.default)(this, "jobSize", void 0);
    (0, _defineProperty2.default)(this, "parameters", void 0);
    (0, _defineProperty2.default)(this, "primaryTerm", void 0);
    (0, _defineProperty2.default)(this, "seqNo", void 0);
    (0, _defineProperty2.default)(this, "createdAt", void 0);
    /**
     * The number of bytes written so far.
     * Does not include data that is still queued for writing.
     */
    (0, _defineProperty2.default)(this, "bytesWritten", 0);
    /**
     * The chunking size of reporting files. Larger CSV files will be split into
     * multiple documents, where the stream is chunked into pieces of approximately
     * this size. The actual document size will be slightly larger due to Base64
     * encoding and JSON metadata.
     */
    (0, _defineProperty2.default)(this, "chunkSize", 4 * ONE_MB);
    this.client = client;
    this.logger = logger;
    this.document = document;
    this.parameters = {
      encoding
    };
    this.createdAt = document.created_at;
  }
  decode(content) {
    return Buffer.from(content, this.parameters.encoding === 'base64' ? 'base64' : undefined);
  }
  encode(buffer) {
    return buffer.toString(this.parameters.encoding === 'base64' ? 'base64' : undefined);
  }
  async readHead() {
    var _response$hits, _response$hits$hits, _hits$_source, _hits$_source$output, _hits$_source2, _hits$_source2$output;
    const {
      id,
      index
    } = this.document;
    const body = {
      _source: {
        includes: ['output.content', 'output.size', 'jobtype']
      },
      query: {
        constant_score: {
          filter: {
            bool: {
              must: [{
                term: {
                  _id: id
                }
              }]
            }
          }
        }
      },
      size: 1
    };
    this.logger.debug(`Reading report contents.`);
    const response = await this.client.search({
      ...body,
      index
    });
    const hits = response === null || response === void 0 ? void 0 : (_response$hits = response.hits) === null || _response$hits === void 0 ? void 0 : (_response$hits$hits = _response$hits.hits) === null || _response$hits$hits === void 0 ? void 0 : _response$hits$hits[0];
    this.jobSize = hits === null || hits === void 0 ? void 0 : (_hits$_source = hits._source) === null || _hits$_source === void 0 ? void 0 : (_hits$_source$output = _hits$_source.output) === null || _hits$_source$output === void 0 ? void 0 : _hits$_source$output.size;
    this.logger.debug(`Reading job of size ${this.jobSize}`);
    return hits === null || hits === void 0 ? void 0 : (_hits$_source2 = hits._source) === null || _hits$_source2 === void 0 ? void 0 : (_hits$_source2$output = _hits$_source2.output) === null || _hits$_source2$output === void 0 ? void 0 : _hits$_source2$output.content;
  }
  async readChunk() {
    var _response$hits2, _response$hits2$hits, _hits$_source3;
    const {
      id
    } = this.document;
    const body = {
      _source: {
        includes: ['output.content']
      },
      query: {
        constant_score: {
          filter: {
            bool: {
              must: [{
                term: {
                  parent_id: id
                }
              }, {
                term: {
                  'output.chunk': this.chunksRead
                }
              }]
            }
          }
        }
      },
      size: 1
    };
    this.logger.debug(`Reading chunk #${this.chunksRead}.`);
    const response = await this.client.search({
      ...body,
      index: _reportingServer.REPORTING_DATA_STREAM_WILDCARD_WITH_LEGACY
    });
    const hits = response === null || response === void 0 ? void 0 : (_response$hits2 = response.hits) === null || _response$hits2 === void 0 ? void 0 : (_response$hits2$hits = _response$hits2.hits) === null || _response$hits2$hits === void 0 ? void 0 : _response$hits2$hits[0];
    return hits === null || hits === void 0 ? void 0 : (_hits$_source3 = hits._source) === null || _hits$_source3 === void 0 ? void 0 : _hits$_source3.output.content;
  }
  isRead() {
    return this.jobSize != null && this.bytesRead >= this.jobSize;
  }
  _read() {
    (this.chunksRead ? this.readChunk() : this.readHead()).then(content => {
      if (!content) {
        this.logger.debug(`Chunk is empty.`);
        this.push(null);
        return;
      }
      const buffer = this.decode(content);
      this.push(buffer);
      this.chunksRead++;
      this.bytesRead += buffer.byteLength;
      if (this.isRead()) {
        this.logger.debug(`Read ${this.bytesRead} of ${this.jobSize} bytes.`);
        this.push(null);
      }
    }).catch(err => this.destroy(err));
  }
  async removeChunks() {
    const {
      id,
      index
    } = this.document;
    await this.client.deleteByQuery({
      index,
      query: {
        match: {
          parent_id: id
        }
      }
    });
  }
  async writeHead(content) {
    this.logger.debug(`Updating chunk #0 (${this.document.id}).`);
    const body = await this.client.update({
      ...this.document,
      doc: {
        output: {
          content
        }
      }
    });
    ({
      _primary_term: this.primaryTerm,
      _seq_no: this.seqNo
    } = body);
  }
  async writeChunk(content) {
    const {
      id: parentId
    } = this.document;
    const id = (0, _uuid.v4)();
    this.logger.debug(`Writing chunk #${this.chunksWritten} (${id}).`);
    await this.client.index({
      id,
      index: _reportingServer.REPORTING_DATA_STREAM_ALIAS,
      op_type: 'create',
      document: {
        parent_id: parentId,
        '@timestamp': this.createdAt || new Date().toISOString(),
        // use report creation time for better diagnosis
        output: {
          content,
          chunk: this.chunksWritten
        }
      }
    });
  }
  async flush(size = this.bytesBuffered) {
    const buffersToFlush = [];
    let bytesToFlush = 0;

    /*
     Loop over each buffer, keeping track of how many bytes we have added
     to the array of buffers to be flushed. The array of buffers to be flushed
     contains buffers by reference, not copies. This avoids putting pressure on
     the CPU for copying buffers or for gc activity. Please profile performance
     with a large byte configuration and a large number of records (900k+)
     before changing this code. Config used at time of writing:
       xpack.reporting:
        csv.maxSizeBytes: 500000000
        csv.scroll.size: 1000
      At the moment this can put memory pressure on Kibana. Up to 1,1 GB in a dev
     build. It is not recommended to have overly large max size bytes but we
     need this code to be as performant as possible.
    */
    while (this.buffers.length) {
      const remainder = size - bytesToFlush;
      if (remainder <= 0) {
        break;
      }
      const buffer = this.buffers.shift();
      const chunkedBuffer = buffer.slice(0, remainder);
      buffersToFlush.push(chunkedBuffer);
      bytesToFlush += chunkedBuffer.byteLength;
      if (buffer.byteLength > remainder) {
        this.buffers.unshift(buffer.slice(remainder));
      }
    }

    // We call Buffer.concat with the fewest number of buffers possible
    const chunk = Buffer.concat(buffersToFlush);
    const content = this.encode(chunk);
    if (!this.chunksWritten) {
      await this.removeChunks();
      await this.writeHead(content);
    } else if (chunk.byteLength) {
      await this.writeChunk(content);
    }
    if (chunk.byteLength) {
      this.chunksWritten++;
    }
    this.bytesWritten += chunk.byteLength;
    this.bytesBuffered -= bytesToFlush;
  }
  async flushAllFullChunks() {
    while (this.bytesBuffered >= this.chunkSize && this.buffers.length) {
      await this.flush(this.chunkSize);
    }
  }
  _write(chunk, encoding, callback) {
    const buffer = Buffer.isBuffer(chunk) ? chunk : Buffer.from(chunk, encoding);
    this.bytesBuffered += buffer.byteLength;
    this.buffers.push(buffer);
    this.flushAllFullChunks().then(() => callback()).catch(callback);
  }
  _final(callback) {
    this.flush().then(() => callback()).catch(callback);
  }
  getSeqNo() {
    return this.seqNo;
  }
  getPrimaryTerm() {
    return this.primaryTerm;
  }
}
exports.ContentStream = ContentStream;
async function getContentStream(reporting, document, parameters) {
  const {
    asInternalUser: client
  } = await reporting.getEsClient();
  const {
    logger
  } = reporting.getPluginSetupDeps();
  return new ContentStream(client, logger.get('content_stream').get(document.id), document, parameters);
}
const MAX_RECURSION = 100;
async function finishedWithNoPendingCallbacks(stream) {
  await (0, _promises.finished)(stream, {
    readable: false
  });

  // Race condition workaround:
  // `finished(...)` will resolve while there's still pending callbacks in the writable part of the `stream`.
  // This introduces a race condition where the code continues before the writable part has completely finished.
  // The `pendingCallbacks` function is a hack to ensure that all pending callbacks have been called before continuing.
  // For more information, see: https://github.com/nodejs/node/issues/46170
  await async function pendingCallbacks(delay = 1, attempts = 1) {
    if (stream._writableState.pendingcb > 0) {
      await (0, _promises2.setTimeout)(delay);
      if (attempts < MAX_RECURSION) {
        await pendingCallbacks(delay < 32 ? delay * 2 : delay, attempts + 1);
      }
    }
  }();
}