"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAllStats = getAllStats;
exports.getStackStats = getStackStats;
exports.handleAllStats = handleAllStats;
exports.mergeXPackStats = mergeXPackStats;
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _lodash = require("lodash");
var _moment = _interopRequireDefault(require("moment"));
var _constants = require("../../common/constants");
var _get_es_stats = require("./get_es_stats");
var _get_kibana_stats = require("./get_kibana_stats");
var _get_beats_stats = require("./get_beats_stats");
var _get_logstash_stats = require("./get_logstash_stats");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get statistics for all products joined by Elasticsearch cluster.
 * Returns the array of clusters joined with the Kibana and Logstash instances.
 *
 */
async function getAllStats(clusterUuids, callCluster, timestamp, maxBucketSize) {
  const start = (0, _moment.default)(timestamp).subtract(_constants.USAGE_FETCH_INTERVAL, 'ms').toISOString();
  const end = (0, _moment.default)(timestamp).toISOString();
  const [esClusters, kibana, logstash, beats] = await Promise.all([(0, _get_es_stats.getElasticsearchStats)(callCluster, clusterUuids, start, end, maxBucketSize),
  // cluster_stats, stack_stats.xpack, cluster_name/uuid, license, version
  (0, _get_kibana_stats.getKibanaStats)(callCluster, clusterUuids, start, end, maxBucketSize),
  // stack_stats.kibana
  (0, _get_logstash_stats.getLogstashStats)(callCluster, clusterUuids, start, end),
  // stack_stats.logstash
  (0, _get_beats_stats.getBeatsStats)(callCluster, clusterUuids, start, end) // stack_stats.beats
  ]);
  return handleAllStats(esClusters, {
    kibana,
    logstash,
    beats
  });
}

/**
 * Combine the statistics from the stack to create "cluster" stats that associate all products together based on the cluster
 * that is attached.
 *
 * @param {Array} clusters The Elasticsearch clusters
 * @param {Object} kibana The Kibana instances keyed by Cluster UUID
 * @param {Object} logstash The Logstash nodes keyed by Cluster UUID
 * @param {Object} beats The Beats nodes keyed by Cluster UUID
 *
 * Returns the clusters joined with the Kibana and Logstash instances under each cluster's {@code stack_stats}.
 */
function handleAllStats(clusters, {
  kibana,
  logstash,
  beats
}) {
  const mappedClusters = clusters.map(cluster => {
    const stats = {
      ...cluster,
      stack_stats: {
        ...cluster.stack_stats,
        // if they are using Kibana or Logstash, then add it to the cluster details under cluster.stack_stats
        ...getStackStats(cluster.cluster_uuid, kibana, _constants.KIBANA_SYSTEM_ID),
        ...getStackStats(cluster.cluster_uuid, logstash, _constants.LOGSTASH_SYSTEM_ID),
        ...getStackStats(cluster.cluster_uuid, beats, _constants.BEATS_SYSTEM_ID)
      }
    };
    mergeXPackStats(stats, kibana, 'graph_workspace', 'graph'); // copy graph_workspace info out of kibana, merge it into stack_stats.xpack.graph

    return stats;
  });
  const mappedClusterUuids = mappedClusters.map(cluster => {
    var _cluster$elasticsearc, _cluster$elasticsearc2;
    return cluster.cluster_uuid || (cluster === null || cluster === void 0 ? void 0 : (_cluster$elasticsearc = cluster.elasticsearch) === null || _cluster$elasticsearc === void 0 ? void 0 : (_cluster$elasticsearc2 = _cluster$elasticsearc.cluster) === null || _cluster$elasticsearc2 === void 0 ? void 0 : _cluster$elasticsearc2.id);
  });

  // Logstash agent driven monitoring isn't based on cluster UUID
  // or standalone LS clusters will be reported with monitoring cluster UUIDs
  const logstashOrphanClusterStats = logstash ? Object.entries(logstash).filter(([clusterUuid]) => !mappedClusterUuids.includes(clusterUuid)).map(([clusterUuid, logstashBaseStats]) => ({
    cluster_name: _constants.LOGSTASH_SYSTEM_ID,
    timestamp: `${_moment.default.utc().format()}`,
    version: logstashBaseStats.versions.length > 0 ? logstashBaseStats.versions[0].version : '',
    cluster_uuid: clusterUuid,
    stack_stats: {
      [_constants.LOGSTASH_SYSTEM_ID]: logstashBaseStats
    },
    cluster_stats: {}
  })) : [];
  return mappedClusters.concat(logstashOrphanClusterStats);
}
function getStackStats(clusterUuid, allProductStats, product) {
  const productStats = allProductStats[clusterUuid];
  // Don't add it if they're not using (or configured to report stats) this product for this cluster
  return productStats ? {
    [product]: productStats
  } : {};
}
function mergeXPackStats(cluster, allProductStats, path, product) {
  const productStats = (0, _lodash.get)(allProductStats, cluster.cluster_uuid + '.' + path);
  if (productStats || productStats === 0) {
    if (!cluster.stack_stats) {
      cluster.stack_stats = {};
    }
    if (!cluster.stack_stats.xpack) {
      cluster.stack_stats.xpack = {};
    }
    const mergeStats = {};
    (0, _saferLodashSet.set)(mergeStats, path, productStats);

    // merge existing data with new stats
    cluster.stack_stats.xpack[product] = cluster.stack_stats.xpack[product] || {};
    (0, _lodash.merge)(cluster.stack_stats.xpack[product], mergeStats);
  }
}