"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.retrieveDocumentation = void 0;
var _tokens = require("../../utils/tokens");
var _summarize_document = require("./summarize_document");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_DOCUMENTS_DEFAULT = 3;
const MAX_TOKENS_DEFAULT = 1000;
const retrieveDocumentation = ({
  outputAPI,
  searchDocAPI,
  logger: log
}) => async ({
  searchTerm,
  connectorId,
  products,
  functionCalling,
  inferenceId,
  max = MAX_DOCUMENTS_DEFAULT,
  maxDocumentTokens = MAX_TOKENS_DEFAULT,
  tokenReductionStrategy = 'highlight'
}) => {
  const applyTokenReductionStrategy = async doc => {
    let content;
    switch (tokenReductionStrategy) {
      case 'highlight':
        content = doc.highlights.join('\n\n');
        break;
      case 'summarize':
        const extractResponse = await (0, _summarize_document.summarizeDocument)({
          searchTerm,
          documentContent: doc.content,
          outputAPI,
          connectorId,
          functionCalling
        });
        content = extractResponse.summary;
        break;
      case 'truncate':
        content = doc.content;
        break;
    }
    return (0, _tokens.truncate)(content, maxDocumentTokens);
  };
  try {
    const highlights = tokenReductionStrategy === 'highlight' ? calculateHighlightCount(maxDocumentTokens) : 0;
    const {
      results
    } = await searchDocAPI({
      query: searchTerm,
      products,
      max,
      highlights,
      inferenceId
    });
    log.debug(`searching with term=[${searchTerm}] returned ${results.length} documents`);
    const processedDocuments = await Promise.all(results.map(async document => {
      const tokenCount = (0, _tokens.count)(document.content);
      const docHasTooManyTokens = tokenCount >= maxDocumentTokens;
      log.debug(`processing doc [${document.url}] - tokens : [${tokenCount}] - tooManyTokens: [${docHasTooManyTokens}]`);
      let content = document.content;
      if (docHasTooManyTokens) {
        content = await applyTokenReductionStrategy(document);
      }
      log.debug(`done processing document [${document.url}]`);
      return {
        title: document.title,
        url: document.url,
        content,
        summarized: docHasTooManyTokens
      };
    }));
    log.debug(() => {
      const docsAsJson = JSON.stringify(processedDocuments);
      return `searching with term=[${searchTerm}] - results: ${docsAsJson}`;
    });
    return {
      success: true,
      documents: processedDocuments.filter(doc => doc.content.length > 0)
    };
  } catch (e) {
    log.error(`Error retrieving documentation: ${e.message}. Returning empty results.`);
    return {
      success: false,
      documents: []
    };
  }
};
exports.retrieveDocumentation = retrieveDocumentation;
const AVG_TOKENS_PER_HIGHLIGHT = 250;
const calculateHighlightCount = maxTokensPerDoc => {
  return Math.ceil(maxTokensPerDoc / AVG_TOKENS_PER_HIGHLIGHT);
};