"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NO_EXPIRATION = void 0;
exports.bulkCreateAgentActionResults = bulkCreateAgentActionResults;
exports.bulkCreateAgentActions = bulkCreateAgentActions;
exports.cancelAgentAction = cancelAgentAction;
exports.createAgentAction = createAgentAction;
exports.createErrorActionResults = createErrorActionResults;
exports.getAgentActions = getAgentActions;
exports.getAgentsByActionsIds = void 0;
exports.getUnenrollAgentActions = getUnenrollAgentActions;
exports.transformDataSecrets = transformDataSecrets;
var _uuid = require("uuid");
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _pMap = _interopRequireDefault(require("p-map"));
var _lodash = require("lodash");
var _app_context = require("../app_context");
var _constants = require("../../../common/constants");
var _errors = require("../../errors");
var _audit_logging = require("../audit_logging");
var _agent_policies_to_agent_ids = require("../agent_policies/agent_policies_to_agent_ids");
var _get_current_namespace = require("../spaces/get_current_namespace");
var _query_namespaces_filtering = require("../spaces/query_namespaces_filtering");
var _constants2 = require("../../constants");
var _secrets = require("../secrets");
var _crud = require("./crud");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ONE_MONTH_IN_MS = 2592000000;
const NO_EXPIRATION = exports.NO_EXPIRATION = 'NONE';
const SIGNED_ACTIONS = new Set(['UNENROLL', 'UPGRADE', 'MIGRATE']);

/**
 * Indexes a new action to the .fleet-actions index.
 * Takes any secret data stored within the secrets field, stores it in saved objects,
 * and replaces them in the action data with a reference to the saved objects.
 */
async function createAgentAction(esClient, soClient, newAgentAction) {
  var _newAgentAction$id, _newAgentAction$expir, _secretReferences;
  const actionId = (_newAgentAction$id = newAgentAction.id) !== null && _newAgentAction$id !== void 0 ? _newAgentAction$id : (0, _uuid.v4)();
  const now = Date.now();
  const timestamp = new Date(now).toISOString();
  let data;
  let secretReferences;
  // Store secret values if enabled, otherwise store them as plain text.
  if (await (0, _secrets.isActionSecretStorageEnabled)(esClient, soClient)) {
    const secretsRes = await (0, _secrets.extractAndWriteActionSecrets)({
      action: newAgentAction,
      esClient
    });
    const mergedData = (0, _lodash.merge)(secretsRes.actionWithSecrets.data, secretsRes.actionWithSecrets.secrets);
    data = transformDataSecrets(mergedData);
    secretReferences = secretsRes.secretReferences;
  } else {
    data = (0, _lodash.merge)(newAgentAction.data, newAgentAction.secrets);
  }
  const body = {
    '@timestamp': timestamp,
    expiration: newAgentAction.expiration === NO_EXPIRATION ? undefined : (_newAgentAction$expir = newAgentAction.expiration) !== null && _newAgentAction$expir !== void 0 ? _newAgentAction$expir : new Date(now + ONE_MONTH_IN_MS).toISOString(),
    agents: newAgentAction.agents,
    namespaces: newAgentAction.namespaces,
    action_id: actionId,
    data,
    type: newAgentAction.type,
    start_time: newAgentAction.start_time,
    minimum_execution_duration: newAgentAction.minimum_execution_duration,
    rollout_duration_seconds: newAgentAction.rollout_duration_seconds,
    total: newAgentAction.total,
    traceparent: _elasticApmNode.default.currentTraceparent,
    is_automatic: newAgentAction.is_automatic,
    policyId: newAgentAction.policyId,
    ...(((_secretReferences = secretReferences) === null || _secretReferences === void 0 ? void 0 : _secretReferences.length) && {
      secret_references: secretReferences
    })
  };
  const messageSigningService = _app_context.appContextService.getMessageSigningService();
  if (SIGNED_ACTIONS.has(newAgentAction.type) && messageSigningService) {
    const signedBody = await messageSigningService.sign(body);
    body.signed = {
      data: signedBody.data.toString('base64'),
      signature: signedBody.signature
    };
  }
  await esClient.create({
    index: _constants.AGENT_ACTIONS_INDEX,
    id: (0, _uuid.v4)(),
    document: body,
    refresh: 'wait_for'
  });
  _audit_logging.auditLoggingService.writeCustomAuditLog({
    message: `${newAgentAction.is_automatic ? 'Automatic Upgrade' : 'User'} created Fleet action [id=${actionId}]`
  });
  return {
    id: actionId,
    ...newAgentAction,
    created_at: timestamp
  };
}

/**
 * Recursively transforms all occurrences of { id: string } objects
 * into `$co.elastic.secret{${id}}`, for any level of nesting.
 */
function transformDataSecrets(mergedData) {
  if (Array.isArray(mergedData)) {
    return mergedData.map(transformDataSecrets);
  } else if (mergedData && typeof mergedData === 'object') {
    const newMergedData = {}; // NewAgentAction.data has any type
    for (const [key, value] of Object.entries(mergedData)) {
      if (value && typeof value === 'object' && !Array.isArray(value) && Object.keys(value).length === 1 && Object.prototype.hasOwnProperty.call(value, 'id') && typeof value.id === 'string') {
        newMergedData[key] = (0, _secrets.toCompiledSecretRef)(value.id);
      } else {
        newMergedData[key] = transformDataSecrets(value);
      }
    }
    return newMergedData;
  }
  return mergedData;
}
async function bulkCreateAgentActions(esClient, newAgentActions) {
  const actions = newAgentActions.map(newAgentAction => {
    var _newAgentAction$id2;
    const id = (_newAgentAction$id2 = newAgentAction.id) !== null && _newAgentAction$id2 !== void 0 ? _newAgentAction$id2 : (0, _uuid.v4)();
    return {
      id,
      ...newAgentAction
    };
  });
  if (actions.length === 0) {
    return [];
  }
  const messageSigningService = _app_context.appContextService.getMessageSigningService();
  const fleetServerAgentActions = await (0, _pMap.default)(actions, async action => {
    var _action$expiration;
    const body = {
      '@timestamp': new Date().toISOString(),
      expiration: (_action$expiration = action.expiration) !== null && _action$expiration !== void 0 ? _action$expiration : new Date(Date.now() + ONE_MONTH_IN_MS).toISOString(),
      start_time: action.start_time,
      rollout_duration_seconds: action.rollout_duration_seconds,
      agents: action.agents,
      action_id: action.id,
      data: action.data,
      type: action.type,
      traceparent: _elasticApmNode.default.currentTraceparent
    };
    if (SIGNED_ACTIONS.has(action.type) && messageSigningService) {
      const signedBody = await messageSigningService.sign(body);
      body.signed = {
        data: signedBody.data.toString('base64'),
        signature: signedBody.signature
      };
    }
    return [{
      create: {
        _id: action.id
      }
    }, body].flat();
  }, {
    concurrency: _constants2.MAX_CONCURRENT_CREATE_ACTIONS
  });
  await esClient.bulk({
    index: _constants.AGENT_ACTIONS_INDEX,
    operations: fleetServerAgentActions
  });
  for (const action of actions) {
    _audit_logging.auditLoggingService.writeCustomAuditLog({
      message: `User created Fleet action [id=${action.id}]`
    });
  }
  return actions;
}
async function createErrorActionResults(esClient, actionId, errors, errorMessage) {
  const errorCount = Object.keys(errors).length;
  if (errorCount > 0) {
    _app_context.appContextService.getLogger().info(`Writing error action results of ${errorCount} agents. Possibly failed validation: ${errorMessage}.`);

    // writing out error result for those agents that have errors, so the action is not going to stay in progress forever
    await bulkCreateAgentActionResults(esClient, Object.keys(errors).map(agentId => ({
      agentId,
      actionId,
      error: errors[agentId].message
    })));
  }
}
async function bulkCreateAgentActionResults(esClient, results) {
  if (results.length === 0) {
    return;
  }
  const bulkBody = results.flatMap(result => {
    const body = {
      '@timestamp': new Date().toISOString(),
      action_id: result.actionId,
      agent_id: result.agentId,
      namespaces: result.namespaces,
      error: result.error
    };
    return [{
      create: {
        _id: (0, _uuid.v4)()
      }
    }, body];
  });
  for (const result of results) {
    _audit_logging.auditLoggingService.writeCustomAuditLog({
      message: `User created Fleet action result [id=${result.actionId}]`
    });
  }
  await esClient.bulk({
    index: _constants.AGENT_ACTIONS_RESULTS_INDEX,
    operations: bulkBody,
    refresh: 'wait_for'
  });
}
async function getAgentActions(esClient, actionId) {
  const res = await esClient.search({
    index: _constants.AGENT_ACTIONS_INDEX,
    query: {
      bool: {
        must: [{
          term: {
            action_id: actionId
          }
        }]
      }
    },
    size: _constants.SO_SEARCH_LIMIT
  });
  if (res.hits.hits.length === 0) {
    throw new _errors.AgentActionNotFoundError('Action not found');
  }
  const result = [];
  for (const hit of res.hits.hits) {
    var _hit$_source;
    _audit_logging.auditLoggingService.writeCustomAuditLog({
      message: `User retrieved Fleet action [id=${(_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : _hit$_source.action_id}]`
    });
    result.push({
      ...hit._source,
      id: hit._id
    });
  }
  return result;
}
async function getUnenrollAgentActions(esClient) {
  const res = await esClient.search({
    index: _constants.AGENT_ACTIONS_INDEX,
    query: {
      bool: {
        must: [{
          term: {
            type: 'UNENROLL'
          }
        }, {
          exists: {
            field: 'agents'
          }
        }, {
          range: {
            expiration: {
              gte: new Date().toISOString()
            }
          }
        }]
      }
    },
    size: _constants.SO_SEARCH_LIMIT
  });
  const result = [];
  for (const hit of res.hits.hits) {
    var _hit$_source2;
    _audit_logging.auditLoggingService.writeCustomAuditLog({
      message: `User retrieved Fleet action [id=${(_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : _hit$_source2.action_id}]`
    });
    result.push({
      ...hit._source,
      id: hit._id
    });
  }
  return result;
}
async function cancelAgentAction(esClient, soClient, actionId) {
  const currentSpaceId = (0, _get_current_namespace.getCurrentNamespace)(soClient);
  const getUpgradeActions = async () => {
    const query = {
      bool: {
        filter: [{
          term: {
            action_id: actionId
          }
        }]
      }
    };
    const res = await esClient.search({
      index: _constants.AGENT_ACTIONS_INDEX,
      query: await (0, _query_namespaces_filtering.addNamespaceFilteringToQuery)(query, currentSpaceId),
      size: _constants.SO_SEARCH_LIMIT
    });
    if (res.hits.hits.length === 0) {
      throw new _errors.AgentActionNotFoundError('Action not found');
    }
    for (const hit of res.hits.hits) {
      var _hit$_source3;
      _audit_logging.auditLoggingService.writeCustomAuditLog({
        message: `User retrieved Fleet action [id=${(_hit$_source3 = hit._source) === null || _hit$_source3 === void 0 ? void 0 : _hit$_source3.action_id}]}]`
      });
    }
    const upgradeActions = res.hits.hits.map(hit => hit._source).filter(action => !!action && !!action.agents && !!action.action_id && action.type === 'UPGRADE');
    return upgradeActions;
  };
  const cancelActionId = (0, _uuid.v4)();
  const now = new Date().toISOString();
  const cancelledActions = [];
  const createAction = async action => {
    await createAgentAction(esClient, soClient, {
      id: cancelActionId,
      type: 'CANCEL',
      namespaces: [currentSpaceId],
      agents: action.agents,
      data: {
        target_id: action.action_id
      },
      created_at: now,
      expiration: action.expiration
    });
    cancelledActions.push({
      agents: action.agents
    });
  };
  let upgradeActions = await getUpgradeActions();
  for (const action of upgradeActions) {
    await createAction(action);
  }
  const updateAgentsToHealthy = async action => {
    _app_context.appContextService.getLogger().info(`Moving back ${action.agents.length} agents from updating to healthy state after cancel upgrade`);
    const errors = {};
    await (0, _crud.bulkUpdateAgents)(esClient, action.agents.map(agentId => ({
      agentId,
      data: {
        upgraded_at: null,
        upgrade_started_at: null
      }
    })), errors);
    if (Object.keys(errors).length > 0) {
      _app_context.appContextService.getLogger().info(`Errors while bulk updating agents for cancel action: ${JSON.stringify(errors)}`);
    }
  };
  for (const action of upgradeActions) {
    await updateAgentsToHealthy(action);
  }

  // At the end of cancel, doing one more query on upgrade action to find those docs that were possibly created by a concurrent upgrade action.
  // This is to make sure we cancel all upgrade batches.
  upgradeActions = await getUpgradeActions();
  if (cancelledActions.length < upgradeActions.length) {
    const missingBatches = upgradeActions.filter(upgradeAction => !cancelledActions.some(cancelled => upgradeAction.agents && cancelled.agents[0] === upgradeAction.agents[0]));
    _app_context.appContextService.getLogger().debug(`missing batches to cancel: ${missingBatches.length}`);
    if (missingBatches.length > 0) {
      for (const missingBatch of missingBatches) {
        await createAction(missingBatch);
        await updateAgentsToHealthy(missingBatch);
      }
    }
  }
  return {
    created_at: now,
    id: cancelActionId,
    type: 'CANCEL'
  };
}
async function getAgentActionsByIds(esClient, actionIds) {
  if (actionIds.length === 0) {
    return [];
  }
  const res = await esClient.search({
    index: _constants.AGENT_ACTIONS_INDEX,
    query: {
      bool: {
        filter: [{
          terms: {
            action_id: actionIds
          }
        }]
      }
    },
    _source: ['agents', 'total'],
    size: _constants.SO_SEARCH_LIMIT
  });
  if (res.hits.hits.length === 0) {
    _app_context.appContextService.getLogger().debug(`No agent action found for ids ${actionIds}`);
    return [];
  }
  const result = [];
  let total = 0;
  for (const hit of res.hits.hits) {
    var _hit$_source4, _hit$_source$total, _hit$_source5;
    _audit_logging.auditLoggingService.writeCustomAuditLog({
      message: `User retrieved Fleet action [id=${(_hit$_source4 = hit._source) === null || _hit$_source4 === void 0 ? void 0 : _hit$_source4.action_id}]`
    });
    result.push({
      ...hit._source,
      id: hit._id
    });
    total = (_hit$_source$total = (_hit$_source5 = hit._source) === null || _hit$_source5 === void 0 ? void 0 : _hit$_source5.total) !== null && _hit$_source$total !== void 0 ? _hit$_source$total : 0;
  }
  const agentIds = [];
  if (result.length > 0) {
    agentIds.push(...result.flatMap(a => a === null || a === void 0 ? void 0 : a.agents).filter(agent => !!agent));
  }
  if (agentIds.length < total) {
    const agentIdsFromResults = await getAgentIdsFromResults(esClient, actionIds);
    return (0, _lodash.uniq)([...agentIds, ...agentIdsFromResults]);
  }
  return agentIds;
}
async function getAgentIdsFromResults(esClient, actionIds) {
  try {
    const results = await esClient.search({
      index: _constants.AGENT_ACTIONS_RESULTS_INDEX,
      ignore_unavailable: true,
      query: {
        bool: {
          filter: [{
            terms: {
              action_id: actionIds
            }
          }, {
            exists: {
              field: 'error'
            }
          }]
        }
      },
      _source: ['agent_id'],
      size: _constants.SO_SEARCH_LIMIT
    });
    const resultAgentIds = new Set();
    for (const hit of results.hits.hits) {
      var _hit$_source6;
      resultAgentIds.add((_hit$_source6 = hit._source) === null || _hit$_source6 === void 0 ? void 0 : _hit$_source6.agent_id);
    }
    return Array.from(resultAgentIds);
  } catch (err) {
    if (err.statusCode === 404) {
      // .fleet-actions-results does not yet exist
      _app_context.appContextService.getLogger().debug(err);
    } else {
      throw err;
    }
  }
  return [];
}
const getAgentsByActionsIds = async (esClient, actionsIds) => {
  // There are two types of actions:
  // 1. Agent actions stored in .fleet-actions, with type AgentActionType except 'POLICY_CHANGE'
  // 2. Agent policy actions, generated from .fleet-policies, with actionId `${hit.policy_id}:${hit.revision_idx}`

  const [agentPolicyActionIds, agentActionIds] = (0, _lodash.partition)(actionsIds, actionsId => actionsId.split(':').length > 1);
  const agentIds = await getAgentActionsByIds(esClient, agentActionIds);
  const policyIds = agentPolicyActionIds.map(actionId => actionId.split(':')[0]);
  const assignedAgentIds = await (0, _agent_policies_to_agent_ids.getAgentIdsForAgentPolicies)(esClient, policyIds);
  if (assignedAgentIds.length > 0) {
    agentIds.push(...assignedAgentIds);
  }
  return agentIds;
};
exports.getAgentsByActionsIds = getAgentsByActionsIds;