"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addPackageToAgentPolicy = addPackageToAgentPolicy;
exports.agentPolicyService = void 0;
exports.getAgentPolicySavedObjectType = getAgentPolicySavedObjectType;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _uuid = require("uuid");
var _jsYaml = require("js-yaml");
var _pMap = _interopRequireDefault(require("p-map"));
var _semver = require("semver");
var _server = require("@kbn/core/server");
var _constants = require("@kbn/spaces-plugin/common/constants");
var _apmUtils = require("@kbn/apm-utils");
var _utils = require("../errors/utils");
var _services = require("../../common/services");
var _constants2 = require("../constants");
var _constants3 = require("../../common/constants");
var _errors = require("../errors");
var _agent_cm_to_yaml = require("../../common/services/agent_cm_to_yaml");
var _ = require(".");
var _utils2 = require("./agent_policies/utils");
var _elastic_agent_manifest = require("./elastic_agent_manifest");
var _packages = require("./epm/packages");
var _agents = require("./agents");
var _package_policy = require("./package_policy");
var _package_policies = require("./package_policies");
var _output = require("./output");
var _agent_policy_update = require("./agent_policy_update");
var _agent_policy_create = require("./agent_policy_create");
var _saved_object = require("./saved_object");
var _agent_policies = require("./agent_policies");
var _audit_logging = require("./audit_logging");
var _license = require("./license");
var _create_so_find_iterable = require("./utils/create_so_find_iterable");
var _agentless = require("./utils/agentless");
var _policy_namespaces = require("./spaces/policy_namespaces");
var _helpers = require("./spaces/helpers");
var _agentless_agent = require("./agents/agentless_agent");
var _deploy_agent_policies_task = require("./agent_policies/deploy_agent_policies_task");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const KEY_EDITABLE_FOR_MANAGED_POLICIES = ['namespace'];
function normalizeKuery(savedObjectType, kuery) {
  if (savedObjectType === _constants2.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE) {
    return (0, _saved_object.normalizeKuery)(savedObjectType, kuery.replace(new RegExp(`${_constants3.AGENT_POLICY_SAVED_OBJECT_TYPE}\\.`, 'g'), `${savedObjectType}.attributes.`));
  } else {
    return (0, _saved_object.normalizeKuery)(savedObjectType, kuery.replace(new RegExp(`${_constants2.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE}\\.`, 'g'), `${savedObjectType}.attributes.`));
  }
}
async function getAgentPolicySavedObjectType() {
  return (await (0, _helpers.isSpaceAwarenessEnabled)()) ? _constants3.AGENT_POLICY_SAVED_OBJECT_TYPE : _constants2.LEGACY_AGENT_POLICY_SAVED_OBJECT_TYPE;
}
class AgentPolicyService {
  constructor() {
    (0, _defineProperty2.default)(this, "triggerAgentPolicyUpdatedEvent", async (esClient, action, agentPolicyId, options) => {
      return (0, _agent_policy_update.agentPolicyUpdateEventHandler)(esClient, action, agentPolicyId, options);
    });
  }
  getLogger(...childContextPaths) {
    return _.appContextService.getLogger().get('AgentPolicyService', ...childContextPaths);
  }
  async _update(soClient, esClient, id, agentPolicy, user, options = {
    bumpRevision: true,
    removeProtection: false,
    skipValidation: false,
    returnUpdatedPolicy: true,
    asyncDeploy: false
  }) {
    const logger = this.getLogger('_update');
    logger.debug(`Starting update of agent policy [${id}] with soClient scoped to [${soClient.getCurrentNamespace()}]`);
    const savedObjectType = await getAgentPolicySavedObjectType();
    const existingAgentPolicy = await this.get(soClient, id, true);
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'update',
      id,
      name: existingAgentPolicy === null || existingAgentPolicy === void 0 ? void 0 : existingAgentPolicy.name,
      savedObjectType
    });
    if (!existingAgentPolicy) {
      throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
    }
    if (existingAgentPolicy.status === _constants3.agentPolicyStatuses.Inactive && agentPolicy.status !== _constants3.agentPolicyStatuses.Active) {
      throw new _errors.FleetError(`Agent policy ${id} cannot be updated because it is ${existingAgentPolicy.status}`);
    }
    if (options.removeProtection) {
      logger.warn(`Setting tamper protection for Agent Policy ${id} to false`);
    }
    if (!options.skipValidation) {
      logger.debug(`Validating agent policy [${id}] before update`);
      await (0, _agent_policies.validateOutputForPolicy)(soClient, agentPolicy, existingAgentPolicy, (0, _services.getAllowedOutputTypesForAgentPolicy)({
        ...existingAgentPolicy,
        ...agentPolicy
      }));
    }
    await soClient.update(savedObjectType, id, {
      ...agentPolicy,
      ...(options.bumpRevision ? {
        revision: existingAgentPolicy.revision + 1
      } : {}),
      ...(options.removeProtection ? {
        is_protected: false
      } : {
        is_protected: agentPolicy.is_protected
      }),
      updated_at: new Date().toISOString(),
      updated_by: user ? user.username : 'system'
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`SO update to agent policy [${id}] failed`));
    const newAgentPolicy = await this.get(soClient, id, false);
    logger.debug(`Agent policy [${id}] Saved Object was updated successfully`);
    newAgentPolicy.package_policies = existingAgentPolicy.package_policies;
    if (options.bumpRevision || options.removeProtection) {
      if (!options.asyncDeploy) {
        logger.debug(`Triggering agent policy [${id}] updated event`);
        await this.triggerAgentPolicyUpdatedEvent(esClient, 'updated', id, {
          spaceId: soClient.getCurrentNamespace(),
          agentPolicy: newAgentPolicy
        });
      } else {
        logger.debug(`Scheduling task to deploy agent policy [${id}]`);
        await (0, _deploy_agent_policies_task.scheduleDeployAgentPoliciesTask)(_.appContextService.getTaskManagerStart(), [{
          id,
          spaceId: soClient.getCurrentNamespace()
        }]);
      }
    }
    logger.debug(`Agent policy ${id} update completed, revision: ${options.bumpRevision ? existingAgentPolicy.revision + 1 : existingAgentPolicy.revision}`);
    if (options.returnUpdatedPolicy !== false) {
      logger.debug(`returning updated policy for [${id}]`);
      return await this.get(soClient, id);
    }
    return newAgentPolicy;
  }
  async ensurePreconfiguredAgentPolicy(soClient, esClient, config) {
    const {
      id,
      space_id: kibanaSpaceId,
      ...preconfiguredAgentPolicy
    } = (0, _lodash.omit)(config, 'package_policies');
    const newAgentPolicyDefaults = {
      namespace: 'default',
      monitoring_enabled: ['logs', 'metrics']
    };
    const newAgentPolicy = {
      ...newAgentPolicyDefaults,
      ...preconfiguredAgentPolicy,
      is_preconfigured: true
    };
    if (!id) throw new _errors.AgentPolicyNotFoundError('Missing ID');
    return await this.ensureAgentPolicy(soClient, esClient, newAgentPolicy, id);
  }
  async ensureAgentPolicy(soClient, esClient, newAgentPolicy, id) {
    // For preconfigured policies with a specified ID
    const agentPolicy = await this.get(soClient, id, false).catch(() => null);
    if (!agentPolicy) {
      return {
        created: true,
        policy: await this.create(soClient, esClient, newAgentPolicy, {
          id
        })
      };
    }
    return {
      created: false,
      policy: agentPolicy
    };
  }
  hasAPMIntegration(agentPolicy) {
    return (0, _services.policyHasAPMIntegration)(agentPolicy);
  }
  hasFleetServerIntegration(agentPolicy) {
    return (0, _services.policyHasFleetServer)(agentPolicy);
  }
  hasSyntheticsIntegration(agentPolicy) {
    return (0, _services.policyHasSyntheticsIntegration)(agentPolicy);
  }
  async runExternalCallbacks(externalCallbackType, agentPolicy, requestSpaceId) {
    const logger = this.getLogger('runExternalCallbacks');
    try {
      const externalCallbacks = _.appContextService.getExternalCallbacks(externalCallbackType);
      let newAgentPolicy = agentPolicy;
      logger.debug(`Running [${externalCallbacks === null || externalCallbacks === void 0 ? void 0 : externalCallbacks.size}] external callbacks for [${externalCallbackType}]`);
      if (externalCallbacks && externalCallbacks.size > 0) {
        let updatedNewAgentPolicy = newAgentPolicy;
        for (const callback of externalCallbacks) {
          let result;
          if (externalCallbackType === 'agentPolicyCreate') {
            result = await callback(newAgentPolicy);
            updatedNewAgentPolicy = result;
          }
          if (externalCallbackType === 'agentPolicyUpdate') {
            result = await callback(newAgentPolicy);
            updatedNewAgentPolicy = result;
          }
          if (externalCallbackType === 'agentPolicyPostUpdate') {
            result = await callback(newAgentPolicy, requestSpaceId);
            updatedNewAgentPolicy = result;
          }
        }
        newAgentPolicy = updatedNewAgentPolicy;
      }
      logger.debug(`Running of external callbacks for [${externalCallbackType}] done`);
      return newAgentPolicy;
    } catch (error) {
      logger.error(`Error running external callbacks for [${externalCallbackType}]`);
      logger.error(error);
      throw error;
    }
  }
  async create(soClient, esClient, agentPolicy, options = {}) {
    var _agentPolicy$is_manag, _options$user, _appContextService$ge, _appContextService$ge2;
    const logger = this.getLogger('create');
    const savedObjectType = await getAgentPolicySavedObjectType();
    // Ensure an ID is provided, so we can include it in the audit logs below
    if (!options.id) {
      options.id = _server.SavedObjectsUtils.generateId();
    }
    logger.debug(`Creating new agent policy [${agentPolicy.name}] with id [${options.id}] using soClient scoped to [${soClient.getCurrentNamespace()}]`);
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'create',
      id: options.id,
      name: agentPolicy.name,
      savedObjectType
    });
    await this.runExternalCallbacks('agentPolicyCreate', agentPolicy);
    this.checkTamperProtectionLicense(agentPolicy);
    if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_protected) {
      logger.warn('Agent policy requires Elastic Defend integration to set tamper protection to true');
    }
    this.checkAgentless(agentPolicy);
    if (agentPolicy.supports_agentless && !agentPolicy.fleet_server_host_id) {
      const {
        fleetServerId
      } = _agentless_agent.agentlessAgentService.getDefaultSettings();
      agentPolicy.fleet_server_host_id = fleetServerId;
    }
    await this.requireUniqueName(soClient, agentPolicy);
    await (0, _policy_namespaces.validatePolicyNamespaceForSpace)({
      spaceId: soClient.getCurrentNamespace(),
      namespace: agentPolicy.namespace
    });
    const policyForOutputValidation = {
      ...agentPolicy,
      has_fleet_server: options === null || options === void 0 ? void 0 : options.hasFleetServer
    };
    await (0, _agent_policies.validateOutputForPolicy)(soClient, policyForOutputValidation, {}, (0, _services.getAllowedOutputTypesForAgentPolicy)(policyForOutputValidation));
    (0, _agent_policies.validateRequiredVersions)(agentPolicy.name, agentPolicy.required_versions);
    const newSo = await soClient.create(savedObjectType, {
      ...agentPolicy,
      status: 'active',
      is_managed: (_agentPolicy$is_manag = agentPolicy.is_managed) !== null && _agentPolicy$is_manag !== void 0 ? _agentPolicy$is_manag : false,
      revision: 1,
      updated_at: new Date().toISOString(),
      updated_by: (options === null || options === void 0 ? void 0 : (_options$user = options.user) === null || _options$user === void 0 ? void 0 : _options$user.username) || 'system',
      schema_version: _constants2.FLEET_AGENT_POLICIES_SCHEMA_VERSION,
      is_protected: false
    }, options).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Attempt to create agent policy [${agentPolicy.id}] failed`));
    await ((_appContextService$ge = _.appContextService.getUninstallTokenService()) === null || _appContextService$ge === void 0 ? void 0 : (_appContextService$ge2 = _appContextService$ge.scoped(soClient.getCurrentNamespace())) === null || _appContextService$ge2 === void 0 ? void 0 : _appContextService$ge2.generateTokenForPolicyId(newSo.id));
    await this.triggerAgentPolicyUpdatedEvent(esClient, 'created', newSo.id, {
      skipDeploy: options.skipDeploy,
      spaceId: soClient.getCurrentNamespace()
    });
    logger.debug(`Created new agent policy with id ${newSo.id}`);
    return {
      id: newSo.id,
      ...newSo.attributes
    };
  }
  async createWithPackagePolicies({
    soClient,
    esClient,
    agentPolicy,
    packagePolicies,
    options: {
      hasFleetServer,
      withSysMonitoring,
      monitoringEnabled,
      spaceId,
      user,
      authorizationHeader,
      force,
      forcePackagePolicyCreation
    }
  }) {
    const logger = _.appContextService.getLogger().get('createWithPackagePolicies');
    logger.debug(`Creating agent policy ${agentPolicy.name} with package policies`);
    const newAgentPolicy = await (0, _agent_policy_create.createAgentPolicyWithPackages)({
      soClient,
      esClient,
      agentPolicyService: this,
      newPolicy: agentPolicy,
      hasFleetServer,
      withSysMonitoring,
      monitoringEnabled,
      spaceId,
      user,
      authorizationHeader,
      force,
      forcePackagePolicyCreation
    });
    const createdPackagePolicyIds = [];
    try {
      for (const packagePolicy of packagePolicies) {
        const newPackagePolicy = {
          ...packagePolicy,
          policy_ids: [newAgentPolicy.id]
        };
        logger.debug(`Creating package policy ${packagePolicy.name} for agent policy ${newAgentPolicy.name}`);
        const createdPackagePolicy = await _package_policy.packagePolicyService.create(soClient, esClient, newPackagePolicy, {
          spaceId,
          user,
          bumpRevision: false,
          authorizationHeader,
          force
        });
        createdPackagePolicyIds.push(createdPackagePolicy.id);
      }
      const agentPolicyWithPackagePolicies = await this.get(soClient, newAgentPolicy.id);
      if (!agentPolicyWithPackagePolicies) {
        throw new _errors.AgentPolicyNotFoundError(`Could not retrieve created agent policy ${newAgentPolicy.id} after creating its package policies`);
      }
      return agentPolicyWithPackagePolicies;
    } catch (e) {
      logger.error(`Error creating package policies for agent policy ${newAgentPolicy.id}: ${e.message}`);
      logger.debug(`Rolling back policy creation: Deleting agent policy ${newAgentPolicy.id} and package policies ${createdPackagePolicyIds.join(', ')}`);

      // If there is an error creating package policies, delete any created package policy
      // and the parent agent policy
      const internalSOClient = _.appContextService.getInternalUserSOClient();
      const internalESClient = _.appContextService.getInternalUserESClient();
      if (createdPackagePolicyIds.length > 0) {
        await _package_policy.packagePolicyService.delete(internalSOClient, internalESClient, createdPackagePolicyIds, {
          force: true,
          skipUnassignFromAgentPolicies: true
        });
      }
      await this.delete(internalSOClient, internalESClient, newAgentPolicy.id, {
        force: true
      });
      throw e;
    }
  }
  async requireUniqueName(soClient, givenPolicy) {
    const savedObjectType = await getAgentPolicySavedObjectType();
    const results = await soClient.find({
      type: savedObjectType,
      searchFields: ['name'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(givenPolicy.name)
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Failed to find agent policies with name [${givenPolicy.name}]`));
    const idsWithName = results.total && results.saved_objects.map(({
      id
    }) => id);
    if (Array.isArray(idsWithName)) {
      const isEditingSelf = givenPolicy.id && idsWithName.includes(givenPolicy.id);
      if (!(givenPolicy !== null && givenPolicy !== void 0 && givenPolicy.supports_agentless) && !givenPolicy.id || !(givenPolicy !== null && givenPolicy !== void 0 && givenPolicy.supports_agentless) && !isEditingSelf) {
        const isSinglePolicy = idsWithName.length === 1;
        const existClause = isSinglePolicy ? `Agent Policy '${idsWithName[0]}' already exists` : `Agent Policies '${idsWithName.join(',')}' already exist`;
        throw new _errors.AgentPolicyNameExistsError(`${existClause} with name '${givenPolicy.name}'`);
      }
      if (givenPolicy !== null && givenPolicy !== void 0 && givenPolicy.supports_agentless && !givenPolicy.id) {
        const integrationName = givenPolicy.name.split(' ').pop();
        throw new _errors.AgentlessPolicyExistsRequestError(`${givenPolicy.name} already exist. Please rename the integration name ${integrationName}.`);
      }
    }
  }
  async get(soClient, id, withPackagePolicies = true, options = {}) {
    const logger = this.getLogger('get');
    logger.debug(() => {
      var _options$spaceId;
      return `Getting agent policy [${id}] for space [${(_options$spaceId = options.spaceId) !== null && _options$spaceId !== void 0 ? _options$spaceId : soClient.getCurrentNamespace()}]`;
    });

    // We're using `getByIds()` here, instead of just `soClient.get()`, because when using an unscoped
    // SO client we are not able to use `*` in the `esClient.get()` `options.namespace`.
    const [agentPolicy] = await this.getByIds(soClient, [{
      id,
      spaceId: options.spaceId
    }], {
      withPackagePolicies
    }).catch(async err => {
      // Emulate prior implementation that threw a Saved Objects error so that backwards compatibility is maintained
      if (err instanceof _errors.FleetNotFoundError) {
        throw _server.SavedObjectsErrorHelpers.createGenericNotFoundError(await getAgentPolicySavedObjectType(), id);
      }
      throw _server.SavedObjectsErrorHelpers.createBadRequestError(err.message);
    });
    return agentPolicy;
  }
  async getByIds(soClient, ids, options = {}) {
    const logger = this.getLogger('getByIds');
    logger.debug(() => `Getting agent policies ${JSON.stringify(ids)} using soClient scoped to [${soClient.getCurrentNamespace()}] and options [${JSON.stringify(options)}]`);
    const savedObjectType = await getAgentPolicySavedObjectType();
    const isSpacesEnabled = await (0, _helpers.isSpaceAwarenessEnabled)();
    const objects = ids.map(id => {
      var _id$spaceId;
      if (typeof id === 'string') {
        return {
          ...options,
          id,
          type: savedObjectType,
          namespaces: isSpacesEnabled && options.spaceId ? [options.spaceId] : undefined
        };
      }
      const spaceForThisAgentPolicy = (_id$spaceId = id.spaceId) !== null && _id$spaceId !== void 0 ? _id$spaceId : options.spaceId;
      return {
        ...options,
        id: id.id,
        namespaces: isSpacesEnabled && spaceForThisAgentPolicy ? [spaceForThisAgentPolicy] : undefined,
        type: savedObjectType
      };
    });
    logger.debug(() => `BulkGet input: ${JSON.stringify(objects)}`);
    const bulkGetResponse = await soClient.bulkGet(objects).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Failed to get agent policy by IDs`));
    const agentPolicies = await (0, _pMap.default)(bulkGetResponse.saved_objects, async agentPolicySO => {
      if (agentPolicySO.error) {
        if (options.ignoreMissing && agentPolicySO.error.statusCode === 404) {
          logger.debug(`Agent policy [${agentPolicySO.id}] was not found, but 'options.ignoreMissing' is 'true'`);
          return null;
        } else if (agentPolicySO.error.statusCode === 404) {
          logger.debug(`Agent policy [${agentPolicySO.id}] was not found. Throwing error`);
          throw new _errors.AgentPolicyNotFoundError(`Agent policy ${agentPolicySO.id} not found`, agentPolicySO.error);
        } else {
          logger.debug(`Error encountered while bulkGet agent policy [${agentPolicySO.id}]: ${agentPolicySO.error.message}`);
          throw new _errors.FleetError(agentPolicySO.error.message, agentPolicySO.error);
        }
      }
      const agentPolicy = (0, _utils2.mapAgentPolicySavedObjectToAgentPolicy)(agentPolicySO);
      if (options.withPackagePolicies) {
        logger.debug(`Retrieving package policies for agent policies [${agentPolicySO.id}]`);
        agentPolicy.package_policies = (await _package_policy.packagePolicyService.findAllForAgentPolicy(soClient, agentPolicySO.id)) || [];
      }
      return agentPolicy;
    }, {
      concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
    });
    const result = agentPolicies.filter(agentPolicy => agentPolicy !== null);
    for (const agentPolicy of result) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'get',
        id: agentPolicy.id,
        name: agentPolicy.name,
        savedObjectType
      });
    }
    logger.debug(`returning [${result.length}] agent policies`);
    return result;
  }
  async list(soClient, options) {
    const savedObjectType = await getAgentPolicySavedObjectType();
    const {
      page = 1,
      perPage = 20,
      sortField = 'updated_at',
      sortOrder = 'desc',
      kuery,
      withPackagePolicies = false,
      fields,
      spaceId
    } = options;
    const baseFindParams = {
      type: savedObjectType,
      sortField,
      sortOrder,
      page,
      perPage,
      ...(fields ? {
        fields
      } : {})
    };
    if (spaceId) {
      baseFindParams.namespaces = [spaceId];
    }
    const filter = kuery ? normalizeKuery(savedObjectType, kuery) : undefined;
    let agentPoliciesSO;
    try {
      agentPoliciesSO = await soClient.find({
        ...baseFindParams,
        filter
      });
    } catch (e) {
      var _e$output, _e$message;
      const isBadRequest = ((_e$output = e.output) === null || _e$output === void 0 ? void 0 : _e$output.statusCode) === 400;
      const isKQLSyntaxError = (_e$message = e.message) === null || _e$message === void 0 ? void 0 : _e$message.startsWith('KQLSyntaxError');
      if (isBadRequest && !isKQLSyntaxError) {
        // fall back to simple search if the kuery is just a search term i.e not KQL
        agentPoliciesSO = await soClient.find({
          ...baseFindParams,
          search: kuery
        }).catch(_utils.catchAndSetErrorStackTrace.withMessage('Failed to find agent policies using simple search term'));
      } else {
        throw e;
      }
    }
    const agentPolicies = agentPoliciesSO.saved_objects.map(agentPolicySO => {
      const agentPolicy = (0, _utils2.mapAgentPolicySavedObjectToAgentPolicy)(agentPolicySO);
      agentPolicy.agents = 0;
      return agentPolicy;
    });
    if (options.withAgentCount || withPackagePolicies) {
      await (0, _pMap.default)(agentPolicies, async agentPolicy => {
        if (withPackagePolicies) {
          agentPolicy.package_policies = (await _package_policy.packagePolicyService.findAllForAgentPolicy(soClient, agentPolicy.id)) || [];
        }
        if (options.withAgentCount) {
          await (0, _agents.getAgentsByKuery)(_.appContextService.getInternalUserESClient(), soClient, {
            showInactive: true,
            perPage: 0,
            page: 1,
            kuery: `${_constants2.AGENTS_PREFIX}.policy_id:"${agentPolicy.id}"`
          }).then(({
            total
          }) => agentPolicy.agents = total);
        } else {
          agentPolicy.agents = 0;
        }
        return agentPolicy;
      }, {
        concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
      });
    }
    for (const agentPolicy of agentPolicies) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'find',
        id: agentPolicy.id,
        name: agentPolicy.name,
        savedObjectType
      });
    }
    return {
      items: agentPolicies,
      total: agentPoliciesSO.total,
      page,
      perPage
    };
  }
  async update(soClient, esClient, id, agentPolicy, options) {
    var _agentPolicy$name, _options$bumpRevision, _options$skipValidati;
    const logger = this.getLogger('update');
    logger.debug(`Starting update of agent policy ${id}`);
    if (agentPolicy.name) {
      await this.requireUniqueName(soClient, {
        id,
        name: agentPolicy.name,
        supports_agentless: agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.supports_agentless
      });
    }
    if (agentPolicy.namespace) {
      await (0, _policy_namespaces.validatePolicyNamespaceForSpace)({
        spaceId: soClient.getCurrentNamespace(),
        namespace: agentPolicy.namespace
      });
    }
    const existingAgentPolicy = await this.get(soClient, id, true);
    if (!existingAgentPolicy) {
      throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
    }
    (0, _agent_policies.validateRequiredVersions)((_agentPolicy$name = agentPolicy.name) !== null && _agentPolicy$name !== void 0 ? _agentPolicy$name : id, agentPolicy.required_versions, existingAgentPolicy.required_versions, options === null || options === void 0 ? void 0 : options.isRequiredVersionsAuthorized);
    try {
      await this.runExternalCallbacks('agentPolicyUpdate', agentPolicy);
    } catch (error) {
      logger.error(`Error running external callbacks for agentPolicyUpdate`);
      if (error.apiPassThrough) {
        throw error;
      }
    }
    this.checkTamperProtectionLicense(agentPolicy);
    this.checkAgentless(agentPolicy);
    await this.checkForValidUninstallToken(agentPolicy, id);
    if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_protected && !(0, _services.policyHasEndpointSecurity)(existingAgentPolicy)) {
      logger.warn('Agent policy requires Elastic Defend integration to set tamper protection to true');
      // force agent policy to be false if elastic defend is not present
      agentPolicy.is_protected = false;
    }
    if (existingAgentPolicy.is_managed && !(options !== null && options !== void 0 && options.force)) {
      Object.entries(agentPolicy).filter(([key]) => !KEY_EDITABLE_FOR_MANAGED_POLICIES.includes(key)).forEach(([key, val]) => {
        if (!(0, _lodash.isEqual)(existingAgentPolicy[key], val)) {
          throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot update ${key}`);
        }
      });
    }
    const {
      monitoring_enabled: monitoringEnabled
    } = agentPolicy;
    const packagesToInstall = [];
    if (!existingAgentPolicy.monitoring_enabled && monitoringEnabled !== null && monitoringEnabled !== void 0 && monitoringEnabled.length) {
      packagesToInstall.push(_constants3.FLEET_ELASTIC_AGENT_PACKAGE);
    }
    if (packagesToInstall.length > 0) {
      await (0, _packages.bulkInstallPackages)({
        savedObjectsClient: soClient,
        esClient,
        packagesToInstall,
        spaceId: (options === null || options === void 0 ? void 0 : options.spaceId) || _constants.DEFAULT_SPACE_ID,
        authorizationHeader: options === null || options === void 0 ? void 0 : options.authorizationHeader,
        force: options === null || options === void 0 ? void 0 : options.force
      });
    }
    return this._update(soClient, esClient, id, agentPolicy, options === null || options === void 0 ? void 0 : options.user, {
      bumpRevision: (_options$bumpRevision = options === null || options === void 0 ? void 0 : options.bumpRevision) !== null && _options$bumpRevision !== void 0 ? _options$bumpRevision : true,
      removeProtection: false,
      skipValidation: (_options$skipValidati = options === null || options === void 0 ? void 0 : options.skipValidation) !== null && _options$skipValidati !== void 0 ? _options$skipValidati : false
    }).then(updatedAgentPolicy => {
      var _options$requestSpace;
      return this.runExternalCallbacks('agentPolicyPostUpdate', updatedAgentPolicy, (_options$requestSpace = options === null || options === void 0 ? void 0 : options.requestSpaceId) !== null && _options$requestSpace !== void 0 ? _options$requestSpace : _constants.DEFAULT_SPACE_ID);
    }).then(response => {
      logger.debug(`Update of agent policy [${id}] done`);
      return response;
    });
  }
  async copy(soClient, esClient, id, newAgentPolicyProps, options) {
    var _baseAgentPolicy$pack;
    const logger = this.getLogger('copy');
    logger.debug(`Starting copy of agent policy ${id}`);

    // Copy base agent policy
    const baseAgentPolicy = await this.get(soClient, id, true);
    if (!baseAgentPolicy) {
      throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
    }
    if ((_baseAgentPolicy$pack = baseAgentPolicy.package_policies) !== null && _baseAgentPolicy$pack !== void 0 && _baseAgentPolicy$pack.length) {
      const hasManagedPackagePolicies = baseAgentPolicy.package_policies.some(packagePolicy => packagePolicy.is_managed);
      if (hasManagedPackagePolicies) {
        throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot copy an agent policy ${id} that contains managed package policies`);
      }
    }
    const newAgentPolicy = await this.create(soClient, esClient, {
      ...(0, _lodash.pick)(baseAgentPolicy, ['namespace', 'monitoring_enabled', 'inactivity_timeout', 'unenroll_timeout', 'agent_features', 'overrides', 'data_output_id', 'monitoring_output_id', 'download_source_id', 'fleet_server_host_id', 'supports_agentless', 'global_data_tags', 'agentless', 'monitoring_pprof_enabled', 'monitoring_http', 'monitoring_diagnostics']),
      ...newAgentPolicyProps
    }, options);
    if (baseAgentPolicy.package_policies) {
      // Copy non-shared package policies and append (copy n) to their names.
      const basePackagePolicies = baseAgentPolicy.package_policies.filter(packagePolicy => packagePolicy.policy_ids.length < 2);
      if (basePackagePolicies.length > 0) {
        const newPackagePolicies = await (0, _pMap.default)(basePackagePolicies, async packagePolicy => {
          const {
            id: packagePolicyId,
            version,
            ...newPackagePolicy
          } = packagePolicy;
          const updatedPackagePolicy = {
            ...newPackagePolicy,
            name: await (0, _package_policies.incrementPackagePolicyCopyName)(soClient, packagePolicy.name)
          };
          return updatedPackagePolicy;
        });
        await _package_policy.packagePolicyService.bulkCreate(soClient, esClient, newPackagePolicies.map(newPackagePolicy => ({
          ...newPackagePolicy,
          policy_ids: [newAgentPolicy.id]
        })), {
          ...options,
          bumpRevision: false
        });
      }
      // Link shared package policies to new agent policy.
      const sharedBasePackagePolicies = baseAgentPolicy.package_policies.filter(packagePolicy => packagePolicy.policy_ids.length > 1);
      if (sharedBasePackagePolicies.length > 0) {
        const updatedSharedPackagePolicies = sharedBasePackagePolicies.map(packagePolicy => ({
          ...packagePolicy,
          policy_ids: [...packagePolicy.policy_ids, newAgentPolicy.id]
        }));
        await _package_policy.packagePolicyService.bulkUpdate(soClient, esClient, updatedSharedPackagePolicies);
      }
    }

    // Tamper protection is dependent on endpoint package policy
    // Match tamper protection setting to the original policy
    if (baseAgentPolicy.is_protected) {
      await this._update(soClient, esClient, newAgentPolicy.id, {
        is_protected: true
      }, options === null || options === void 0 ? void 0 : options.user, {
        bumpRevision: false,
        removeProtection: false,
        skipValidation: false
      });
    }
    const policyNeedsBump = baseAgentPolicy.package_policies || baseAgentPolicy.is_protected;

    // bump revision if agent policy is updated after creation
    if (policyNeedsBump) {
      await this.bumpRevision(soClient, esClient, newAgentPolicy.id, {
        user: options === null || options === void 0 ? void 0 : options.user
      });
    } else {
      await this.deployPolicy(soClient, newAgentPolicy.id);
    }

    // Get updated agent policy with package policies and adjusted tamper protection
    const updatedAgentPolicy = await this.get(soClient, newAgentPolicy.id, true);
    if (!updatedAgentPolicy) {
      throw new _errors.AgentPolicyNotFoundError('Copied agent policy not found');
    }
    logger.debug(`Completed copy of agent policy ${id}`);
    return updatedAgentPolicy;
  }
  async bumpRevision(soClient, esClient, id, options) {
    return (0, _apmUtils.withSpan)('bump_agent_policy_revision', async () => {
      var _options$removeProtec, _options$skipValidati2;
      await this._update(soClient, esClient, id, {}, options === null || options === void 0 ? void 0 : options.user, {
        bumpRevision: true,
        removeProtection: (_options$removeProtec = options === null || options === void 0 ? void 0 : options.removeProtection) !== null && _options$removeProtec !== void 0 ? _options$removeProtec : false,
        skipValidation: (_options$skipValidati2 = options === null || options === void 0 ? void 0 : options.skipValidation) !== null && _options$skipValidati2 !== void 0 ? _options$skipValidati2 : true,
        returnUpdatedPolicy: false,
        asyncDeploy: options === null || options === void 0 ? void 0 : options.asyncDeploy
      });
    });
  }

  /**
   * Remove an output from all agent policies that are using it, and replace the output by the default ones.
   * @param esClient
   * @param outputId
   */
  async removeOutputFromAll(esClient, outputId, options) {
    const savedObjectType = await getAgentPolicySavedObjectType();
    const agentPolicies = (await _.appContextService.getInternalUserSOClientWithoutSpaceExtension().find({
      type: savedObjectType,
      fields: ['revision', 'data_output_id', 'monitoring_output_id'],
      searchFields: ['data_output_id', 'monitoring_output_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(outputId),
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    })).saved_objects.map(_utils2.mapAgentPolicySavedObjectToAgentPolicy);
    if (agentPolicies.length > 0) {
      const getAgentPolicy = agentPolicy => ({
        data_output_id: agentPolicy.data_output_id === outputId ? null : agentPolicy.data_output_id,
        monitoring_output_id: agentPolicy.monitoring_output_id === outputId ? null : agentPolicy.monitoring_output_id
      });
      // Validate that the output is not used by any agent policy before updating any policy
      await (0, _pMap.default)(agentPolicies, async agentPolicy => {
        var _agentPolicy$space_id;
        const soClient = _.appContextService.getInternalUserSOClientForSpaceId((_agentPolicy$space_id = agentPolicy.space_ids) === null || _agentPolicy$space_id === void 0 ? void 0 : _agentPolicy$space_id[0]);
        const existingAgentPolicy = await this.get(soClient, agentPolicy.id, true);
        if (!existingAgentPolicy) {
          throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
        }
        await (0, _agent_policies.validateOutputForPolicy)(soClient, getAgentPolicy(agentPolicy), existingAgentPolicy, (0, _services.getAllowedOutputTypesForAgentPolicy)(existingAgentPolicy));
      }, {
        concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
      });
      await (0, _pMap.default)(agentPolicies, agentPolicy => {
        var _agentPolicy$space_id2;
        const soClient = _.appContextService.getInternalUserSOClientForSpaceId((_agentPolicy$space_id2 = agentPolicy.space_ids) === null || _agentPolicy$space_id2 === void 0 ? void 0 : _agentPolicy$space_id2[0]);
        return this.update(soClient, esClient, agentPolicy.id, getAgentPolicy(agentPolicy), {
          skipValidation: true,
          force: options === null || options === void 0 ? void 0 : options.force
        });
      }, {
        concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
      });
    }
  }

  /**
   * Remove a Fleet Server from all agent policies that are using it, to use the default one instead.
   */
  async removeFleetServerHostFromAll(esClient, fleetServerHostId, options) {
    const savedObjectType = await getAgentPolicySavedObjectType();
    const agentPolicies = (await _.appContextService.getInternalUserSOClientWithoutSpaceExtension().find({
      type: savedObjectType,
      fields: ['revision', 'fleet_server_host_id'],
      searchFields: ['fleet_server_host_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(fleetServerHostId),
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    })).saved_objects.map(_utils2.mapAgentPolicySavedObjectToAgentPolicy);
    if (agentPolicies.length > 0) {
      await (0, _pMap.default)(agentPolicies, agentPolicy => {
        var _agentPolicy$space_id3;
        return this.update(_.appContextService.getInternalUserSOClientForSpaceId((_agentPolicy$space_id3 = agentPolicy.space_ids) === null || _agentPolicy$space_id3 === void 0 ? void 0 : _agentPolicy$space_id3[0]), esClient, agentPolicy.id, {
          fleet_server_host_id: null
        }, {
          force: options === null || options === void 0 ? void 0 : options.force
        });
      }, {
        concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
      });
    }
  }
  async _bumpPolicies(internalSoClientWithoutSpaceExtension, savedObjectsResults, options) {
    const bumpedPolicies = savedObjectsResults.map(policy => {
      var _policy$namespaces;
      return {
        id: policy.id,
        type: policy.type,
        attributes: {
          ...policy.attributes,
          revision: policy.attributes.revision + 1,
          updated_at: new Date().toISOString(),
          updated_by: options !== null && options !== void 0 && options.user ? options.user.username : 'system'
        },
        version: policy.version,
        namespace: (_policy$namespaces = policy.namespaces) === null || _policy$namespaces === void 0 ? void 0 : _policy$namespaces[0]
      };
    });
    const bumpedPoliciesBySpaceId = (0, _lodash.groupBy)(bumpedPolicies, policy => policy.namespace || _constants.DEFAULT_SPACE_ID);
    const res = (await Promise.all(Object.entries(bumpedPoliciesBySpaceId).map(([spaceId, policies]) => internalSoClientWithoutSpaceExtension.bulkUpdate(policies, {
      namespace: spaceId
    })))).reduce((acc, r) => {
      if (r !== null && r !== void 0 && r.saved_objects) {
        acc.saved_objects.push(...r.saved_objects);
      }
      return acc;
    }, {
      saved_objects: []
    });
    await (0, _deploy_agent_policies_task.scheduleDeployAgentPoliciesTask)(_.appContextService.getTaskManagerStart(), savedObjectsResults.map(policy => {
      var _policy$namespaces2;
      return {
        id: policy.id,
        spaceId: (_policy$namespaces2 = policy.namespaces) === null || _policy$namespaces2 === void 0 ? void 0 : _policy$namespaces2[0]
      };
    }));
    return res;
  }
  async bumpAllAgentPoliciesForOutput(esClient, outputId, options) {
    const {
      useSpaceAwareness
    } = _.appContextService.getExperimentalFeatures();
    const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const savedObjectType = await getAgentPolicySavedObjectType();
    // All agent policies directly using output
    const agentPoliciesUsingOutput = await internalSoClientWithoutSpaceExtension.find({
      type: savedObjectType,
      fields: ['revision', 'data_output_id', 'monitoring_output_id', 'namespaces'],
      searchFields: ['data_output_id', 'monitoring_output_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(outputId),
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    });

    // All package policies directly using output
    const packagePoliciesUsingOutput = await internalSoClientWithoutSpaceExtension.find({
      type: await (0, _package_policy.getPackagePolicySavedObjectType)(),
      fields: ['output_id', 'namespaces', 'policy_ids'],
      searchFields: ['output_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(outputId),
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    });
    const agentPolicyIdsDirectlyUsingOutput = agentPoliciesUsingOutput.saved_objects.map(agentPolicySO => agentPolicySO.id);
    const agentPolicyIdsOfPackagePoliciesUsingOutput = packagePoliciesUsingOutput.saved_objects.reduce((acc, packagePolicySO) => {
      const newIds = packagePolicySO.attributes.policy_ids.filter(policyId => {
        return !agentPolicyIdsDirectlyUsingOutput.includes(policyId);
      });
      return new Set([...acc, ...newIds]);
    }, new Set());

    // Agent policies of the identified package policies, excluding ones already retrieved directly
    const agentPoliciesOfPackagePoliciesUsingOutput = await internalSoClientWithoutSpaceExtension.bulkGet([...agentPolicyIdsOfPackagePoliciesUsingOutput].map(id => ({
      type: savedObjectType,
      id,
      fields: ['revision', 'data_output_id', 'monitoring_output_id', 'namespaces'],
      ...(useSpaceAwareness ? {
        namespaces: ['*']
      } : {})
    })));
    return this._bumpPolicies(internalSoClientWithoutSpaceExtension, [...agentPoliciesUsingOutput.saved_objects, ...agentPoliciesOfPackagePoliciesUsingOutput.saved_objects], options);
  }
  async bumpAllAgentPolicies(esClient, options) {
    const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const savedObjectType = await getAgentPolicySavedObjectType();
    const currentPolicies = await internalSoClientWithoutSpaceExtension.find({
      type: savedObjectType,
      fields: ['name', 'revision', 'namespaces'],
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    });
    return this._bumpPolicies(internalSoClientWithoutSpaceExtension, currentPolicies.saved_objects, options);
  }
  async delete(soClient, esClient, id, options) {
    const logger = this.getLogger('delete');
    logger.debug(`Deleting agent policy ${id}`);
    const savedObjectType = await getAgentPolicySavedObjectType();
    const agentPolicy = await this.get(soClient, id, false);
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'delete',
      id,
      name: agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.name,
      savedObjectType
    });
    if (!agentPolicy) {
      throw new _errors.AgentPolicyNotFoundError('Agent policy not found');
    }
    if (agentPolicy.is_managed && !(options !== null && options !== void 0 && options.force)) {
      throw new _errors.HostedAgentPolicyRestrictionRelatedError(`Cannot delete hosted agent policy ${id}`);
    }

    // Prevent deleting policy when assigned agents are inactive
    const {
      total
    } = await (0, _agents.getAgentsByKuery)(esClient, soClient, {
      showInactive: true,
      perPage: 0,
      page: 1,
      kuery: `${_constants2.AGENTS_PREFIX}.policy_id:"${id}"`
    });
    if (total > 0 && !(agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.supports_agentless)) {
      throw new _errors.FleetError('Cannot delete an agent policy that is assigned to any active or inactive agents');
    }
    if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.supports_agentless) {
      logger.debug(`Starting  unenrolling agent from agentless policy ${id}`);
      // unenroll  offline agents for agentless policies first to avoid 404 Save Object error
      await this.triggerAgentPolicyUpdatedEvent(esClient, 'deleted', id, {
        spaceId: soClient.getCurrentNamespace()
      });
      try {
        // Deleting agentless deployment
        await _agentless_agent.agentlessAgentService.deleteAgentlessAgent(id);
        logger.debug(`[Agentless API] Successfully deleted agentless deployment for single agent policy id ${id}`);
      } catch (error) {
        logger.error(`[Agentless API] Error deleting agentless deployment for single agent policy id ${id}`);
        logger.error(error);
      }
    }
    const packagePolicies = await _package_policy.packagePolicyService.findAllForAgentPolicy(soClient, id);
    if (packagePolicies.length) {
      const hasManagedPackagePolicies = packagePolicies.some(packagePolicy => packagePolicy.is_managed);
      if (hasManagedPackagePolicies && !(options !== null && options !== void 0 && options.force)) {
        throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot delete agent policy ${id} that contains managed package policies`);
      }
      const {
        policiesWithSingleAP: packagePoliciesToDelete,
        policiesWithMultipleAP
      } = this.packagePoliciesWithSingleAndMultiplePolicies(packagePolicies);
      if (packagePoliciesToDelete.length > 0) {
        await _package_policy.packagePolicyService.delete(soClient, esClient, packagePoliciesToDelete.map(p => p.id), {
          force: options === null || options === void 0 ? void 0 : options.force,
          skipUnassignFromAgentPolicies: true
        });
        logger.debug(`Deleted package policies with single agent policy with ids ${packagePoliciesToDelete.map(policy => policy.id).join(', ')}`);
      }
      if (policiesWithMultipleAP.length > 0) {
        await _package_policy.packagePolicyService.bulkUpdate(soClient, esClient, policiesWithMultipleAP.map(policy => {
          const newPolicyIds = policy.policy_ids.filter(policyId => policyId !== id);
          return {
            ...policy,
            policy_id: newPolicyIds[0],
            policy_ids: newPolicyIds
          };
        }));
        logger.debug(`Updated package policies with multiple agent policies with ids ${policiesWithMultipleAP.map(policy => policy.id).join(', ')}`);
      }
    }
    if (agentPolicy.is_preconfigured && !(options !== null && options !== void 0 && options.force)) {
      await soClient.create(_constants2.PRECONFIGURATION_DELETION_RECORD_SAVED_OBJECT_TYPE, {
        id: String(id)
      }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Failed to create [${_constants2.PRECONFIGURATION_DELETION_RECORD_SAVED_OBJECT_TYPE}] with id [${id}]`));
    }
    await soClient.delete(savedObjectType, id, {
      force: true // need to delete through multiple space
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Failed to delete agent policy [${id}]`));
    if (!(agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.supports_agentless)) {
      await this.triggerAgentPolicyUpdatedEvent(esClient, 'deleted', id, {
        spaceId: soClient.getCurrentNamespace()
      });
    }

    // cleanup .fleet-policies docs on delete
    await this.deleteFleetServerPoliciesForPolicyId(esClient, id);
    logger.debug(`Deleted agent policy ${id}`);
    return {
      id,
      name: agentPolicy.name
    };
  }
  async deployPolicy(soClient, agentPolicyId, agentPolicy, options) {
    await this.deployPolicies(soClient, [agentPolicyId], agentPolicy ? [agentPolicy] : undefined, options);
  }
  async deployPolicies(soClient, agentPolicyIds, agentPolicies, options) {
    const logger = this.getLogger('deployPolicies');
    logger.debug(() => `Deploying agent policies: [${agentPolicyIds.join(', ')}] using soClient scoped to [${soClient.getCurrentNamespace()}]`);

    // Use internal ES client so we have permissions to write to .fleet* indices
    const esClient = _.appContextService.getInternalUserESClient();
    const defaultOutputId = await _output.outputService.getDefaultDataOutputId();
    if (!defaultOutputId) {
      logger.debug(`Deployment canceled!! Default output ID is not defined.`);
      return;
    }
    for (const policyId of agentPolicyIds) {
      _audit_logging.auditLoggingService.writeCustomAuditLog({
        message: `User deploying policy [id=${policyId}]`
      });
    }
    const policies = await agentPolicyService.getByIds(soClient, agentPolicyIds);
    const policiesMap = (0, _lodash.keyBy)(policies, 'id');
    logger.debug(`Retrieving full agent policies`);
    const fullPolicies = await (0, _pMap.default)(agentPolicyIds,
    // There are some potential performance concerns around using `getFullAgentPolicy` in this context, e.g.
    // re-fetching outputs, settings, and upgrade download source URI data for each policy. This could potentially
    // be a bottleneck in environments with several thousand agent policies being deployed here.
    agentPolicyId => agentPolicyService.getFullAgentPolicy(soClient, agentPolicyId, {
      agentPolicy: agentPolicies === null || agentPolicies === void 0 ? void 0 : agentPolicies.find(policy => policy.id === agentPolicyId)
    }).then(response => {
      if (!response) {
        logger.debug(`Unable to retrieve FULL agent policy for [${agentPolicyId}] - Deployment will not be done for this policy`);
      }
      return response;
    }), {
      concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS_20
    });
    const fleetServerPolicies = fullPolicies.reduce((acc, fullPolicy) => {
      if (!fullPolicy || !fullPolicy.revision) {
        return acc;
      }
      const policy = policiesMap[fullPolicy.id];
      if (!policy) {
        return acc;
      }
      const fleetServerPolicy = {
        '@timestamp': new Date().toISOString(),
        revision_idx: fullPolicy.revision,
        coordinator_idx: 0,
        namespaces: fullPolicy.namespaces,
        data: fullPolicy,
        policy_id: fullPolicy.id,
        default_fleet_server: policy.is_default_fleet_server === true
      };
      acc.push(fleetServerPolicy);
      return acc;
    }, []);
    logger.debug(() => `Deploying policies: ${fleetServerPolicies.map(pol => `${pol.policy_id}:${pol.revision_idx}`).join(', ')}`);
    const fleetServerPoliciesBulkBody = fleetServerPolicies.flatMap(fleetServerPolicy => [{
      index: {
        _id: (0, _uuid.v5)(`${fleetServerPolicy.policy_id}:${fleetServerPolicy.revision_idx}`, _uuid.v5.DNS)
      }
    }, fleetServerPolicy]);
    const bulkResponse = await esClient.bulk({
      index: _constants3.AGENT_POLICY_INDEX,
      operations: fleetServerPoliciesBulkBody,
      refresh: 'wait_for'
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage('ES bulk operation failed'));
    logger.debug(`Bulk update against index [${_constants3.AGENT_POLICY_INDEX}] with deployment updates done`);
    if (bulkResponse.errors) {
      const erroredDocuments = bulkResponse.items.reduce((acc, item) => {
        const value = item.index;
        if (!value || !value.error) {
          return acc;
        }
        acc.push(value);
        return acc;
      }, []);
      logger.warn(`Failed to index documents during policy deployment: ${JSON.stringify(erroredDocuments)}`);
    }
    for (const agentPolicy of policies) {
      if (!agentPolicy.supports_agentless) {
        continue;
      }
      try {
        await _agentless_agent.agentlessAgentService.createAgentlessAgent(esClient, soClient, agentPolicy);
        logger.debug(`[Agentless API] Successfully deployed agentless deployment for single agent policy id ${agentPolicy.id}`);
      } catch (error) {
        // Swallow errors
        logger.error(`[Agentless API] Error deploying agentless deployment for single agent policy id ${agentPolicy.id}`, {
          error
        });
        if (options !== null && options !== void 0 && options.throwOnAgentlessError) {
          throw error;
        }
      }
    }
    const filteredFleetServerPolicies = fleetServerPolicies.filter(fleetServerPolicy => {
      const policy = policiesMap[fleetServerPolicy.policy_id];
      return !policy.schema_version || (0, _semver.lt)(policy.schema_version, _constants2.FLEET_AGENT_POLICIES_SCHEMA_VERSION);
    });
    await (0, _pMap.default)(filteredFleetServerPolicies, fleetServerPolicy => {
      logger.debug(`Updating agent policy id [${fleetServerPolicy.policy_id}] following successful deployment of fleet server policy`);

      // There are some potential performance concerns around using `agentPolicyService.update` in this context.
      // This could potentially be a bottleneck in environments with several thousand agent policies being deployed here.
      return agentPolicyService.update(soClient, esClient, fleetServerPolicy.policy_id, {
        schema_version: _constants2.FLEET_AGENT_POLICIES_SCHEMA_VERSION
      }, {
        force: true
      });
    }, {
      concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
    });
  }
  async deleteFleetServerPoliciesForPolicyId(esClient, agentPolicyId) {
    _audit_logging.auditLoggingService.writeCustomAuditLog({
      message: `User deleting policy [id=${agentPolicyId}]`
    });
    let hasMore = true;
    while (hasMore) {
      var _res$deleted;
      const res = await esClient.deleteByQuery({
        index: _constants3.AGENT_POLICY_INDEX,
        ignore_unavailable: true,
        scroll_size: _constants2.SO_SEARCH_LIMIT,
        refresh: true,
        query: {
          term: {
            policy_id: agentPolicyId
          }
        }
      });
      hasMore = ((_res$deleted = res.deleted) !== null && _res$deleted !== void 0 ? _res$deleted : 0) === _constants2.SO_SEARCH_LIMIT;
    }
  }
  async getLatestFleetPolicy(esClient, agentPolicyId) {
    const res = await esClient.search({
      index: _constants3.AGENT_POLICY_INDEX,
      ignore_unavailable: true,
      rest_total_hits_as_int: true,
      query: {
        term: {
          policy_id: agentPolicyId
        }
      },
      size: 1,
      sort: [{
        revision_idx: {
          order: 'desc'
        }
      }]
    });
    if (res.hits.total === 0) {
      return null;
    }
    return res.hits.hits[0]._source;
  }
  async getFullAgentConfigMap(soClient, id, agentVersion, options) {
    const fullAgentPolicy = await (0, _agent_policies.getFullAgentPolicy)(soClient, id, options);
    if (fullAgentPolicy) {
      const fullAgentConfigMap = {
        apiVersion: 'v1',
        kind: 'ConfigMap',
        metadata: {
          name: 'agent-node-datastreams',
          namespace: 'kube-system',
          labels: {
            'k8s-app': 'elastic-agent'
          }
        },
        data: {
          'agent.yml': fullAgentPolicy
        }
      };
      const configMapYaml = (0, _agent_cm_to_yaml.fullAgentConfigMapToYaml)(fullAgentConfigMap, _jsYaml.dump);
      const updateManifestVersion = _elastic_agent_manifest.elasticAgentStandaloneManifest.replace('VERSION', agentVersion);
      const fixedAgentYML = configMapYaml.replace('agent.yml:', 'agent.yml: |-');
      return [fixedAgentYML, updateManifestVersion].join('\n');
    } else {
      return '';
    }
  }
  async getFullAgentManifest(fleetServer, enrolToken, agentVersion) {
    const updateManifestVersion = _elastic_agent_manifest.elasticAgentManagedManifest.replace('VERSION', agentVersion);
    let updateManifest = updateManifestVersion;
    if (fleetServer !== '') {
      updateManifest = updateManifest.replace('https://fleet-server:8220', fleetServer);
    }
    if (enrolToken !== '') {
      updateManifest = updateManifest.replace('token-id', enrolToken);
    }
    return updateManifest;
  }
  async getFullAgentPolicy(soClient, id, options) {
    return (0, _agent_policies.getFullAgentPolicy)(soClient, id, options);
  }

  /**
   * Remove a download source from all agent policies that are using it, and replace the output by the default ones.
   * @param soClient
   * @param esClient
   * @param downloadSourceId
   */
  async removeDefaultSourceFromAll(esClient, downloadSourceId) {
    const savedObjectType = await getAgentPolicySavedObjectType();
    const agentPolicies = (await _.appContextService.getInternalUserSOClientWithoutSpaceExtension().find({
      type: savedObjectType,
      fields: ['revision', 'download_source_id'],
      searchFields: ['download_source_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(downloadSourceId),
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    })).saved_objects.map(_utils2.mapAgentPolicySavedObjectToAgentPolicy);
    if (agentPolicies.length > 0) {
      await (0, _pMap.default)(agentPolicies, agentPolicy => {
        var _agentPolicy$space_id4;
        return this.update(_.appContextService.getInternalUserSOClientForSpaceId((_agentPolicy$space_id4 = agentPolicy.space_ids) === null || _agentPolicy$space_id4 === void 0 ? void 0 : _agentPolicy$space_id4[0]), esClient, agentPolicy.id, {
          download_source_id: agentPolicy.download_source_id === downloadSourceId ? null : agentPolicy.download_source_id
        });
      }, {
        concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
      });
    }
  }
  async bumpAllAgentPoliciesForDownloadSource(esClient, downloadSourceId, options) {
    const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const savedObjectType = await getAgentPolicySavedObjectType();
    const currentPolicies = await internalSoClientWithoutSpaceExtension.find({
      type: savedObjectType,
      fields: ['revision', 'download_source_id', 'namespaces'],
      searchFields: ['download_source_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(downloadSourceId),
      perPage: _constants2.SO_SEARCH_LIMIT,
      namespaces: ['*']
    });
    return this._bumpPolicies(internalSoClientWithoutSpaceExtension, currentPolicies.saved_objects, options);
  }
  async bumpAllAgentPoliciesForFleetServerHosts(esClient, fleetServerHostId, options) {
    const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const savedObjectType = await getAgentPolicySavedObjectType();
    const currentPolicies = await internalSoClientWithoutSpaceExtension.find({
      type: savedObjectType,
      fields: ['revision', 'fleet_server_host_id', 'namespaces'],
      searchFields: ['fleet_server_host_id'],
      search: (0, _saved_object.escapeSearchQueryPhrase)(fleetServerHostId),
      perPage: _constants2.SO_SEARCH_LIMIT
    });
    return this._bumpPolicies(internalSoClientWithoutSpaceExtension, currentPolicies.saved_objects, options);
  }
  async bumpAgentPoliciesByIds(agentPolicyIds, options, spaceId) {
    const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const savedObjectType = await getAgentPolicySavedObjectType();
    const objects = agentPolicyIds.map(id => ({
      id,
      type: savedObjectType
    }));
    const opts = spaceId ? {
      namespace: spaceId
    } : undefined;
    const bulkGetResponse = await internalSoClientWithoutSpaceExtension.bulkGet(objects, opts).catch(_utils.catchAndSetErrorStackTrace.withMessage(`Failed to bulk get agent policies [${agentPolicyIds.join(', ')}]`));
    return this._bumpPolicies(internalSoClientWithoutSpaceExtension, bulkGetResponse.saved_objects, options);
  }
  async getInactivityTimeouts() {
    const savedObjectType = await getAgentPolicySavedObjectType();
    const internalSoClientWithoutSpaceExtension = _.appContextService.getInternalUserSOClientWithoutSpaceExtension();
    const findRes = await internalSoClientWithoutSpaceExtension.find({
      type: savedObjectType,
      page: 1,
      perPage: _constants2.SO_SEARCH_LIMIT,
      filter: `${savedObjectType}.attributes.inactivity_timeout > 0`,
      fields: [`inactivity_timeout`],
      namespaces: ['*']
    });
    const groupedResults = (0, _lodash.groupBy)(findRes.saved_objects, so => so.attributes.inactivity_timeout);
    return Object.entries(groupedResults).map(([inactivityTimeout, policies]) => ({
      inactivityTimeout: parseInt(inactivityTimeout, 10),
      policyIds: policies.map(policy => policy.id)
    }));
  }
  async turnOffAgentTamperProtections(soClient) {
    const savedObjectType = await getAgentPolicySavedObjectType();
    const agentPolicyFetcher = await this.fetchAllAgentPolicies(soClient, {
      kuery: `${savedObjectType}.is_protected: true`
    });
    const updatedAgentPolicies = [];
    for await (const agentPolicyPageResults of agentPolicyFetcher) {
      const {
        saved_objects: bulkUpdateSavedObjects
      } = await soClient.bulkUpdate(agentPolicyPageResults.map(agentPolicy => {
        const {
          id,
          revision
        } = agentPolicy;
        return {
          id,
          type: savedObjectType,
          attributes: {
            is_protected: false,
            revision: revision + 1,
            updated_at: new Date().toISOString(),
            updated_by: 'system'
          }
        };
      })).catch(_utils.catchAndSetErrorStackTrace.withMessage('bulkUpdate of agent policies failed'));
      updatedAgentPolicies.push(...bulkUpdateSavedObjects);
    }
    if (!updatedAgentPolicies.length) {
      return {
        updatedPolicies: null,
        failedPolicies: []
      };
    }
    const failedPolicies = [];
    updatedAgentPolicies.forEach(policy => {
      if (policy.error) {
        failedPolicies.push({
          id: policy.id,
          error: policy.error
        });
      }
    });
    const updatedPoliciesSuccess = updatedAgentPolicies.filter(policy => !policy.error);
    await (0, _deploy_agent_policies_task.scheduleDeployAgentPoliciesTask)(_.appContextService.getTaskManagerStart(), updatedPoliciesSuccess.map(policy => {
      var _policy$namespaces3;
      return {
        id: policy.id,
        spaceId: (_policy$namespaces3 = policy.namespaces) === null || _policy$namespaces3 === void 0 ? void 0 : _policy$namespaces3[0]
      };
    }));
    return {
      updatedPolicies: updatedPoliciesSuccess,
      failedPolicies
    };
  }
  async getAllManagedAgentPolicies(soClient) {
    const savedObjectType = await getAgentPolicySavedObjectType();
    const {
      saved_objects: agentPolicies
    } = await soClient.find({
      type: savedObjectType,
      page: 1,
      perPage: _constants2.SO_SEARCH_LIMIT,
      filter: normalizeKuery(savedObjectType, 'ingest-agent-policies.is_managed: true')
    }).catch(_utils.catchAndSetErrorStackTrace.withMessage('Failed to get all managed agent policies'));
    return agentPolicies;
  }
  async fetchAllAgentPolicyIds(soClient, {
    perPage = 1000,
    kuery = undefined,
    spaceId = undefined
  } = {}) {
    const savedObjectType = await getAgentPolicySavedObjectType();
    return (0, _create_so_find_iterable.createSoFindIterable)({
      soClient,
      findRequest: {
        type: savedObjectType,
        perPage,
        sortField: 'created_at',
        sortOrder: 'asc',
        fields: ['id', 'name'],
        filter: kuery ? normalizeKuery(savedObjectType, kuery) : undefined,
        namespaces: spaceId ? [spaceId] : undefined
      },
      resultsMapper: data => {
        return data.saved_objects.map(agentPolicySO => {
          _audit_logging.auditLoggingService.writeCustomSoAuditLog({
            action: 'find',
            id: agentPolicySO.id,
            name: agentPolicySO.attributes.name,
            savedObjectType
          });
          return agentPolicySO.id;
        });
      }
    });
  }
  async fetchAllAgentPolicies(soClient, {
    perPage = 1000,
    kuery,
    sortOrder = 'asc',
    sortField = 'created_at',
    fields = [],
    spaceId = undefined
  } = {}) {
    const savedObjectType = await getAgentPolicySavedObjectType();
    return (0, _create_so_find_iterable.createSoFindIterable)({
      soClient,
      findRequest: {
        type: savedObjectType,
        sortField,
        sortOrder,
        perPage,
        fields,
        filter: kuery ? normalizeKuery(savedObjectType, kuery) : undefined,
        namespaces: spaceId ? [spaceId] : undefined
      },
      resultsMapper(data) {
        return data.saved_objects.map(agentPolicySO => {
          _audit_logging.auditLoggingService.writeCustomSoAuditLog({
            action: 'find',
            id: agentPolicySO.id,
            savedObjectType
          });
          return (0, _utils2.mapAgentPolicySavedObjectToAgentPolicy)(agentPolicySO);
        });
      }
    });
  }

  // Get all the outputs per agent policy
  async getAllOutputsForPolicy(agentPolicy) {
    var _monitoringOutput$nam, _monitoringOutput$id, _dataOutput$name, _dataOutput$id, _integrationsDataOutp;
    const logger = this.getLogger('getAllOutputsForPolicy');
    const [defaultDataOutputId, defaultMonitoringOutputId] = await Promise.all([_output.outputService.getDefaultDataOutputId(), _output.outputService.getDefaultMonitoringOutputId()]);
    if (!defaultDataOutputId) {
      throw new _errors.OutputNotFoundError('Default output is not setup');
    }
    const dataOutputId = agentPolicy.data_output_id || defaultDataOutputId;
    const monitoringOutputId = agentPolicy.monitoring_output_id || defaultMonitoringOutputId || dataOutputId;
    const outputIds = (0, _lodash.uniq)([dataOutputId, monitoringOutputId]);
    const fetchedOutputs = await _output.outputService.bulkGet(outputIds, {
      ignoreNotFound: true
    });
    const dataOutput = fetchedOutputs.find(output => output.id === dataOutputId);
    const monitoringOutput = fetchedOutputs.find(output => output.id === monitoringOutputId);
    let integrationsDataOutputs = [];
    if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.package_policies) {
      const integrationsWithOutputs = agentPolicy.package_policies.filter(pkgPolicy => !!(pkgPolicy !== null && pkgPolicy !== void 0 && pkgPolicy.output_id));
      integrationsDataOutputs = await (0, _pMap.default)(integrationsWithOutputs, async pkgPolicy => {
        var _pkgPolicy$output_id;
        if (pkgPolicy !== null && pkgPolicy !== void 0 && pkgPolicy.output_id) {
          try {
            const output = await _output.outputService.get(pkgPolicy.output_id);
            return {
              integrationPolicyName: pkgPolicy === null || pkgPolicy === void 0 ? void 0 : pkgPolicy.name,
              id: output.id,
              name: output.name
            };
          } catch (error) {
            logger.error(`error while retrieving output with id "${pkgPolicy.output_id}": ${error}`);
          }
        }
        return {
          integrationPolicyName: pkgPolicy === null || pkgPolicy === void 0 ? void 0 : pkgPolicy.name,
          id: (_pkgPolicy$output_id = pkgPolicy === null || pkgPolicy === void 0 ? void 0 : pkgPolicy.output_id) !== null && _pkgPolicy$output_id !== void 0 ? _pkgPolicy$output_id : ''
        };
      }, {
        concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS_20
      });
    }
    const outputs = {
      monitoring: {
        output: {
          name: (_monitoringOutput$nam = monitoringOutput === null || monitoringOutput === void 0 ? void 0 : monitoringOutput.name) !== null && _monitoringOutput$nam !== void 0 ? _monitoringOutput$nam : '',
          id: (_monitoringOutput$id = monitoringOutput === null || monitoringOutput === void 0 ? void 0 : monitoringOutput.id) !== null && _monitoringOutput$id !== void 0 ? _monitoringOutput$id : ''
        }
      },
      data: {
        output: {
          name: (_dataOutput$name = dataOutput === null || dataOutput === void 0 ? void 0 : dataOutput.name) !== null && _dataOutput$name !== void 0 ? _dataOutput$name : '',
          id: (_dataOutput$id = dataOutput === null || dataOutput === void 0 ? void 0 : dataOutput.id) !== null && _dataOutput$id !== void 0 ? _dataOutput$id : ''
        },
        integrations: (_integrationsDataOutp = integrationsDataOutputs) !== null && _integrationsDataOutp !== void 0 ? _integrationsDataOutp : []
      }
    };
    return outputs;
  }
  async listAllOutputsForPolicies(agentPolicies) {
    const allOutputs = await (0, _pMap.default)(agentPolicies, async agentPolicy => {
      const output = await this.getAllOutputsForPolicy(agentPolicy);
      return {
        agentPolicyId: agentPolicy.id,
        ...output
      };
    }, {
      concurrency: _constants2.MAX_CONCURRENT_AGENT_POLICIES_OPERATIONS
    });
    return allOutputs;
  }
  checkTamperProtectionLicense(agentPolicy) {
    if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_protected && !_license.licenseService.isPlatinum()) {
      throw new _errors.FleetUnauthorizedError('Tamper protection requires Platinum license');
    }
  }
  async checkForValidUninstallToken(agentPolicy, policyId) {
    if (agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.is_protected) {
      const uninstallTokenService = _.appContextService.getUninstallTokenService();
      const uninstallTokenError = await (uninstallTokenService === null || uninstallTokenService === void 0 ? void 0 : uninstallTokenService.checkTokenValidityForPolicy(policyId));
      if (uninstallTokenError) {
        throw new _errors.FleetError(`Cannot enable Agent Tamper Protection: ${uninstallTokenError.error.message}`);
      }
    }
  }
  checkAgentless(agentPolicy) {
    if (!(0, _agentless.isAgentlessEnabled)() && agentPolicy !== null && agentPolicy !== void 0 && agentPolicy.supports_agentless) {
      throw new _errors.AgentPolicyInvalidError('supports_agentless is only allowed in serverless and cloud environments that support the agentless feature');
    }
  }
  packagePoliciesWithSingleAndMultiplePolicies(packagePolicies) {
    // Find package policies that don't have multiple agent policies and mark them for deletion
    const policiesWithSingleAP = packagePolicies.filter(policy => !(policy !== null && policy !== void 0 && policy.policy_ids) || (policy === null || policy === void 0 ? void 0 : policy.policy_ids.length) <= 1);
    const policiesWithMultipleAP = packagePolicies.filter(policy => (policy === null || policy === void 0 ? void 0 : policy.policy_ids) && (policy === null || policy === void 0 ? void 0 : policy.policy_ids.length) > 1);
    return {
      policiesWithSingleAP,
      policiesWithMultipleAP
    };
  }
}
const agentPolicyService = exports.agentPolicyService = new AgentPolicyService();
async function addPackageToAgentPolicy(soClient, esClient, agentPolicy, packageInfo, packagePolicyName, packagePolicyId, packagePolicyDescription, transformPackagePolicy, bumpAgentPolicyRevison = false) {
  var _agentPolicy$namespac;
  const basePackagePolicy = (0, _services.packageToPackagePolicy)(packageInfo, agentPolicy.id, (_agentPolicy$namespac = agentPolicy.namespace) !== null && _agentPolicy$namespac !== void 0 ? _agentPolicy$namespac : 'default', packagePolicyName, packagePolicyDescription);
  const newPackagePolicy = transformPackagePolicy ? transformPackagePolicy(basePackagePolicy) : basePackagePolicy;

  // If an ID is provided via preconfiguration, use that value. Otherwise fall back to
  // a UUID v5 value seeded from the agent policy's ID and the provided package policy name.
  const id = packagePolicyId ? String(packagePolicyId) : (0, _uuid.v5)(`${agentPolicy.id}-${packagePolicyName}`, _constants3.UUID_V5_NAMESPACE);
  await _package_policy.packagePolicyService.create(soClient, esClient, newPackagePolicy, {
    id,
    bumpRevision: bumpAgentPolicyRevison,
    skipEnsureInstalled: true,
    skipUniqueNameVerification: true,
    overwrite: true,
    force: true,
    // To add package to managed policy we need the force flag
    packageInfo
  });
}