"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchRelatedSavedObjects = fetchRelatedSavedObjects;
var _lodash = require("lodash");
var _output = require("../output");
var _source_uri_utils = require("../../routes/agent/source_uri_utils");
var _fleet_server_host = require("../fleet_server_host");
var _app_context = require("../app_context");
var _fleet_proxies = require("../fleet_proxies");
var _errors = require("../../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchRelatedSavedObjects(soClient, agentPolicy) {
  const logger = _app_context.appContextService.getLogger().get('fetchRelatedSavedObjects');
  logger.debug(() => `getting related saved objects for policy [${agentPolicy.id}] with soClient scoped to [${soClient.getCurrentNamespace()}]`);
  const [defaultDataOutputId, defaultMonitoringOutputId] = await Promise.all([_output.outputService.getDefaultDataOutputId(), _output.outputService.getDefaultMonitoringOutputId()]);
  if (!defaultDataOutputId) {
    throw new _errors.OutputNotFoundError('Default output is not setup');
  }
  const dataOutputId = agentPolicy.data_output_id || defaultDataOutputId;
  const monitoringOutputId = agentPolicy.monitoring_output_id || defaultMonitoringOutputId || dataOutputId;
  const outputIds = (0, _lodash.uniq)([dataOutputId, monitoringOutputId, ...(agentPolicy.package_policies || []).reduce((acc, packagePolicy) => {
    if (packagePolicy.output_id) {
      acc.push(packagePolicy.output_id);
    }
    return acc;
  }, [])]);
  logger.debug(`Fetching outputs, download source and fleet server hosts for agent policy [${agentPolicy.id}]`);
  const [outputs, downloadSource, fleetServerHosts] = await Promise.all([_output.outputService.bulkGet(outputIds, {
    ignoreNotFound: true
  }), (0, _source_uri_utils.getDownloadSourceForAgentPolicy)(agentPolicy), (0, _fleet_server_host.getFleetServerHostsForAgentPolicy)(soClient, agentPolicy).catch(err => {
    logger.warn(`Unable to get fleet server hosts for policy ${agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.id}: ${err.message}`);
    return undefined;
  })]);
  const {
    proxy_id: downloadSourceProxyId
  } = downloadSource;
  const dataOutput = outputs.find(output => output.id === dataOutputId);
  if (!dataOutput) {
    throw new _errors.OutputNotFoundError(`Data output not found ${dataOutputId}`);
  }
  const monitoringOutput = outputs.find(output => output.id === monitoringOutputId);
  if (!monitoringOutput) {
    throw new _errors.OutputNotFoundError(`Monitoring output not found ${monitoringOutputId}`);
  }
  const proxyIds = (0, _lodash.uniq)(outputs.flatMap(output => output.proxy_id).filter(proxyId => typeof proxyId !== 'undefined' && proxyId !== null).concat(fleetServerHosts !== null && fleetServerHosts !== void 0 && fleetServerHosts.proxy_id ? [fleetServerHosts.proxy_id] : []).concat(downloadSourceProxyId ? [downloadSourceProxyId] : []));
  logger.debug(`fetching list of fleet-server proxies`);
  const proxies = proxyIds.length ? await (0, _fleet_proxies.bulkGetFleetProxies)(soClient, proxyIds) : [];
  let downloadSourceProxyUri = null;
  if (downloadSourceProxyId) {
    const downloadSourceProxy = proxies.find(proxy => proxy.id === downloadSourceProxyId);
    if (downloadSourceProxy) {
      downloadSourceProxyUri = downloadSourceProxy.url;
    }
  }
  logger.debug(`Returning related saved objects for policy [${agentPolicy.id}]`);
  return {
    outputs,
    proxies,
    dataOutput,
    monitoringOutput,
    downloadSource,
    downloadSourceProxyUri,
    fleetServerHost: fleetServerHosts
  };
}