"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isLikelyPunctuation = exports.getPosition = exports.findVisibleToken = exports.findTokens = exports.findPunctuationToken = exports.findFirstToken = void 0;
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Finds all tokens in the given range using binary search. Allows to further
 * filter the tokens using a predicate.
 *
 * @param tokens List of ANTLR tokens.
 * @param min Text position to start searching from.
 * @param max Text position to stop searching at.
 * @param predicate Function to test each token.
 */
const findTokens = function* (tokens, min = 0, max = tokens.length ? tokens[tokens.length - 1].stop : 0, predicate = () => true) {
  let index = 0;
  let left = 0;
  let right = tokens.length - 1;

  // Find the first token index.
  while (left <= right) {
    const mid = left + Math.floor((right - left) / 2);
    const token = tokens[mid];
    if (token.start < min) {
      left = mid + 1;
    } else if (token.stop > min) {
      right = mid - 1;
    } else {
      index = mid;
      break;
    }
  }

  // Return all tokens in the range, which satisfy the predicate.
  for (; index < tokens.length; index++) {
    const token = tokens[index];
    if (token.start > max) {
      break;
    }
    if (predicate(token)) {
      yield token;
    }
  }
};

/**
 * Finds the first token in the given range using binary search. Allows to
 * further filter the tokens using a predicate.
 *
 * @param tokens List of ANTLR tokens.
 * @param min Text position to start searching from.
 * @param max Text position to stop searching at.
 * @param predicate Function to test each token.
 * @returns The first token that matches the predicate or `null` if no token is found.
 */
exports.findTokens = findTokens;
const findFirstToken = (tokens, min = 0, max = tokens.length ? tokens[tokens.length - 1].stop : 0, predicate = () => true) => {
  for (const token of findTokens(tokens, min, max, predicate)) {
    return token;
  }
  return null;
};

/**
 * Finds the first visible token in the given token range using binary search.
 *
 * @param tokens List of ANTLR tokens.
 * @param min Text position to start searching from.
 * @param max Text position to stop searching at.
 * @returns The first punctuation token or `null` if no token is found.
 */
exports.findFirstToken = findFirstToken;
const findVisibleToken = (tokens, min = 0, max = tokens.length ? tokens[tokens.length - 1].stop : 0) => {
  return findFirstToken(tokens, min, max, ({
    channel,
    text
  }) => channel === _constants.DEFAULT_CHANNEL && text.length > 0);
};

/**
 * A heuristic set of punctuation characters.
 */
exports.findVisibleToken = findVisibleToken;
const punctuationChars = new Set(['.', ',', ';', ':', '(', ')', '[', ']', '{', '}']);
const isLikelyPunctuation = text => text.length === 1 && punctuationChars.has(text);

/**
 * Finds the first punctuation token in the given token range using binary
 * search.
 *
 * @param tokens List of ANTLR tokens.
 * @param min Text position to start searching from.
 * @param max Text position to stop searching at.
 * @returns The first punctuation token or `null` if no token is found.
 */
exports.isLikelyPunctuation = isLikelyPunctuation;
const findPunctuationToken = (tokens, min = 0, max = tokens.length ? tokens[tokens.length - 1].stop : 0) => {
  return findFirstToken(tokens, min, max, ({
    channel,
    text
  }) => channel === _constants.DEFAULT_CHANNEL && text.length === 1 && punctuationChars.has(text));
};
exports.findPunctuationToken = findPunctuationToken;
const getPosition = (start, stop) => {
  var _ref, _stop$stop;
  if (!start || start.start < 0) {
    return {
      min: 0,
      max: 0
    };
  }
  const endFirstToken = start.stop > -1 ? Math.max(start.stop + 1, start.start) : undefined;
  return {
    min: start.start,
    max: (_ref = (_stop$stop = stop === null || stop === void 0 ? void 0 : stop.stop) !== null && _stop$stop !== void 0 ? _stop$stop : endFirstToken) !== null && _ref !== void 0 ? _ref : Infinity
  };
};
exports.getPosition = getPosition;