"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.autocomplete = autocomplete;
var _helpers = require("../../../definitions/utils/autocomplete/helpers");
var _utils = require("../../../definitions/utils");
var _expressions = require("../../../definitions/utils/expressions");
var _complete_items = require("../../complete_items");
var _types = require("../../types");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function autocomplete(query, command, callbacks, context, cursorPosition = query.length) {
  if (!(callbacks !== null && callbacks !== void 0 && callbacks.getByType)) {
    return [];
  }
  const innerText = query.substring(0, cursorPosition);
  const prependSpace = suggestion => ({
    ...suggestion,
    text: ' ' + suggestion.text
  });
  const commandText = innerText.slice(command.location.min);
  const {
    position: pos,
    context: posContext
  } = (0, _utils2.getSortPos)(commandText, command, cursorPosition);
  switch (pos) {
    case 'empty_expression':
      {
        return await (0, _utils.suggestForExpression)({
          query,
          expressionRoot: undefined,
          command,
          cursorPosition,
          location: _types.Location.SORT,
          context,
          callbacks,
          options: {
            addSpaceAfterFirstField: false,
            addSpaceAfterOperator: true,
            openSuggestions: true
          }
        });
      }
    case 'expression':
      {
        const expressionRoot = command.args[command.args.length - 1];
        if (!expressionRoot || Array.isArray(expressionRoot)) {
          // guaranteed by the getSortPos function, but we check it here for type safety
          return [];
        }
        const suggestions = [];
        const columnExists = name => (0, _helpers.columnExists)(name, context);
        const expressionType = (0, _expressions.getExpressionType)(expressionRoot, context === null || context === void 0 ? void 0 : context.columns);
        const isComplete = (0, _expressions.isExpressionComplete)(expressionType, innerText);
        if (isComplete && !(posContext !== null && posContext !== void 0 && posContext.insideFunction)) {
          suggestions.push(...(0, _utils2.getSuggestionsAfterCompleteExpression)(innerText, expressionRoot, columnExists));
        }
        if (!(0, _utils2.rightAfterColumn)(innerText, expressionRoot, columnExists)) {
          const expressionSuggestions = await (0, _utils.suggestForExpression)({
            query,
            expressionRoot,
            command,
            cursorPosition,
            location: _types.Location.SORT,
            context,
            callbacks,
            options: {
              addSpaceAfterFirstField: false,
              addSpaceAfterOperator: true,
              openSuggestions: true
            }
          });
          suggestions.push(...expressionSuggestions);
        }
        const nullsPrefixRange = (0, _utils2.getNullsPrefixRange)(innerText);
        if (nullsPrefixRange) {
          suggestions.forEach(suggestion => {
            suggestion.rangeToReplace = nullsPrefixRange;
          });
        }
        return suggestions;
      }
    case 'order_complete':
      {
        const {
          fragment,
          rangeToReplace
        } = (0, _helpers.getFragmentData)(innerText);
        return [{
          ..._complete_items.pipeCompleteItem,
          text: ' | '
        }, {
          ..._complete_items.commaCompleteItem,
          text: ', '
        }, prependSpace(_utils2.sortModifierSuggestions.NULLS_FIRST), prependSpace(_utils2.sortModifierSuggestions.NULLS_LAST)].map(suggestion => (0, _helpers.withAutoSuggest)({
          ...suggestion,
          filterText: fragment,
          text: fragment + suggestion.text,
          rangeToReplace
        }));
      }
    case 'after_order':
      {
        const nullsPrefixRange = (0, _utils2.getNullsPrefixRange)(innerText);
        return [_utils2.sortModifierSuggestions.NULLS_FIRST, _utils2.sortModifierSuggestions.NULLS_LAST, _complete_items.pipeCompleteItem, (0, _helpers.withAutoSuggest)({
          ..._complete_items.commaCompleteItem,
          text: ', '
        })].map(suggestion => ({
          ...suggestion,
          rangeToReplace: nullsPrefixRange
        }));
      }
    case 'nulls_complete':
      {
        const {
          fragment,
          rangeToReplace
        } = (0, _helpers.getFragmentData)(innerText);
        return [{
          ..._complete_items.pipeCompleteItem,
          text: ' | '
        }, {
          ..._complete_items.commaCompleteItem,
          text: ', '
        }].map(suggestion => (0, _helpers.withAutoSuggest)({
          ...suggestion,
          filterText: fragment,
          text: fragment + suggestion.text,
          rangeToReplace
        }));
      }
    case 'after_nulls':
      {
        return [_complete_items.pipeCompleteItem, (0, _helpers.withAutoSuggest)({
          ..._complete_items.commaCompleteItem,
          text: ', '
        })];
      }
    default:
      {
        return [];
      }
  }
}