"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TraceDataState = void 0;
exports.createColorLookupMap = createColorLookupMap;
exports.getClockSkew = getClockSkew;
exports.getLegends = getLegends;
exports.getRootItemOrFallback = getRootItemOrFallback;
exports.getTraceParentChildrenMap = getTraceParentChildrenMap;
exports.getTraceWaterfall = getTraceWaterfall;
exports.getTraceWaterfallDuration = getTraceWaterfallDuration;
exports.useTraceWaterfall = useTraceWaterfall;
var _eui = require("@elastic/eui");
var _react = require("react");
var _legend = require("../../../../common/waterfall/legend");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function useTraceWaterfall({
  traceItems
}) {
  const waterfall = (0, _react.useMemo)(() => {
    const legends = getLegends(traceItems);
    const colorBy = legends.filter(({
      type
    }) => type === _legend.WaterfallLegendType.ServiceName).length > 1 ? _legend.WaterfallLegendType.ServiceName : _legend.WaterfallLegendType.Type;
    const colorMap = createColorLookupMap(legends);
    const traceParentChildrenMap = getTraceParentChildrenMap(traceItems);
    const {
      rootItem,
      traceState,
      orphans
    } = getRootItemOrFallback(traceParentChildrenMap, traceItems);
    const traceWaterfall = rootItem ? getTraceWaterfall({
      rootItem,
      parentChildMap: traceParentChildrenMap,
      orphans,
      colorMap,
      colorBy
    }) : [];
    return {
      rootItem,
      traceState,
      traceWaterfall,
      duration: getTraceWaterfallDuration(traceWaterfall),
      maxDepth: Math.max(...traceWaterfall.map(item => item.depth)),
      legends,
      colorBy
    };
  }, [traceItems]);
  return waterfall;
}
function getLegends(traceItems) {
  const serviceNames = Array.from(new Set(traceItems.map(item => item.serviceName)));
  const types = Array.from(new Set(traceItems.map(item => {
    var _item$type;
    return (_item$type = item.type) !== null && _item$type !== void 0 ? _item$type : '';
  })));
  const palette = (0, _eui.euiPaletteColorBlind)({
    rotations: Math.ceil((serviceNames.length + types.length) / 10)
  });
  let colorIndex = 0;
  const legends = [];
  serviceNames.forEach(serviceName => {
    legends.push({
      type: _legend.WaterfallLegendType.ServiceName,
      value: serviceName,
      color: palette[colorIndex++]
    });
  });
  types.forEach(type => {
    legends.push({
      type: _legend.WaterfallLegendType.Type,
      value: type,
      color: palette[colorIndex++]
    });
  });
  return legends;
}
function createColorLookupMap(legends) {
  return new Map(legends.map(legend => [`${legend.type}:${legend.value}`, legend.color]));
}
function getTraceParentChildrenMap(traceItems) {
  const traceMap = traceItems.reduce((acc, item) => {
    if (!item.parentId) {
      acc.root = [item];
    } else {
      if (!acc[item.parentId]) {
        acc[item.parentId] = [];
      }
      acc[item.parentId].push(item);
    }
    return acc;
  }, {});
  return traceMap;
}
let TraceDataState = exports.TraceDataState = /*#__PURE__*/function (TraceDataState) {
  TraceDataState["Full"] = "full";
  TraceDataState["Partial"] = "partial";
  TraceDataState["Empty"] = "empty";
  return TraceDataState;
}({});
function getRootItemOrFallback(traceParentChildrenMap, traceItems) {
  var _traceParentChildrenM;
  if (traceItems.length === 0) {
    return {
      traceState: TraceDataState.Empty
    };
  }
  const rootItem = (_traceParentChildrenM = traceParentChildrenMap.root) === null || _traceParentChildrenM === void 0 ? void 0 : _traceParentChildrenM[0];
  const parentIds = new Set(traceItems.map(({
    id
  }) => id));
  // TODO: Reuse waterfall util methods where possible or if logic is the same
  const orphans = traceItems.filter(item => item.parentId && !parentIds.has(item.parentId));
  if (rootItem) {
    return {
      traceState: orphans.length === 0 ? TraceDataState.Full : TraceDataState.Partial,
      rootItem,
      orphans
    };
  }
  const [fallbackRootItem, ...remainingOrphans] = orphans;
  return {
    traceState: TraceDataState.Partial,
    rootItem: fallbackRootItem,
    orphans: remainingOrphans
  };
}

// TODO: Reuse waterfall util methods where possible or if logic is the same
function reparentOrphansToRoot(rootItem, parentChildMap, orphans) {
  // Some cases with orphans, the root item has no direct link or children, so this
  // might be not initialised. This assigns the array in case of undefined/null to the
  // map.
  const children = parentChildMap[rootItem.id] ??= [];
  children.push(...orphans.map(orphan => ({
    ...orphan,
    parentId: rootItem.id,
    isOrphan: true
  })));
}
function getTraceWaterfall({
  rootItem,
  parentChildMap,
  orphans,
  colorMap,
  colorBy
}) {
  const rootStartMicroseconds = rootItem.timestampUs;
  reparentOrphansToRoot(rootItem, parentChildMap, orphans);
  function getTraceWaterfallItem(item, depth, parent) {
    var _item$type2, _parentChildMap$item$;
    const startMicroseconds = item.timestampUs;
    const color = colorBy === _legend.WaterfallLegendType.ServiceName && item.serviceName ? colorMap.get(`${_legend.WaterfallLegendType.ServiceName}:${item.serviceName}`) : colorMap.get(`${_legend.WaterfallLegendType.Type}:${(_item$type2 = item.type) !== null && _item$type2 !== void 0 ? _item$type2 : ''}`);
    const traceWaterfallItem = {
      ...item,
      depth,
      offset: startMicroseconds - rootStartMicroseconds,
      skew: getClockSkew({
        itemTimestamp: startMicroseconds,
        itemDuration: item.duration,
        parent
      }),
      color
    };
    const sortedChildren = ((_parentChildMap$item$ = parentChildMap[item.id]) === null || _parentChildMap$item$ === void 0 ? void 0 : _parentChildMap$item$.sort((a, b) => a.timestampUs - b.timestampUs)) || [];
    const flattenedChildren = sortedChildren.flatMap(child => getTraceWaterfallItem(child, depth + 1, traceWaterfallItem));
    return [traceWaterfallItem, ...flattenedChildren];
  }
  return getTraceWaterfallItem(rootItem, 0);
}
function getClockSkew({
  itemTimestamp,
  itemDuration,
  parent
}) {
  let skew = 0;
  if (parent) {
    const parentTimestamp = parent.timestampUs;
    const parentStart = parentTimestamp + parent.skew;
    const offsetStart = parentStart - itemTimestamp;
    if (offsetStart > 0) {
      const latency = Math.max(parent.duration - itemDuration, 0) / 2;
      skew = offsetStart + latency;
    }
  }
  return skew;
}
function getTraceWaterfallDuration(flattenedTraceWaterfall) {
  return Math.max(...flattenedTraceWaterfall.map(item => item.offset + item.skew + item.duration), 0);
}