"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateGapsInEventLog = void 0;
var _apmUtils = require("@kbn/apm-utils");
var _mget_gaps = require("../mget_gaps");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CONFLICT_STATUS_CODE = 409;
const MAX_RETRIES = 3;
const delay = ms => new Promise(resolve => setTimeout(resolve, ms));
const updateGapsInEventLog = async ({
  gaps,
  prepareGaps,
  alertingEventLogger,
  logger,
  eventLogClient
}) => {
  let gapsToUpdate = gaps;
  let retryCount = 0;
  let erroredItemsCount = 0;
  try {
    while (retryCount < MAX_RETRIES) {
      if (gapsToUpdate.length === 0) {
        return true;
      }

      // Prepare all gaps for update
      const preparedGaps = await prepareGaps(gapsToUpdate);
      if (preparedGaps.length === 0) {
        return true;
      }

      // Attempt bulk update
      const bulkResponse = await (0, _apmUtils.withSpan)({
        name: 'updateGapsInEventLog.alertingEventLogger.updateGaps',
        type: 'rule'
      }, () => alertingEventLogger.updateGaps(preparedGaps));
      if (!bulkResponse.errors) {
        return true;
      }
      erroredItemsCount = bulkResponse.items.length;
      retryCount++;
      logger.info(`Retrying update of ${erroredItemsCount} gaps due to conflicts. Retry ${retryCount} of ${MAX_RETRIES}`);
      const maxDelaySec = Math.min(Math.pow(3, retryCount), 30);
      const minDelaySec = Math.pow(2, retryCount);
      const delaySec = Math.random() * (maxDelaySec - minDelaySec) + minDelaySec;
      await delay(delaySec * 1000);
      const failedUpdatesDocs = bulkResponse === null || bulkResponse === void 0 ? void 0 : bulkResponse.items.filter(item => {
        var _item$update;
        return ((_item$update = item.update) === null || _item$update === void 0 ? void 0 : _item$update.status) === CONFLICT_STATUS_CODE;
      }).map(item => {
        var _item$update2, _item$update3;
        return {
          _id: (_item$update2 = item.update) === null || _item$update2 === void 0 ? void 0 : _item$update2._id,
          _index: (_item$update3 = item.update) === null || _item$update3 === void 0 ? void 0 : _item$update3._index
        };
      }).filter(doc => doc._id !== undefined && doc._index !== undefined);

      // Fetch latest versions of failed gaps
      gapsToUpdate = await (0, _mget_gaps.mgetGaps)({
        eventLogClient,
        logger,
        params: {
          docs: failedUpdatesDocs
        }
      });
    }
    logger.error(`Failed to update ${erroredItemsCount} gaps after ${MAX_RETRIES} retries due to conflicts`);
    return false;
  } catch (e) {
    logger.error(`Failed to update gaps in event log: ${e.message}`);
    return false;
  }
};
exports.updateGapsInEventLog = updateGapsInEventLog;