/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
package org.elasticsearch.xpack.esql.plan.logical.local;

import org.elasticsearch.TransportVersion;
import org.elasticsearch.common.io.stream.NamedWriteableRegistry;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.xpack.esql.core.expression.Attribute;
import org.elasticsearch.xpack.esql.core.tree.NodeInfo;
import org.elasticsearch.xpack.esql.core.tree.Source;
import org.elasticsearch.xpack.esql.io.stream.PlanStreamInput;
import org.elasticsearch.xpack.esql.plan.logical.LeafPlan;
import org.elasticsearch.xpack.esql.plan.logical.LogicalPlan;
import org.elasticsearch.xpack.esql.plan.physical.LocalSourceExec;

import java.io.IOException;
import java.util.List;
import java.util.Objects;

public class LocalRelation extends LeafPlan {
    public static final NamedWriteableRegistry.Entry ENTRY = new NamedWriteableRegistry.Entry(
        LogicalPlan.class,
        "LocalRelation",
        LocalRelation::new
    );

    private static final TransportVersion ESQL_LOCAL_RELATION_WITH_NEW_BLOCKS = TransportVersion.fromName(
        "esql_local_relation_with_new_blocks"
    );

    private final List<Attribute> output;
    private final LocalSupplier supplier;

    public LocalRelation(Source source, List<Attribute> output, LocalSupplier supplier) {
        super(source);
        this.output = output;
        this.supplier = supplier;
    }

    public LocalRelation(StreamInput in) throws IOException {
        super(Source.readFrom((PlanStreamInput) in));
        this.output = in.readNamedWriteableCollectionAsList(Attribute.class);
        if (in.getTransportVersion().supports(ESQL_LOCAL_RELATION_WITH_NEW_BLOCKS)) {
            this.supplier = in.readNamedWriteable(LocalSupplier.class);
        } else {
            this.supplier = LocalSourceExec.readLegacyLocalSupplierFrom((PlanStreamInput) in);
        }
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        source().writeTo(out);
        out.writeNamedWriteableCollection(output);
        if (out.getTransportVersion().supports(ESQL_LOCAL_RELATION_WITH_NEW_BLOCKS)) {
            out.writeNamedWriteable(supplier);
        } else {
            if (hasEmptySupplier()) {
                out.writeVInt(0);
            } else {// here we can only have an ImmediateLocalSupplier as this was the only implementation apart from EMPTY
                ((ImmediateLocalSupplier) supplier).writeTo(out);
            }
        }
    }

    @Override
    public String getWriteableName() {
        return ENTRY.name;
    }

    @Override
    protected NodeInfo<LocalRelation> info() {
        return NodeInfo.create(this, LocalRelation::new, output, supplier);
    }

    public LocalSupplier supplier() {
        return supplier;
    }

    public boolean hasEmptySupplier() {
        return supplier == EmptyLocalSupplier.EMPTY;
    }

    @Override
    public boolean expressionsResolved() {
        return true;
    }

    @Override
    public List<Attribute> output() {
        return output;
    }

    @Override
    public int hashCode() {
        return Objects.hash(output, supplier);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }

        if (obj == null || getClass() != obj.getClass()) {
            return false;
        }

        LocalRelation other = (LocalRelation) obj;
        return Objects.equals(supplier, other.supplier) && Objects.equals(output, other.output);
    }

}
