/*
 * @notice
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Modifications copyright (C) 2025 Elasticsearch B.V.
 */
package org.elasticsearch.index.codec.vectors.es93;

import org.apache.lucene.codecs.KnnVectorsReader;
import org.apache.lucene.codecs.KnnVectorsWriter;
import org.apache.lucene.codecs.hnsw.FlatVectorsFormat;
import org.apache.lucene.codecs.lucene99.Lucene99HnswVectorsReader;
import org.apache.lucene.codecs.lucene99.Lucene99HnswVectorsWriter;
import org.apache.lucene.index.SegmentReadState;
import org.apache.lucene.index.SegmentWriteState;
import org.elasticsearch.index.codec.vectors.AbstractHnswVectorsFormat;

import java.io.IOException;
import java.util.concurrent.ExecutorService;

public class ES93HnswBinaryQuantizedVectorsFormat extends AbstractHnswVectorsFormat {

    public static final String NAME = "ES93HnswBinaryQuantizedVectorsFormat";

    /** The format for storing, reading, merging vectors on disk */
    private final FlatVectorsFormat flatVectorsFormat;

    /** Constructs a format using default graph construction parameters */
    public ES93HnswBinaryQuantizedVectorsFormat() {
        super(NAME);
        flatVectorsFormat = new ES93BinaryQuantizedVectorsFormat();
    }

    /**
     * Constructs a format using the given graph construction parameters.
     *
     * @param useDirectIO whether to use direct IO when reading raw vectors
     */
    public ES93HnswBinaryQuantizedVectorsFormat(ES93GenericFlatVectorsFormat.ElementType elementType, boolean useDirectIO) {
        super(NAME);
        flatVectorsFormat = new ES93BinaryQuantizedVectorsFormat(elementType, useDirectIO);
    }

    /**
     * Constructs a format using the given graph construction parameters.
     *
     * @param maxConn the maximum number of connections to a node in the HNSW graph
     * @param beamWidth the size of the queue maintained during graph construction.
     * @param useDirectIO whether to use direct IO when reading raw vectors
     */
    public ES93HnswBinaryQuantizedVectorsFormat(
        int maxConn,
        int beamWidth,
        ES93GenericFlatVectorsFormat.ElementType elementType,
        boolean useDirectIO
    ) {
        super(NAME, maxConn, beamWidth);
        flatVectorsFormat = new ES93BinaryQuantizedVectorsFormat(elementType, useDirectIO);
    }

    /**
     * Constructs a format using the given graph construction parameters and scalar quantization.
     *
     * @param maxConn the maximum number of connections to a node in the HNSW graph
     * @param beamWidth the size of the queue maintained during graph construction.
     * @param useDirectIO whether to use direct IO when reading raw vectors
     * @param numMergeWorkers number of workers (threads) that will be used when doing merge. If
     *     larger than 1, a non-null {@link ExecutorService} must be passed as mergeExec
     * @param mergeExec the {@link ExecutorService} that will be used by ALL vector writers that are
     *     generated by this format to do the merge
     */
    public ES93HnswBinaryQuantizedVectorsFormat(
        int maxConn,
        int beamWidth,
        ES93GenericFlatVectorsFormat.ElementType elementType,
        boolean useDirectIO,
        int numMergeWorkers,
        ExecutorService mergeExec
    ) {
        super(NAME, maxConn, beamWidth, numMergeWorkers, mergeExec);
        flatVectorsFormat = new ES93BinaryQuantizedVectorsFormat(elementType, useDirectIO);
    }

    @Override
    protected FlatVectorsFormat flatVectorsFormat() {
        return flatVectorsFormat;
    }

    @Override
    public KnnVectorsWriter fieldsWriter(SegmentWriteState state) throws IOException {
        return new Lucene99HnswVectorsWriter(state, maxConn, beamWidth, flatVectorsFormat.fieldsWriter(state), numMergeWorkers, mergeExec);
    }

    @Override
    public KnnVectorsReader fieldsReader(SegmentReadState state) throws IOException {
        return new Lucene99HnswVectorsReader(state, flatVectorsFormat.fieldsReader(state));
    }
}
