/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
package org.elasticsearch.health.stats;

import org.elasticsearch.action.FailedNodeException;
import org.elasticsearch.action.support.ActionFilters;
import org.elasticsearch.action.support.nodes.TransportNodesAction;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.injection.guice.Inject;
import org.elasticsearch.tasks.Task;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.TransportService;

import java.io.IOException;
import java.util.List;

/**
 * Performs the health api stats operation.
 */
public class HealthApiStatsTransportAction extends TransportNodesAction<
    HealthApiStatsAction.Request,
    HealthApiStatsAction.Response,
    HealthApiStatsAction.Request.Node,
    HealthApiStatsAction.Response.Node,
    Void> {

    private final HealthApiStats healthApiStats;

    @Inject
    public HealthApiStatsTransportAction(
        TransportService transportService,
        ClusterService clusterService,
        ThreadPool threadPool,
        ActionFilters actionFilters,
        HealthApiStats healthApiStats
    ) {
        super(
            HealthApiStatsAction.NAME,
            clusterService,
            transportService,
            actionFilters,
            HealthApiStatsAction.Request.Node::new,
            threadPool.executor(ThreadPool.Names.MANAGEMENT)
        );
        this.healthApiStats = healthApiStats;
    }

    @Override
    protected HealthApiStatsAction.Response newResponse(
        HealthApiStatsAction.Request request,
        List<HealthApiStatsAction.Response.Node> nodes,
        List<FailedNodeException> failures
    ) {
        return new HealthApiStatsAction.Response(clusterService.getClusterName(), nodes, failures);
    }

    @Override
    protected HealthApiStatsAction.Request.Node newNodeRequest(HealthApiStatsAction.Request request) {
        return new HealthApiStatsAction.Request.Node(request);
    }

    @Override
    protected HealthApiStatsAction.Response.Node newNodeResponse(StreamInput in, DiscoveryNode node) throws IOException {
        return new HealthApiStatsAction.Response.Node(in);
    }

    @Override
    protected HealthApiStatsAction.Response.Node nodeOperation(HealthApiStatsAction.Request.Node request, Task task) {
        HealthApiStatsAction.Response.Node statsResponse = new HealthApiStatsAction.Response.Node(clusterService.localNode());
        if (healthApiStats.hasCounters()) {
            statsResponse.setStats(healthApiStats.getStats());
        }
        return statsResponse;
    }
}
