/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.transport;

import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.security.GeneralSecurityException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import javax.net.ssl.X509ExtendedKeyManager;
import javax.net.ssl.X509ExtendedTrustManager;
import javax.net.ssl.X509KeyManager;
import javax.net.ssl.X509TrustManager;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchSecurityException;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.ssl.DiagnosticTrustManager;
import org.elasticsearch.common.ssl.SslConfiguration;
import org.elasticsearch.common.ssl.SslKeyConfig;
import org.elasticsearch.common.ssl.SslTrustConfig;
import org.elasticsearch.common.ssl.SslUtil;
import org.elasticsearch.env.Environment;
import org.elasticsearch.xpack.core.ssl.SslSettingsLoader;
import org.elasticsearch.xpack.security.transport.X509CertificateSignature;

public class CrossClusterApiKeySignatureManager {
    private final org.elasticsearch.logging.Logger logger = org.elasticsearch.logging.LogManager.getLogger(this.getClass());
    private final Environment environment;
    private final AtomicReference<X509ExtendedTrustManager> trustManager = new AtomicReference();
    private final Map<String, X509KeyPair> keyPairByClusterAlias = new ConcurrentHashMap<String, X509KeyPair>();
    private final Map<String, SslConfiguration> sslSigningConfigByClusterAlias = new ConcurrentHashMap<String, SslConfiguration>();
    private final AtomicReference<SslConfiguration> sslTrustConfig = new AtomicReference();
    private static final Map<String, String> SIGNATURE_ALGORITHM_BY_TYPE = Map.of("RSA", "SHA256withRSA", "EC", "SHA256withECDSA");

    public CrossClusterApiKeySignatureManager(Environment environment) {
        this.environment = environment;
        this.loadSigningConfigs();
        this.loadTrustConfig();
    }

    public void reload(Settings settings) {
        this.logger.trace("Loading trust config with settings [{}]", new Object[]{settings});
        try {
            SslConfiguration sslConfig = CrossClusterApiKeySignatureManager.loadSslConfig(this.environment, settings);
            SslTrustConfig trustConfig = sslConfig.trustConfig();
            if (trustConfig.hasExplicitConfig() || trustConfig.isSystemDefault()) {
                X509ExtendedTrustManager newTrustManager;
                X509ExtendedTrustManager x509ExtendedTrustManager = newTrustManager = settings.getAsBoolean("signing.diagnose.trust", Boolean.valueOf(true)) != false ? this.wrapInDiagnosticTrustManager(trustConfig.createTrustManager()) : trustConfig.createTrustManager();
                if (newTrustManager.getAcceptedIssuers().length == 0) {
                    this.logger.warn("Cross cluster API Key trust configuration [{}] has no accepted certificate issuers", new Object[]{trustConfig});
                    this.trustManager.set(null);
                } else {
                    this.sslTrustConfig.set(sslConfig);
                    this.trustManager.set(newTrustManager);
                }
            } else {
                this.trustManager.set(null);
            }
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to load trust config", e);
        }
    }

    public void reload(String clusterAlias, Settings settings) {
        this.logger.trace("Loading signing config for [{}] with settings [{}]", new Object[]{clusterAlias, settings});
        if (!settings.getByPrefix("signing").isEmpty()) {
            try {
                SslConfiguration sslConfig = CrossClusterApiKeySignatureManager.loadSslConfig(this.environment, settings);
                this.sslSigningConfigByClusterAlias.put(clusterAlias, sslConfig);
                SslKeyConfig keyConfig = sslConfig.keyConfig();
                if (keyConfig.hasKeyMaterial()) {
                    String alias = settings.get("signing.keystore.alias");
                    X509ExtendedKeyManager keyManager = keyConfig.createKeyManager();
                    if (keyManager == null) {
                        throw new IllegalStateException("Cannot create key manager for key config [" + String.valueOf(keyConfig) + "]");
                    }
                    X509KeyPair keyPair = Strings.isNullOrEmpty((String)alias) ? this.buildKeyPair(keyManager, keyConfig) : this.buildKeyPair(keyManager, keyConfig, alias);
                    this.logger.trace("Key pair [{}] found for [{}]", new Object[]{keyPair, clusterAlias});
                    this.keyPairByClusterAlias.put(clusterAlias, keyPair);
                }
                this.keyPairByClusterAlias.remove(clusterAlias);
            }
            catch (Exception e) {
                throw new IllegalStateException(Strings.format((String)"Failed to load signing config for cluster [%s]", (Object[])new Object[]{clusterAlias}), e);
            }
        } else {
            this.logger.trace("No valid signing config settings found for [{}] with settings [{}]", new Object[]{clusterAlias, settings});
            this.keyPairByClusterAlias.remove(clusterAlias);
        }
    }

    public Collection<Path> getDependentTrustFiles() {
        SslConfiguration sslConfig = this.sslTrustConfig.get();
        return sslConfig == null ? Collections.emptyList() : sslConfig.getDependentFiles();
    }

    public Collection<Path> getDependentSigningFiles(String clusterAlias) {
        SslConfiguration sslConfig = this.sslSigningConfigByClusterAlias.get(clusterAlias);
        return sslConfig == null ? Collections.emptyList() : sslConfig.getDependentFiles();
    }

    public void validate(Settings settings) {
        SslConfiguration sslConfig;
        if (!settings.getByPrefix("signing").isEmpty() && (sslConfig = CrossClusterApiKeySignatureManager.loadSslConfig(this.environment, settings)) != null) {
            sslConfig.getDependentFiles().forEach(path -> {
                if (!Files.exists(path, new LinkOption[0])) {
                    throw new IllegalArgumentException(Strings.format((String)"Configured file [%s] not found", (Object[])new Object[]{path}));
                }
            });
        }
    }

    X509TrustManager getTrustManager() {
        return this.trustManager.get();
    }

    public Verifier verifier() {
        return new Verifier();
    }

    public Signer signerForClusterAlias(String clusterAlias) {
        return this.keyPairByClusterAlias.containsKey(clusterAlias) ? new Signer(clusterAlias) : null;
    }

    private X509ExtendedTrustManager wrapInDiagnosticTrustManager(X509ExtendedTrustManager trustManager) {
        if (!(trustManager instanceof DiagnosticTrustManager)) {
            Logger diagnosticLogger = LogManager.getLogger(DiagnosticTrustManager.class);
            return new DiagnosticTrustManager(trustManager, () -> "cluster.remote.signing", (arg_0, arg_1) -> ((Logger)diagnosticLogger).warn(arg_0, arg_1));
        }
        return trustManager;
    }

    private static String calculateFingerprint(X509Certificate certificate) {
        try {
            return SslUtil.calculateFingerprint((X509Certificate)certificate, (String)"SHA-1");
        }
        catch (CertificateEncodingException e) {
            return "<?>";
        }
    }

    private static SslConfiguration loadSslConfig(Environment environment, Settings settings) {
        return SslSettingsLoader.load((Settings)settings, (String)"signing.", (Environment)environment);
    }

    private static byte[] getSignableBytes(String ... headers) {
        return String.join((CharSequence)"\n", headers).getBytes(StandardCharsets.UTF_8);
    }

    private void loadSigningConfigs() {
        this.environment.settings().getGroups("cluster.remote.", true).forEach(this::reload);
    }

    private void loadTrustConfig() {
        this.reload(this.environment.settings().getByPrefix("cluster.remote."));
    }

    /*
     * Enabled aggressive block sorting
     */
    private X509KeyPair buildKeyPair(X509KeyManager keyManager, SslKeyConfig keyConfig) {
        Set aliases = SIGNATURE_ALGORITHM_BY_TYPE.keySet().stream().map(keyType -> keyManager.getServerAliases((String)keyType, null)).filter(Objects::nonNull).flatMap(Arrays::stream).collect(Collectors.toSet());
        this.logger.trace("KeyConfig [{}] has compatible entries: [{}]", new Object[]{keyConfig, aliases});
        switch (aliases.size()) {
            case 0: {
                throw new IllegalStateException("Cannot find a signing key in [" + String.valueOf(keyConfig) + "]");
            }
            case 1: {
                String aliasFromKeyStore = (String)aliases.iterator().next();
                X509Certificate[] chain = keyManager.getCertificateChain(aliasFromKeyStore);
                return new X509KeyPair(chain, keyManager.getPrivateKey(aliasFromKeyStore));
            }
        }
        throw new IllegalStateException("The configured signing key store has multiple signing keys [" + String.valueOf(aliases) + "] but no alias has been specified in signing configuration.");
    }

    private X509KeyPair buildKeyPair(X509KeyManager keyManager, SslKeyConfig keyConfig, String alias) {
        assert (alias != null);
        String keyType = keyManager.getPrivateKey(alias).getAlgorithm();
        if (!SIGNATURE_ALGORITHM_BY_TYPE.containsKey(keyType)) {
            throw new IllegalStateException(Strings.format((String)"The key associated with alias [%s] uses unsupported key algorithm type [%s], only %s is supported", (Object[])new Object[]{alias, keyType, SIGNATURE_ALGORITHM_BY_TYPE.keySet()}));
        }
        X509Certificate[] chain = keyManager.getCertificateChain(alias);
        this.logger.trace("KeyConfig [{}] has entry for alias: [{}] [{}]", new Object[]{keyConfig, alias, chain != null});
        if (chain == null) {
            throw new IllegalStateException("Key config missing certificate chain for alias [" + alias + "]");
        }
        return new X509KeyPair(chain, keyManager.getPrivateKey(alias));
    }

    public record X509KeyPair(X509Certificate[] certificates, PrivateKey privateKey, String signatureAlgorithm, String fingerprint) {
        X509KeyPair(X509Certificate[] certificates, PrivateKey privateKey) {
            this(Objects.requireNonNull(certificates), Objects.requireNonNull(privateKey), Optional.ofNullable(SIGNATURE_ALGORITHM_BY_TYPE.get(privateKey.getAlgorithm())).orElseThrow(() -> new IllegalArgumentException("Unsupported Key Type [" + privateKey.getAlgorithm() + "] in private key for [" + String.valueOf(certificates[0].getSubjectX500Principal()) + "]")), CrossClusterApiKeySignatureManager.calculateFingerprint(certificates[0]));
        }
    }

    public class Verifier {
        private Verifier() {
        }

        public boolean verify(X509CertificateSignature signature, String ... headers) throws GeneralSecurityException {
            assert (signature.certificates().length > 0) : "Signature not valid without trusted certificate chain";
            X509ExtendedTrustManager authTrustManager = CrossClusterApiKeySignatureManager.this.trustManager.get();
            if (authTrustManager == null) {
                CrossClusterApiKeySignatureManager.this.logger.warn("Cannot verify signed cross-cluster headers because [cluster.remote.signing] has not trust configuration");
                throw new IllegalStateException("Cannot verify signed cross-cluster headers because [cluster.remote.signing] has not trust configuration");
            }
            X509Certificate leaf = signature.leafCertificate();
            if (CrossClusterApiKeySignatureManager.this.logger.isTraceEnabled()) {
                CrossClusterApiKeySignatureManager.this.logger.trace("checking signing chain (len={}) [{}] with leaf subject [{}] using algorithm [{}]", new Object[]{signature.certificates().length, Arrays.stream(signature.certificates()).map(CrossClusterApiKeySignatureManager::calculateFingerprint).collect(Collectors.joining(",")), leaf.getSubjectX500Principal().getName("RFC2253"), leaf.getPublicKey().getAlgorithm()});
            }
            authTrustManager.checkClientTrusted(signature.certificates(), leaf.getPublicKey().getAlgorithm());
            signature.leafCertificate().checkValidity();
            Signature signer = Signature.getInstance(signature.algorithm());
            signer.initVerify(leaf);
            signer.update(CrossClusterApiKeySignatureManager.getSignableBytes(headers));
            return signer.verify(signature.signature().array());
        }
    }

    public class Signer {
        private final String clusterAlias;

        private Signer(String clusterAlias) {
            this.clusterAlias = clusterAlias;
        }

        public X509CertificateSignature sign(String ... headers) {
            X509KeyPair keyPair = CrossClusterApiKeySignatureManager.this.keyPairByClusterAlias.get(this.clusterAlias);
            if (keyPair == null) {
                CrossClusterApiKeySignatureManager.this.logger.trace("No signing config found for [{}] returning null signature", new Object[]{this.clusterAlias});
                return null;
            }
            try {
                String algorithm = keyPair.signatureAlgorithm();
                Signature signature = Signature.getInstance(algorithm);
                signature.initSign(keyPair.privateKey);
                signature.update(CrossClusterApiKeySignatureManager.getSignableBytes(headers));
                byte[] sigBytes = signature.sign();
                return new X509CertificateSignature(keyPair.certificates, algorithm, (BytesReference)new BytesArray(sigBytes));
            }
            catch (GeneralSecurityException e) {
                throw new ElasticsearchSecurityException(Strings.format((String)"Failed to sign cross cluster headers for cluster [%s]", (Object[])new Object[]{this.clusterAlias}), (Exception)e, new Object[0]);
            }
        }
    }
}

