/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index.store;

import com.carrotsearch.hppc.IntArrayDeque;
import com.carrotsearch.hppc.IntDeque;
import com.carrotsearch.hppc.LongArrayDeque;
import com.carrotsearch.hppc.LongDeque;
import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.lucene.store.IndexInput;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.logging.LogManager;
import org.elasticsearch.logging.Logger;

public class AsyncDirectIOIndexInput
extends IndexInput {
    private static final Logger LOGGER;
    static final OpenOption ExtendedOpenOption_DIRECT;
    private final DirectIOPrefetcher prefetcher;
    private final ByteBuffer buffer;
    private final FileChannel channel;
    private final int blockSize;
    private final long offset;
    private final long length;
    private final boolean isClosable;
    private boolean isOpen;
    private long filePos;

    static OpenOption getDirectOpenOption() {
        if (ExtendedOpenOption_DIRECT == null) {
            throw new UnsupportedOperationException("com.sun.nio.file.ExtendedOpenOption.DIRECT is not available in the current JDK version.");
        }
        return ExtendedOpenOption_DIRECT;
    }

    @SuppressForbidden(reason="requires FileChannel#read")
    private static void readDirectChannel(FileChannel c, ByteBuffer bb, long p) throws IOException {
        c.read(bb, p);
    }

    public AsyncDirectIOIndexInput(Path path, int blockSize, int bufferSize, int maxPrefetches) throws IOException {
        super("DirectIOIndexInput(path=\"" + String.valueOf(path) + "\")");
        this.channel = FileChannel.open(path, StandardOpenOption.READ, AsyncDirectIOIndexInput.getDirectOpenOption());
        this.blockSize = blockSize;
        this.prefetcher = new DirectIOPrefetcher(blockSize, this.channel, bufferSize, maxPrefetches, maxPrefetches * 16);
        this.buffer = AsyncDirectIOIndexInput.allocateBuffer(bufferSize, blockSize);
        this.isOpen = true;
        this.isClosable = true;
        this.length = this.channel.size();
        this.offset = 0L;
        this.filePos = -bufferSize;
        this.buffer.limit(0);
    }

    private AsyncDirectIOIndexInput(String description, AsyncDirectIOIndexInput other, long offset, long length) throws IOException {
        super(description);
        Objects.checkFromIndexSize(offset, length, other.channel.size());
        int bufferSize = other.buffer.capacity();
        this.buffer = AsyncDirectIOIndexInput.allocateBuffer(bufferSize, other.blockSize);
        this.blockSize = other.blockSize;
        this.channel = other.channel;
        this.prefetcher = new DirectIOPrefetcher(this.blockSize, this.channel, bufferSize, other.prefetcher.maxConcurrentPrefetches, other.prefetcher.maxTotalPrefetches);
        this.isOpen = true;
        this.isClosable = false;
        this.length = length;
        this.offset = offset;
        this.filePos = -bufferSize;
        this.buffer.limit(0);
    }

    private static ByteBuffer allocateBuffer(int bufferSize, int blockSize) {
        return ByteBuffer.allocateDirect(bufferSize + blockSize - 1).alignedSlice(blockSize).order(ByteOrder.LITTLE_ENDIAN);
    }

    public void prefetch(long pos, long length) throws IOException {
        if (pos < 0L || length < 0L || pos + length > this.length) {
            throw new IllegalArgumentException("Invalid prefetch range: pos=" + pos + ", length=" + length + ", fileLength=" + this.length);
        }
        long absPos = pos + this.offset;
        long alignedPos = absPos - absPos % (long)this.blockSize;
        this.prefetcher.prefetch(alignedPos, length);
    }

    public void close() throws IOException {
        this.prefetcher.close();
        if (this.isOpen && this.isClosable) {
            this.channel.close();
            this.isOpen = false;
        }
    }

    public long getFilePointer() {
        long filePointer = this.filePos + (long)this.buffer.position() - this.offset;
        assert (filePointer == (long)(-this.buffer.capacity()) - this.offset || filePointer >= 0L) : "filePointer should either be initial value equal to negative buffer capacity, or larger than or equal to 0";
        return Math.max(filePointer, 0L);
    }

    public void seek(long pos) throws IOException {
        if (pos != this.getFilePointer()) {
            long absolutePos = pos + this.offset;
            if (absolutePos >= this.filePos && absolutePos < this.filePos + (long)this.buffer.limit()) {
                this.buffer.position(Math.toIntExact(absolutePos - this.filePos));
            } else {
                this.seekInternal(pos);
            }
        }
        assert (pos == this.getFilePointer());
    }

    private void seekInternal(long pos) throws IOException {
        long absPos = pos + this.offset;
        long alignedPos = absPos - absPos % (long)this.blockSize;
        this.filePos = alignedPos - (long)this.buffer.capacity();
        int delta = (int)(absPos - alignedPos);
        this.refill(delta, delta);
    }

    private void refill(int bytesToRead) throws IOException {
        assert (this.filePos % (long)this.blockSize == 0L);
        this.refill(bytesToRead, 0);
    }

    private void refill(int bytesToRead, int delta) throws IOException {
        long nextFilePos = this.filePos + (long)this.buffer.capacity();
        if (nextFilePos > this.offset + this.length || this.offset + this.length - nextFilePos < (long)bytesToRead) {
            this.filePos = nextFilePos;
            throw new EOFException("read past EOF: " + String.valueOf((Object)this));
        }
        this.buffer.clear();
        try {
            if (this.prefetcher.readBytes(nextFilePos, this.buffer, delta)) {
                long currentLogicalPos = nextFilePos + (long)delta;
                this.filePos = currentLogicalPos - (long)this.buffer.position();
                return;
            }
            this.filePos = nextFilePos;
            assert (this.filePos % (long)this.blockSize == 0L) : "filePos [" + this.filePos + "] must be aligned to block size [" + this.blockSize + "]";
            AsyncDirectIOIndexInput.readDirectChannel(this.channel, this.buffer, this.filePos);
            this.buffer.flip();
            this.buffer.position(delta);
        }
        catch (IOException ioe) {
            throw new IOException(ioe.getMessage() + ": " + String.valueOf((Object)this), ioe);
        }
    }

    public long length() {
        return this.length;
    }

    public byte readByte() throws IOException {
        if (!this.buffer.hasRemaining()) {
            this.refill(1);
        }
        return this.buffer.get();
    }

    public short readShort() throws IOException {
        if (this.buffer.remaining() >= 2) {
            return this.buffer.getShort();
        }
        return super.readShort();
    }

    public int readInt() throws IOException {
        if (this.buffer.remaining() >= 4) {
            return this.buffer.getInt();
        }
        return super.readInt();
    }

    public long readLong() throws IOException {
        if (this.buffer.remaining() >= 8) {
            return this.buffer.getLong();
        }
        return super.readLong();
    }

    public void readBytes(byte[] dst, int offset, int len) throws IOException {
        int left;
        int toRead = len;
        while ((left = this.buffer.remaining()) < toRead) {
            this.buffer.get(dst, offset, left);
            offset += left;
            this.refill(toRead -= left);
        }
        this.buffer.get(dst, offset, toRead);
    }

    public void readInts(int[] dst, int offset, int len) throws IOException {
        int remainingDst = len;
        while (remainingDst > 0) {
            int cnt = Math.min(this.buffer.remaining() / 4, remainingDst);
            this.buffer.asIntBuffer().get(dst, offset + len - remainingDst, cnt);
            this.buffer.position(this.buffer.position() + 4 * cnt);
            if ((remainingDst -= cnt) <= 0) continue;
            if (this.buffer.hasRemaining()) {
                dst[offset + len - remainingDst] = this.readInt();
                --remainingDst;
                continue;
            }
            this.refill(remainingDst * 4);
        }
    }

    public void readFloats(float[] dst, int offset, int len) throws IOException {
        int remainingDst = len;
        while (remainingDst > 0) {
            int cnt = Math.min(this.buffer.remaining() / 4, remainingDst);
            this.buffer.asFloatBuffer().get(dst, offset + len - remainingDst, cnt);
            this.buffer.position(this.buffer.position() + 4 * cnt);
            if ((remainingDst -= cnt) <= 0) continue;
            if (this.buffer.hasRemaining()) {
                dst[offset + len - remainingDst] = Float.intBitsToFloat(this.readInt());
                --remainingDst;
                continue;
            }
            this.refill(remainingDst * 4);
        }
    }

    public void readLongs(long[] dst, int offset, int len) throws IOException {
        int remainingDst = len;
        while (remainingDst > 0) {
            int cnt = Math.min(this.buffer.remaining() / 8, remainingDst);
            this.buffer.asLongBuffer().get(dst, offset + len - remainingDst, cnt);
            this.buffer.position(this.buffer.position() + 8 * cnt);
            if ((remainingDst -= cnt) <= 0) continue;
            if (this.buffer.hasRemaining()) {
                dst[offset + len - remainingDst] = this.readLong();
                --remainingDst;
                continue;
            }
            this.refill(remainingDst * 8);
        }
    }

    public AsyncDirectIOIndexInput clone() {
        try {
            AsyncDirectIOIndexInput clone = new AsyncDirectIOIndexInput("clone:" + String.valueOf((Object)this), this, this.offset, this.length);
            clone.seekInternal(this.getFilePointer());
            return clone;
        }
        catch (IOException ioe) {
            throw new UncheckedIOException(ioe);
        }
    }

    public IndexInput slice(String sliceDescription, long offset, long length) throws IOException {
        if ((length | offset) < 0L || length > this.length - offset) {
            throw new IllegalArgumentException("slice() " + sliceDescription + " out of bounds: " + String.valueOf((Object)this));
        }
        AsyncDirectIOIndexInput slice = new AsyncDirectIOIndexInput(sliceDescription, this, this.offset + offset, length);
        slice.seekInternal(0L);
        return slice;
    }

    static {
        OpenOption option;
        LOGGER = LogManager.getLogger(AsyncDirectIOIndexInput.class);
        try {
            Class<OpenOption> clazz = Class.forName("com.sun.nio.file.ExtendedOpenOption").asSubclass(OpenOption.class);
            option = Arrays.stream(clazz.getEnumConstants()).filter(e -> e.toString().equalsIgnoreCase("DIRECT")).findFirst().orElse(null);
        }
        catch (Exception ex) {
            option = null;
        }
        ExtendedOpenOption_DIRECT = option;
    }

    private static class DirectIOPrefetcher
    implements Closeable {
        private final int maxConcurrentPrefetches;
        private final int maxTotalPrefetches;
        private final FileChannel channel;
        private final int blockSize;
        private final long[] prefetchPos;
        private final List<Future<ByteBuffer>> prefetchThreads;
        private final TreeMap<Long, Integer> posToSlot;
        private final IntDeque slots;
        private final ByteBuffer[] prefetchBuffers;
        private final int prefetchBytesSize;
        private final LongDeque pendingPrefetches = new LongArrayDeque();
        private final ExecutorService executor = Executors.newVirtualThreadPerTaskExecutor();

        DirectIOPrefetcher(int blockSize, FileChannel channel, int prefetchBytesSize, int maxConcurrentPrefetches, int maxTotalPrefetches) {
            this.blockSize = blockSize;
            this.channel = channel;
            this.maxConcurrentPrefetches = maxConcurrentPrefetches;
            this.prefetchPos = new long[maxConcurrentPrefetches];
            this.prefetchThreads = new ArrayList<Future<ByteBuffer>>(maxConcurrentPrefetches);
            this.slots = new IntArrayDeque(maxConcurrentPrefetches);
            for (int i = 0; i < maxConcurrentPrefetches; ++i) {
                this.prefetchThreads.add(null);
                this.slots.addLast(i);
            }
            this.posToSlot = new TreeMap();
            this.prefetchBuffers = new ByteBuffer[maxConcurrentPrefetches];
            this.prefetchBytesSize = prefetchBytesSize;
            this.maxTotalPrefetches = maxTotalPrefetches;
        }

        void prefetch(long pos, long length) {
            for (int numSlots = (int)Math.min((length + (long)this.prefetchBytesSize - 1L) / (long)this.prefetchBytesSize, (long)(this.maxTotalPrefetches - (this.posToSlot.size() + this.pendingPrefetches.size()))); numSlots > 0 && this.posToSlot.size() + this.pendingPrefetches.size() < this.maxTotalPrefetches; --numSlots) {
                int slot;
                Integer existingSlot = this.posToSlot.get(pos);
                if (existingSlot != null && this.prefetchThreads.get(existingSlot) != null) {
                    return;
                }
                if (this.posToSlot.size() < this.maxConcurrentPrefetches && !this.slots.isEmpty()) {
                    slot = this.slots.removeFirst();
                    this.posToSlot.put(pos, slot);
                    this.prefetchPos[slot] = pos;
                } else {
                    slot = -1;
                    LOGGER.debug("queueing prefetch of pos [{}] with length [{}], waiting for open slot", new Object[]{pos, length});
                    this.pendingPrefetches.addLast(pos);
                }
                if (slot != -1) {
                    this.startPrefetch(pos, slot);
                }
                pos += (long)this.prefetchBytesSize;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        boolean readBytes(long pos, ByteBuffer slice, int delta) throws IOException {
            Map.Entry<Long, Integer> entry = this.posToSlot.floorEntry(pos + (long)delta);
            if (entry == null) {
                return false;
            }
            int slot = entry.getValue();
            long prefetchedPos = entry.getKey();
            if (pos + (long)delta >= prefetchedPos + (long)this.prefetchBytesSize) {
                return false;
            }
            Future<ByteBuffer> thread = this.prefetchThreads.get(slot);
            ByteBuffer prefetchBuffer = null;
            try {
                prefetchBuffer = thread == null ? null : thread.get();
            }
            catch (ExecutionException e) {
                IOException ioException = (IOException)ExceptionsHelper.unwrap(e, IOException.class);
                if (ioException != null) {
                    throw ioException;
                }
                throw new IOException(e.getCause());
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            finally {
                if (prefetchBuffer == null) {
                    this.clearSlotAndMaybeStartPending(slot);
                }
            }
            if (prefetchBuffer == null) {
                return false;
            }
            slice.put(prefetchBuffer);
            slice.flip();
            slice.position(Math.toIntExact(pos - prefetchedPos) + delta);
            this.clearSlotAndMaybeStartPending(slot);
            return true;
        }

        void clearSlotAndMaybeStartPending(int slot) {
            assert (this.prefetchThreads.get(slot) != null && this.prefetchThreads.get(slot).isDone());
            this.prefetchThreads.set(slot, null);
            this.posToSlot.remove(this.prefetchPos[slot]);
            if (this.pendingPrefetches.isEmpty()) {
                this.slots.addLast(slot);
                return;
            }
            long req = this.pendingPrefetches.removeFirst();
            this.posToSlot.put(req, slot);
            this.prefetchPos[slot] = req;
            this.startPrefetch(req, slot);
        }

        private boolean assertSlotsConsistent() {
            this.posToSlot.forEach((k, v) -> {
                if (this.prefetchThreads.get((int)v) == null) {
                    throw new AssertionError((Object)("posToSlot inconsistent: slot " + v + " for pos " + k + " has no prefetch thread"));
                }
                if (this.prefetchPos[v] != k) {
                    throw new AssertionError((Object)("posToSlot inconsistent: slot " + v + " for pos " + k + " has prefetchPos " + this.prefetchPos[v]));
                }
            });
            return true;
        }

        void startPrefetch(long pos, int slot) {
            Future<ByteBuffer> future = this.executor.submit(() -> {
                ByteBuffer[] prefetchBuffers = this.prefetchBuffers;
                ByteBuffer prefetchBuffer = prefetchBuffers[slot];
                if (prefetchBuffer == null) {
                    prefetchBuffers[slot] = prefetchBuffer = AsyncDirectIOIndexInput.allocateBuffer(this.prefetchBytesSize, this.blockSize);
                } else {
                    prefetchBuffer.clear();
                }
                assert (pos % (long)this.blockSize == 0L) : "prefetch pos [" + pos + "] must be aligned to block size [" + this.blockSize + "]";
                AsyncDirectIOIndexInput.readDirectChannel(this.channel, prefetchBuffer, pos);
                prefetchBuffer.flip();
                return prefetchBuffer;
            });
            this.prefetchThreads.set(slot, future);
            assert (this.assertSlotsConsistent());
        }

        @Override
        public void close() throws IOException {
            this.executor.shutdownNow();
        }
    }
}

