/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.SortedNumericSortField;
import org.apache.lucene.search.SortedSetSortField;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.IndexMode;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.index.fielddata.IndexFieldData;
import org.elasticsearch.index.mapper.MappedFieldType;
import org.elasticsearch.search.MultiValueMode;
import org.elasticsearch.search.lookup.SearchLookup;
import org.elasticsearch.search.sort.SortOrder;

public final class IndexSortConfig {
    private static final DeprecationLogger DEPRECATION_LOGGER = DeprecationLogger.getLogger(IndexSortConfig.class);
    public static final Setting<List<String>> INDEX_SORT_FIELD_SETTING = Setting.stringListSettingWithDefaultProvider("index.sort.field", IndexSortConfigDefaults::getDefaultSortFields, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    public static final Setting<List<SortOrder>> INDEX_SORT_ORDER_SETTING = Setting.listSetting("index.sort.order", IndexSortConfigDefaults::getDefaultSortOrder, IndexSortConfig::parseOrderMode, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    public static final Setting<List<MultiValueMode>> INDEX_SORT_MODE_SETTING = Setting.listSetting("index.sort.mode", IndexSortConfigDefaults::getDefaultSortMode, IndexSortConfig::parseMultiValueMode, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    public static final Setting<List<String>> INDEX_SORT_MISSING_SETTING = Setting.listSetting("index.sort.missing", IndexSortConfigDefaults::getDefaultSortMissing, IndexSortConfig::validateMissingValue, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    final FieldSortSpec[] sortSpecs;
    private final IndexVersion indexCreatedVersion;
    private final String indexName;
    private final IndexMode indexMode;
    private static final EnumSet<SortField.Type> ALLOWED_INDEX_SORT_TYPES = EnumSet.of(SortField.Type.STRING, SortField.Type.LONG, SortField.Type.INT, SortField.Type.DOUBLE, SortField.Type.FLOAT);

    private static String validateMissingValue(String missing) {
        if (!"_last".equals(missing) && !"_first".equals(missing)) {
            throw new IllegalArgumentException("Illegal missing value:[" + missing + "], must be one of [_last, _first]");
        }
        return missing;
    }

    private static SortOrder parseOrderMode(String value) {
        try {
            return SortOrder.fromString(value);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Illegal sort order:" + value);
        }
    }

    private static MultiValueMode parseMultiValueMode(String value) {
        MultiValueMode mode = MultiValueMode.fromString(value);
        if (mode != MultiValueMode.MAX && mode != MultiValueMode.MIN) {
            throw new IllegalArgumentException("Illegal index sort mode:[" + String.valueOf(mode) + "], must be one of [" + String.valueOf(MultiValueMode.MAX) + ", " + String.valueOf(MultiValueMode.MIN) + "]");
        }
        return mode;
    }

    private static void checkSizeMismatch(String firstKey, List<?> first, String secondKey, List<?> second) {
        if (first.size() != second.size()) {
            throw new IllegalArgumentException(firstKey + ":" + String.valueOf(first) + " " + secondKey + ":" + String.valueOf(second) + ", size mismatch");
        }
    }

    private static void validateSortSettings(Settings settings) {
        if (!INDEX_SORT_FIELD_SETTING.exists(settings)) {
            for (Setting setting : new Setting[]{INDEX_SORT_ORDER_SETTING, INDEX_SORT_MODE_SETTING, INDEX_SORT_MISSING_SETTING}) {
                if (!setting.exists(settings)) continue;
                throw new IllegalArgumentException("setting [" + setting.getKey() + "] requires [" + INDEX_SORT_FIELD_SETTING.getKey() + "] to be configured");
            }
        }
        List<String> fields = INDEX_SORT_FIELD_SETTING.get(settings);
        List<SortOrder> order = INDEX_SORT_ORDER_SETTING.get(settings);
        IndexSortConfig.checkSizeMismatch(INDEX_SORT_FIELD_SETTING.getKey(), fields, INDEX_SORT_ORDER_SETTING.getKey(), order);
        List<MultiValueMode> mode = INDEX_SORT_MODE_SETTING.get(settings);
        IndexSortConfig.checkSizeMismatch(INDEX_SORT_FIELD_SETTING.getKey(), fields, INDEX_SORT_MODE_SETTING.getKey(), mode);
        List<String> missing = INDEX_SORT_MISSING_SETTING.get(settings);
        IndexSortConfig.checkSizeMismatch(INDEX_SORT_FIELD_SETTING.getKey(), fields, INDEX_SORT_MISSING_SETTING.getKey(), missing);
    }

    public IndexSortConfig(IndexSettings indexSettings) {
        Settings settings = indexSettings.getSettings();
        this.indexCreatedVersion = indexSettings.getIndexVersionCreated();
        this.indexName = indexSettings.getIndex().getName();
        this.indexMode = indexSettings.getMode();
        IndexSortConfig.validateSortSettings(settings);
        List<String> fields = INDEX_SORT_FIELD_SETTING.get(settings);
        this.sortSpecs = (FieldSortSpec[])fields.stream().map(FieldSortSpec::new).toArray(FieldSortSpec[]::new);
        List<SortOrder> orders = INDEX_SORT_ORDER_SETTING.get(settings);
        for (int i = 0; i < this.sortSpecs.length; ++i) {
            this.sortSpecs[i].order = orders.get(i);
        }
        List<MultiValueMode> modes = INDEX_SORT_MODE_SETTING.get(settings);
        for (int i = 0; i < this.sortSpecs.length; ++i) {
            this.sortSpecs[i].mode = modes.get(i);
        }
        List<String> missingValues = INDEX_SORT_MISSING_SETTING.get(settings);
        for (int i = 0; i < this.sortSpecs.length; ++i) {
            this.sortSpecs[i].missingValue = missingValues.get(i);
        }
    }

    public boolean hasIndexSort() {
        return this.sortSpecs.length > 0;
    }

    public boolean hasPrimarySortOnField(String field) {
        return this.sortSpecs.length > 0 && this.sortSpecs[0].field.equals(field);
    }

    public Sort buildIndexSort(Function<String, MappedFieldType> fieldTypeLookup, BiFunction<MappedFieldType, Supplier<SearchLookup>, IndexFieldData<?>> fieldDataLookup) {
        if (!this.hasIndexSort()) {
            return null;
        }
        SortField[] sortFields = new SortField[this.sortSpecs.length];
        for (int i = 0; i < this.sortSpecs.length; ++i) {
            IndexFieldData<?> fieldData;
            FieldSortSpec sortSpec = this.sortSpecs[i];
            MappedFieldType ft = fieldTypeLookup.apply(sortSpec.field);
            if (ft == null) {
                String err = "unknown index sort field:[" + sortSpec.field + "]";
                if (this.indexMode == IndexMode.TIME_SERIES) {
                    err = err + " required by [" + IndexSettings.MODE.getKey() + "=time_series]";
                }
                throw new IllegalArgumentException(err);
            }
            if (!Objects.equals(ft.name(), sortSpec.field)) {
                if (this.indexCreatedVersion.onOrAfter(IndexVersions.V_7_13_0)) {
                    throw new IllegalArgumentException("Cannot use alias [" + sortSpec.field + "] as an index sort field");
                }
                DEPRECATION_LOGGER.warn(DeprecationCategory.MAPPINGS, "index-sort-aliases", "Index sort for index [" + this.indexName + "] defined on field [" + sortSpec.field + "] which resolves to field [" + ft.name() + "]. You will not be able to define an index sort over aliased fields in new indexes", new Object[0]);
            }
            boolean reverse = sortSpec.order == null ? false : sortSpec.order == SortOrder.DESC;
            MultiValueMode mode = sortSpec.mode;
            if (mode == null) {
                mode = reverse ? MultiValueMode.MAX : MultiValueMode.MIN;
            }
            try {
                fieldData = fieldDataLookup.apply(ft, () -> {
                    throw new UnsupportedOperationException("index sorting not supported on runtime field [" + ft.name() + "]");
                });
            }
            catch (Exception e) {
                throw new IllegalArgumentException("docvalues not found for index sort field:[" + sortSpec.field + "]", e);
            }
            if (fieldData == null) {
                throw new IllegalArgumentException("docvalues not found for index sort field:[" + sortSpec.field + "]");
            }
            sortFields[i] = fieldData.sortField(this.indexCreatedVersion, sortSpec.missingValue, mode, null, reverse);
            IndexSortConfig.validateIndexSortField(sortFields[i]);
        }
        return new Sort(sortFields);
    }

    private static void validateIndexSortField(SortField sortField) {
        SortField.Type type = IndexSortConfig.getSortFieldType(sortField);
        if (!ALLOWED_INDEX_SORT_TYPES.contains(type)) {
            throw new IllegalArgumentException("invalid index sort field:[" + sortField.getField() + "]");
        }
    }

    public boolean hasSortOnField(String fieldName) {
        for (FieldSortSpec sortSpec : this.sortSpecs) {
            if (!sortSpec.field.equals(fieldName)) continue;
            return true;
        }
        return false;
    }

    public static SortField.Type getSortFieldType(SortField sortField) {
        if (sortField instanceof SortedSetSortField) {
            return SortField.Type.STRING;
        }
        if (sortField instanceof SortedNumericSortField) {
            return ((SortedNumericSortField)sortField).getNumericType();
        }
        return sortField.getType();
    }

    public static class FieldSortSpec {
        final String field;
        SortOrder order;
        MultiValueMode mode;
        String missingValue;

        FieldSortSpec(String field) {
            this.field = field;
        }

        public String getField() {
            return this.field;
        }

        public SortOrder getOrder() {
            return this.order;
        }
    }

    public static class IndexSortConfigDefaults {
        public static final FieldSortSpec[] TIME_SERIES_SORT;
        public static final FieldSortSpec[] TIMESTAMP_SORT;
        public static final FieldSortSpec[] HOSTNAME_TIMESTAMP_SORT;
        public static final FieldSortSpec[] HOSTNAME_TIMESTAMP_BWC_SORT;

        public static FieldSortSpec[] getDefaultSortSpecs(Settings settings) {
            if (settings.isEmpty()) {
                return new FieldSortSpec[0];
            }
            String indexMode = settings.get(IndexSettings.MODE.getKey());
            if (indexMode != null) {
                indexMode = indexMode.toLowerCase(Locale.ROOT);
            }
            if (IndexMode.TIME_SERIES.getName().equals(indexMode)) {
                return TIME_SERIES_SORT;
            }
            if (IndexMode.LOGSDB.getName().equals(indexMode)) {
                IndexVersion version = IndexMetadata.SETTING_INDEX_VERSION_CREATED.get(settings);
                if (version.onOrAfter(IndexVersions.LOGSB_OPTIONAL_SORTING_ON_HOST_NAME) || version.between(IndexVersions.LOGSB_OPTIONAL_SORTING_ON_HOST_NAME_BACKPORT, IndexVersions.UPGRADE_TO_LUCENE_10_0_0)) {
                    return IndexSettings.LOGSDB_SORT_ON_HOST_NAME.get(settings) != false ? HOSTNAME_TIMESTAMP_SORT : TIMESTAMP_SORT;
                }
                return HOSTNAME_TIMESTAMP_BWC_SORT;
            }
            return new FieldSortSpec[0];
        }

        public static FieldSortSpec[] getSortSpecs(Settings settings) {
            if (!INDEX_SORT_FIELD_SETTING.exists(settings)) {
                return IndexSortConfigDefaults.getDefaultSortSpecs(settings);
            }
            List<String> fields = INDEX_SORT_FIELD_SETTING.get(settings);
            FieldSortSpec[] sortSpecs = (FieldSortSpec[])fields.stream().map(FieldSortSpec::new).toArray(FieldSortSpec[]::new);
            if (INDEX_SORT_ORDER_SETTING.exists(settings)) {
                List<SortOrder> orders = INDEX_SORT_ORDER_SETTING.get(settings);
                for (int i = 0; i < sortSpecs.length; ++i) {
                    sortSpecs[i].order = orders.get(i);
                }
            }
            return sortSpecs;
        }

        public static List<String> getDefaultSortFields(Settings settings) {
            return Arrays.stream(IndexSortConfigDefaults.getDefaultSortSpecs(settings)).map(sortSpec -> sortSpec.field).toList();
        }

        public static List<String> getDefaultSortOrder(Settings settings) {
            return Arrays.stream(IndexSortConfigDefaults.getSortSpecs(settings)).map(sortSpec -> sortSpec.order != null ? sortSpec.order : SortOrder.ASC).map(Enum::toString).toList();
        }

        public static List<String> getDefaultSortMode(Settings settings) {
            return Arrays.stream(IndexSortConfigDefaults.getSortSpecs(settings)).map(sortSpec -> {
                if (sortSpec.mode != null) {
                    return sortSpec.mode;
                }
                if (sortSpec.order == SortOrder.DESC) {
                    return MultiValueMode.MAX;
                }
                return MultiValueMode.MIN;
            }).map(order -> order.toString().toLowerCase(Locale.ROOT)).toList();
        }

        public static List<String> getDefaultSortMissing(Settings settings) {
            return Arrays.stream(IndexSortConfigDefaults.getSortSpecs(settings)).map(sortSpec -> sortSpec.missingValue != null ? sortSpec.missingValue : "_last").toList();
        }

        static {
            FieldSortSpec timeStampSpec = new FieldSortSpec("@timestamp");
            timeStampSpec.order = SortOrder.DESC;
            TIME_SERIES_SORT = new FieldSortSpec[]{new FieldSortSpec("_tsid"), timeStampSpec};
            TIMESTAMP_SORT = new FieldSortSpec[]{timeStampSpec};
            FieldSortSpec hostnameSpec = new FieldSortSpec("host.name");
            hostnameSpec.order = SortOrder.ASC;
            hostnameSpec.missingValue = "_last";
            hostnameSpec.mode = MultiValueMode.MIN;
            HOSTNAME_TIMESTAMP_SORT = new FieldSortSpec[]{hostnameSpec, timeStampSpec};
            HOSTNAME_TIMESTAMP_BWC_SORT = new FieldSortSpec[]{new FieldSortSpec("host.name"), new FieldSortSpec("@timestamp")};
        }
    }
}

