"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateUrl = exports.validatePath = exports.validateMimeType = exports.validateFileSignature = void 0;
var Path = _interopRequireWildcard(require("path"));
var _url = require("url");
var _promises = require("fs/promises");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Protects against path traversal attacks in file paths
const validatePath = filePath => {
  const dirPath = Path.dirname(filePath);
  const normalizedPath = Path.normalize(filePath);
  if (normalizedPath.includes('..') || dirPath !== '.' && !normalizedPath.startsWith(dirPath)) {
    throw new Error('Path traversal attempt detected');
  }
};

// Validates URLs to prevent malicious protocol usage and path traversal
exports.validatePath = validatePath;
const validateUrl = urlString => {
  try {
    const parsedUrl = new _url.URL(urlString);
    const allowedProtocols = ['http:', 'https:', 'file:'];
    if (!allowedProtocols.includes(parsedUrl.protocol)) {
      throw new Error(`Unsupported protocol: ${parsedUrl.protocol}`);
    }
  } catch (error) {
    throw error;
  }
};

// Validates HTTP Content-Type headers against expected MIME types
exports.validateUrl = validateUrl;
const validateMimeType = (contentType, expectedMimeType) => {
  if (!contentType) {
    throw new Error('Missing Content-Type header');
  }
  const mimeType = contentType.split(';')[0].trim().toLowerCase();
  if (mimeType !== expectedMimeType.toLowerCase()) {
    throw new Error(`Invalid MIME type: ${mimeType}. Expected: ${expectedMimeType}`);
  }
};

// Validates file signatures to prevent file type spoofing
exports.validateMimeType = validateMimeType;
const validateFileSignature = async (filePath, expectedMimeType) => {
  const fileBuffer = await readFileHeader(filePath);
  const firstBytes = fileBuffer.subarray(0, 8);
  switch (expectedMimeType) {
    case 'application/zip':
      if (firstBytes[0] !== 0x50 || firstBytes[1] !== 0x4b) {
        throw new Error('File content does not match ZIP format');
      }
      break;
    case 'application/xml':
      if (firstBytes[0] !== 0x3c || firstBytes[1] !== 0x3f) {
        throw new Error('File content does not match XML format');
      }
      break;
    default:
      throw new Error(`File signature validation is not supported for MIME type: ${expectedMimeType}`);
  }
};
exports.validateFileSignature = validateFileSignature;
const readFileHeader = async (filePath, bytesToRead = 8) => {
  const fileHandle = await (0, _promises.open)(filePath, 'r');
  try {
    const buffer = Buffer.alloc(bytesToRead);
    const {
      bytesRead
    } = await fileHandle.read(buffer, 0, bytesToRead, 0);
    return buffer.subarray(0, bytesRead);
  } finally {
    await fileHandle.close();
  }
};