"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TrackingService = exports.ToolCallSource = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _usage_counters = require("./usage_counters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/**
 * Tool call source - identifies where the tool was called from
 */
let ToolCallSource = exports.ToolCallSource = /*#__PURE__*/function (ToolCallSource) {
  ToolCallSource["DEFAULT_AGENT"] = "default_agent";
  ToolCallSource["CUSTOM_AGENT"] = "custom_agent";
  ToolCallSource["MCP"] = "mcp";
  ToolCallSource["API"] = "api";
  ToolCallSource["A2A"] = "a2a";
  return ToolCallSource;
}({});
/**
 * Tracking service for telemetry collection
 *
 * Centralized service for tracking all metrics related to Agent Builder usage:
 * - Tool calls by source
 * - Errors by type and context
 * - Conversation rounds
 * - Query-to-result times
 * - LLM provider and model usage
 * - Token consumption
 */
class TrackingService {
  constructor(usageCounter, logger) {
    // In-memory tracking for query times (request ID → start time)
    (0, _defineProperty2.default)(this, "queryStartTimes", new Map());
    this.usageCounter = usageCounter;
    this.logger = logger;
  }

  /**
   * Track a tool call
   * @param toolId - Tool identifier
   * @param source - Where the tool was called from
   */
  trackToolCall(toolId, source) {
    try {
      this.usageCounter.incrementCounter({
        counterName: `${_usage_counters.ONECHAT_USAGE_DOMAIN}_tool_call_${source}`,
        counterType: 'count',
        incrementBy: 1
      });
      this.logger.debug(`Tracked tool call: ${toolId} from ${source}`);
    } catch (error) {
      this.logger.error(`Failed to track tool call: ${error.message}`);
    }
  }

  /**
   * Track conversation round count
   * @param conversationId - Conversation identifier
   * @param roundNumber - Current round number
   */
  trackConversationRound(conversationId, roundNumber) {
    try {
      // Determine bucket for round count
      let bucket;
      if (roundNumber <= 5) {
        bucket = 'rounds_1-5';
      } else if (roundNumber <= 10) {
        bucket = 'rounds_6-10';
      } else if (roundNumber <= 20) {
        bucket = 'rounds_11-20';
      } else if (roundNumber <= 50) {
        bucket = 'rounds_21-50';
      } else {
        bucket = 'rounds_51+';
      }
      this.usageCounter.incrementCounter({
        counterName: `${_usage_counters.ONECHAT_USAGE_DOMAIN}_${bucket}`,
        counterType: 'count',
        incrementBy: 1
      });
      this.logger.debug(`Tracked conversation round: ${conversationId} round ${roundNumber}`);
    } catch (error) {
      this.logger.error(`Failed to track conversation round: ${error.message}`);
    }
  }

  /**
   * Track query start time
   * @param requestId - Unique request identifier
   */
  trackQueryStart(requestId) {
    try {
      if (!requestId) {
        requestId = (0, _uuid.v4)();
      }
      this.queryStartTimes.set(requestId, Date.now());
      this.logger.debug(`Tracked query start: ${requestId}`);
      return requestId;
    } catch (error) {
      this.logger.error(`Failed to track query start: ${error.message}`);
    }
  }

  /**
   * Track query end time and calculate duration
   * @param requestId - Unique request identifier
   */
  trackQueryEnd(requestId) {
    try {
      const startTime = this.queryStartTimes.get(requestId);
      if (!startTime) {
        this.logger.warn(`No start time found for request: ${requestId}`);
        return;
      }
      const durationMs = Date.now() - startTime;
      this.queryStartTimes.delete(requestId);

      // Determine bucket for duration
      let bucket;
      if (durationMs < 1000) {
        bucket = 'query_to_result_time_<1s';
      } else if (durationMs < 5000) {
        bucket = 'query_to_result_time_1-5s';
      } else if (durationMs < 10000) {
        bucket = 'query_to_result_time_5-10s';
      } else if (durationMs < 30000) {
        bucket = 'query_to_result_time_10-30s';
      } else {
        bucket = 'query_to_result_time_30s+';
      }
      this.usageCounter.incrementCounter({
        counterName: `${_usage_counters.ONECHAT_USAGE_DOMAIN}_${bucket}`,
        counterType: 'count',
        incrementBy: 1
      });
      this.logger.debug(`Tracked query end: ${requestId} duration: ${durationMs}ms`);
    } catch (error) {
      this.logger.error(`Failed to track query end: ${error.message}`);
    }
  }

  /**
   * Track LLM usage by provider and model
   * @param provider - LLM provider (e.g., 'openai', 'bedrock')
   * @param model - Model identifier
   */
  trackLLMUsage(provider, model) {
    try {
      const normalizedProvider = provider || 'unknown';
      const normalizedModel = model || 'unknown';
      const sanitizedProvider = normalizedProvider.replace(/[^a-zA-Z0-9_-]/g, '_');
      const sanitizedModel = normalizedModel.replace(/[^a-zA-Z0-9_-]/g, '_');
      (0, _usage_counters.trackLLMUsage)(this.usageCounter, sanitizedProvider, sanitizedModel);
      this.logger.debug(`Tracked LLM usage: provider=${sanitizedProvider}, model=${sanitizedModel}`);
    } catch (error) {
      this.logger.error(`Failed to track LLM usage: ${error.message}`);
    }
  }
}
exports.TrackingService = TrackingService;