"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runTool = exports.createToolHandlerContext = void 0;
var _onechatCommon = require("@kbn/onechat-common");
var _inferenceTracing = require("@kbn/inference-tracing");
var _onechatServer = require("@kbn/onechat-server");
var _tools = require("@kbn/onechat-server/tools");
var _telemetry = require("../../telemetry");
var _utils = require("../tools/utils");
var _run_context = require("./utils/run_context");
var _events = require("./utils/events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const runTool = async ({
  toolExecutionParams,
  parentManager
}) => {
  const {
    toolId,
    toolParams
  } = toolExecutionParams;
  const context = (0, _run_context.forkContextForToolRun)({
    parentContext: parentManager.context,
    toolId
  });
  const manager = parentManager.createChild(context);
  const {
    toolsService,
    request,
    resultStore,
    trackingService
  } = manager.deps;
  if (trackingService) {
    try {
      trackingService.trackToolCall(toolId, _telemetry.ToolCallSource.API);
    } catch (error) {
      /* empty */
    }
  }
  const toolRegistry = await toolsService.getRegistry({
    request
  });
  const tool = await toolRegistry.get(toolId);
  const {
    results
  } = await (0, _inferenceTracing.withExecuteToolSpan)(tool.id, {
    tool: {
      input: toolParams
    }
  }, async () => {
    const schema = await tool.getSchema();
    const validation = schema.safeParse(toolParams);
    if (validation.error) {
      throw (0, _onechatCommon.createBadRequestError)(`Tool ${toolId} was called with invalid parameters: ${validation.error.message}`);
    }
    const toolHandlerContext = await createToolHandlerContext({
      toolExecutionParams,
      manager
    });
    try {
      const toolHandler = await tool.getHandler();
      return await toolHandler(validation.data, toolHandlerContext);
    } catch (err) {
      return {
        results: [(0, _onechatServer.createErrorResult)(err.message)]
      };
    }
  });
  const resultsWithIds = results.map(result => {
    var _result$tool_result_i;
    return {
      ...result,
      tool_result_id: (_result$tool_result_i = result.tool_result_id) !== null && _result$tool_result_i !== void 0 ? _result$tool_result_i : (0, _tools.getToolResultId)()
    };
  });
  resultsWithIds.forEach(result => {
    resultStore.add(result);
  });
  return {
    results: resultsWithIds
  };
};
exports.runTool = runTool;
const createToolHandlerContext = async ({
  manager,
  toolExecutionParams
}) => {
  const {
    onEvent
  } = toolExecutionParams;
  const {
    request,
    elasticsearch,
    modelProvider,
    toolsService,
    resultStore,
    logger
  } = manager.deps;
  return {
    request,
    logger,
    esClient: elasticsearch.client.asScoped(request),
    modelProvider,
    runner: manager.getRunner(),
    toolProvider: (0, _utils.registryToProvider)({
      registry: await toolsService.getRegistry({
        request
      }),
      getRunner: manager.getRunner,
      request
    }),
    resultStore: resultStore.asReadonly(),
    events: (0, _events.createToolEventEmitter)({
      eventHandler: onEvent,
      context: manager.context
    })
  };
};
exports.createToolHandlerContext = createToolHandlerContext;