"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.transformFormDataToEsqlTool = exports.transformEsqlToolToFormData = exports.transformEsqlFormDataForUpdate = exports.transformEsqlFormDataForCreate = void 0;
var _esqlUtils = require("@kbn/esql-utils");
var _onechatCommon = require("@kbn/onechat-common");
var _lodash = require("lodash");
var _tool_form_types = require("../components/tools/form/types/tool_form_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Transforms an ES|QL tool into its UI form representation.
 * @param tool - The ES|QL tool to transform.
 * @returns The ES|QL tool form data.
 */
const transformEsqlToolToFormData = tool => {
  return {
    toolId: tool.id,
    description: tool.description,
    esql: tool.configuration.query,
    labels: tool.tags,
    params: Object.entries(tool.configuration.params).map(([name, {
      type,
      description,
      optional
    }]) => ({
      name,
      type,
      description,
      source: _tool_form_types.EsqlParamSource.Custom,
      optional: optional !== null && optional !== void 0 ? optional : false
    })),
    type: _onechatCommon.ToolType.esql
  };
};

/**
 * Transforms ES|QL tool form data into a `ToolDefinition` entity.
 * @param data - The ES|QL form data to transform.
 * @returns The transformed data as an ES|QL tool.
 */
exports.transformEsqlToolToFormData = transformEsqlToolToFormData;
const transformFormDataToEsqlTool = data => {
  const esqlParams = new Set((0, _esqlUtils.getESQLQueryVariables)(data.esql));
  return {
    id: data.toolId,
    description: data.description,
    readonly: false,
    configuration: {
      query: data.esql,
      params: data.params.filter(param => esqlParams.has(param.name)).reduce((paramsMap, param) => {
        paramsMap[param.name] = {
          type: param.type,
          description: param.description,
          optional: param.optional
        };
        return paramsMap;
      }, {})
    },
    type: _onechatCommon.ToolType.esql,
    tags: data.labels
  };
};

/**
 * Transforms ES|QL form data into a payload for the create tools API.
 * @param data - The ES|QL form data to transform.
 * @returns The payload for the create tools API.
 */
exports.transformFormDataToEsqlTool = transformFormDataToEsqlTool;
const transformEsqlFormDataForCreate = data => {
  return (0, _lodash.omit)(transformFormDataToEsqlTool(data), ['readonly']);
};

/**
 * Transforms ES|QL tool form data into a payload for the update tool API.
 * @param data - The ES|QL form data to transform.
 * @returns The payload for the update tool API.
 */
exports.transformEsqlFormDataForCreate = transformEsqlFormDataForCreate;
const transformEsqlFormDataForUpdate = data => {
  return (0, _lodash.omit)(transformFormDataToEsqlTool(data), ['id', 'type', 'readonly']);
};
exports.transformEsqlFormDataForUpdate = transformEsqlFormDataForUpdate;