"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReferencedPanelManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ReferencedPanelManager {
  constructor(logger, soClient) {
    // The uid refers to the ID of the saved object reference, while the panelId refers to the ID of the saved object itself (the panel).
    // So, if the same saved object panel is referenced in two different dashboards, it will have different uid values in each dashboard, but the same panelId, since they're both referencing the same panel.
    (0, _defineProperty2.default)(this, "panelsById", new Map());
    (0, _defineProperty2.default)(this, "panelUidToId", new Map());
    (0, _defineProperty2.default)(this, "panelsTypeById", new Map());
    this.logger = logger;
    this.soClient = soClient;
  }
  async fetchReferencedPanels() {
    if (this.panelsTypeById.size === 0) {
      return;
    }
    const panelsToFetch = [...this.panelsTypeById.entries()].map(([id, type]) => ({
      id,
      type
    }));
    try {
      const {
        saved_objects: savedObjects
      } = await this.soClient.bulkGet(panelsToFetch);
      savedObjects.forEach(so => {
        this.panelsById.set(so.id, {
          ...so.attributes,
          references: so.references
        });
      });
    } catch (error) {
      this.logger.error(`Error fetching ${panelsToFetch.length} panels : ${error.message}`);
    }
  }
  getByUid(uid) {
    const panelId = this.panelUidToId.get(uid);
    return panelId ? this.panelsById.get(panelId) : undefined;
  }

  // This method adds the panel type to the map, so that we can fetch the panel later and it links the panel uid to the panelId.
  addReferencedPanel({
    dashboardId,
    references,
    panel
  }) {
    const {
      uid,
      type
    } = panel;
    if (!uid) return;
    const panelReference = references.find(r => r.name.includes(uid) && r.type === type);
    // A reference of the panel was not found
    if (!panelReference) {
      this.logger.error(`Reference for panel of type ${type} and uid ${uid} was not found in dashboard with id ${dashboardId}`);
      return;
    }
    const panelId = panelReference.id;
    this.panelUidToId.set(uid, panelId);
    if (!this.panelsTypeById.has(panelId)) {
      this.panelsTypeById.set(panelId, panel.type);
    }
  }
}
exports.ReferencedPanelManager = ReferencedPanelManager;