"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ANNOTATION_MIN_WIDTH = exports.ANNOTATION_MASK_ID = void 0;
exports.getAnnotationBrush = getAnnotationBrush;
exports.getAnnotationLevels = getAnnotationLevels;
exports.getAnnotationWidth = getAnnotationWidth;
exports.highlightFocusChartAnnotation = highlightFocusChartAnnotation;
exports.renderAnnotations = renderAnnotations;
exports.unhighlightFocusChartAnnotation = unhighlightFocusChartAnnotation;
var _d = _interopRequireDefault(require("d3"));
var _moment = _interopRequireDefault(require("moment"));
var _annotations = require("../../../../../common/constants/annotations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ANNOTATION_MASK_ID = exports.ANNOTATION_MASK_ID = 'mlAnnotationMask';

// getAnnotationBrush() is expected to be called like getAnnotationBrush.call(this)
// so it gets passed on the context of the component it gets called from.
function getAnnotationBrush() {
  const {
    annotationUpdatesService
  } = this.props;
  const focusXScale = this.focusXScale;
  const annotateBrush = _d.default.svg.brush().x(focusXScale).on('brushend', brushend.bind(this));

  // cast a reference to this so we get the latest state when brushend() gets called
  function brushend() {
    const {
      selectedJob
    } = this.props;

    // TS TODO make this work with the actual types.
    const extent = annotateBrush.extent();
    const timestamp = extent[0].getTime();
    const endTimestamp = extent[1].getTime();
    if (timestamp === endTimestamp) {
      annotationUpdatesService.setValue(null);
      return;
    }
    const annotation = {
      timestamp,
      end_timestamp: endTimestamp,
      annotation: '',
      job_id: selectedJob.job_id,
      type: _annotations.ANNOTATION_TYPE.ANNOTATION
    };
    annotationUpdatesService.setValue(annotation);
  }
  return annotateBrush;
}

// Used to resolve overlapping annotations in the UI.
// The returned levels can be used to create a vertical offset.
function getAnnotationLevels(focusAnnotationData) {
  const levels = {};
  focusAnnotationData.forEach((d, i) => {
    if (d.key !== undefined) {
      const longerAnnotations = focusAnnotationData.filter((d2, i2) => i2 < i);
      levels[d.key] = longerAnnotations.reduce((level, d2) => {
        // For now we only support overlap removal for annotations which have both
        // `timestamp` and `end_timestamp` set.
        if (d.end_timestamp === undefined || d2.end_timestamp === undefined || d2.key === undefined) {
          return level;
        }
        if (
        // d2 is completely before d
        d2.timestamp < d.timestamp && d2.end_timestamp < d.timestamp ||
        // d2 is completely after d
        d2.timestamp > d.end_timestamp && d2.end_timestamp > d.end_timestamp) {
          return level;
        }
        return levels[d2.key] + 1;
      }, 0);
    }
  });
  return levels;
}
const ANNOTATION_DEFAULT_LEVEL = 1;
const ANNOTATION_LEVEL_HEIGHT = 28;
const ANNOTATION_UPPER_RECT_MARGIN = 0;
const ANNOTATION_UPPER_TEXT_MARGIN = -7;
const ANNOTATION_MIN_WIDTH = exports.ANNOTATION_MIN_WIDTH = 8;
const ANNOTATION_TEXT_VERTICAL_OFFSET = 26;
const ANNOTATION_TEXT_RECT_VERTICAL_OFFSET = 12;
const ANNOTATION_TEXT_RECT_WIDTH = 24;
const ANNOTATION_TEXT_RECT_HEIGHT = 20;
function renderAnnotations(focusChart, focusAnnotationData, focusZoomPanelHeight, focusChartHeight, focusXScale, showAnnotations, showFocusChartTooltip, hideFocusChartTooltip, annotationUpdatesService) {
  const upperRectMargin = ANNOTATION_UPPER_RECT_MARGIN;
  const upperTextMargin = ANNOTATION_UPPER_TEXT_MARGIN;
  const durations = {};
  focusAnnotationData.forEach(d => {
    if (d.key !== undefined) {
      const duration = (d.end_timestamp || 0) - d.timestamp;
      durations[d.key] = duration;
    }
  });

  // sort by duration
  focusAnnotationData.sort((a, b) => {
    if (a.key === undefined || b.key === undefined) {
      return 0;
    }
    return durations[b.key] - durations[a.key];
  });
  const levelHeight = ANNOTATION_LEVEL_HEIGHT;
  const levels = getAnnotationLevels(focusAnnotationData);
  const onAnnotationMouseOver = function (d) {
    showFocusChartTooltip(d, this);
  };
  const onAnnotationClick = d => {
    // clear a possible existing annotation previously set for editing before setting the new one.
    // this needs to be done explicitly here because a new annotation created using the brush tool
    // could still be present in the chart.
    annotationUpdatesService.setValue(null);
    // set the actual annotation and trigger the flyout
    annotationUpdatesService.setValue(d);
  };
  const annotations = focusChart.select('.ml-annotations').selectAll('g.ml-annotation').data(focusAnnotationData || [], d => d._id || '');
  annotations.enter().append('g').classed('ml-annotation', true);
  const rects = annotations.selectAll('.ml-annotation__rect').data(d => [d]);
  rects.enter().append('rect').classed('ml-annotation__rect', true).attr('mask', `url(#${ANNOTATION_MASK_ID})`).on('mouseover', onAnnotationMouseOver).on('mouseout', hideFocusChartTooltip).on('click', onAnnotationClick);
  rects.attr('x', d => {
    const date = (0, _moment.default)(d.timestamp);
    return Math.max(focusXScale(date), 0);
  }).attr('y', d => {
    const level = d.key !== undefined ? levels[d.key] : ANNOTATION_DEFAULT_LEVEL;
    return focusZoomPanelHeight + 1 + upperRectMargin + level * levelHeight;
  }).attr('height', d => {
    const level = d.key !== undefined ? levels[d.key] : ANNOTATION_DEFAULT_LEVEL;
    return focusChartHeight - 2 - upperRectMargin - level * levelHeight;
  }).attr('width', d => {
    const s = focusXScale((0, _moment.default)(d.timestamp)) + 1;
    const e = typeof d.end_timestamp !== 'undefined' ? focusXScale((0, _moment.default)(d.end_timestamp)) - 1 : s + ANNOTATION_MIN_WIDTH;
    const width = Math.max(ANNOTATION_MIN_WIDTH, e - s);
    return width;
  });
  rects.exit().remove();
  const textRects = annotations.selectAll('.ml-annotation__text-rect').data(d => [d]);
  const texts = annotations.selectAll('.ml-annotation__text').data(d => [d]);
  textRects.enter().append('rect').classed('ml-annotation__text-rect', true).attr('width', ANNOTATION_TEXT_RECT_WIDTH).attr('height', ANNOTATION_TEXT_RECT_HEIGHT).on('mouseover', onAnnotationMouseOver).on('mouseout', hideFocusChartTooltip).on('click', onAnnotationClick);
  texts.enter().append('text').classed('ml-annotation__text', true).on('mouseover', onAnnotationMouseOver).on('mouseout', hideFocusChartTooltip).on('click', onAnnotationClick);
  function labelXOffset(ts) {
    const earliestMs = focusXScale.domain()[0];
    const latestMs = focusXScale.domain()[1];
    const date = (0, _moment.default)(ts);
    const minX = Math.max(focusXScale(earliestMs), focusXScale(date));
    // To avoid overflow to the right, substract maxOffset which is
    // the width of the text label (24px) plus left margin (8xp).
    const maxOffset = 32;
    return Math.min(focusXScale(latestMs) - maxOffset, minX);
  }
  texts.attr('x', d => {
    const leftInnerOffset = 17;
    return labelXOffset(d.timestamp) + leftInnerOffset;
  }).attr('y', d => {
    const level = d.key !== undefined ? levels[d.key] : ANNOTATION_DEFAULT_LEVEL;
    return focusZoomPanelHeight + upperTextMargin + ANNOTATION_TEXT_VERTICAL_OFFSET + level * levelHeight;
  }).text(d => d.key);
  textRects.attr('x', d => {
    const leftInnerOffset = 5;
    return labelXOffset(d.timestamp) + leftInnerOffset;
  }).attr('y', d => {
    const level = d.key !== undefined ? levels[d.key] : ANNOTATION_DEFAULT_LEVEL;
    return focusZoomPanelHeight + upperTextMargin + ANNOTATION_TEXT_RECT_VERTICAL_OFFSET + level * levelHeight;
  });
  textRects.exit().remove();
  texts.exit().remove();
  annotations.classed('ml-annotation--hidden', !showAnnotations);
  annotations.exit().remove();
}
function getAnnotationWidth(annotation, focusXScale) {
  const start = focusXScale(annotation.timestamp) + 1;
  const end = typeof annotation.end_timestamp !== 'undefined' ? focusXScale(annotation.end_timestamp) - 1 : start + ANNOTATION_MIN_WIDTH;
  const width = Math.max(ANNOTATION_MIN_WIDTH, end - start);
  return width;
}
function highlightFocusChartAnnotation(annotation) {
  const annotations = _d.default.selectAll('.ml-annotation');
  annotations.each(function (d) {
    // @ts-ignore
    const element = _d.default.select(this);
    if (d._id === annotation._id) {
      element.selectAll('.ml-annotation__rect').classed('ml-annotation__rect--highlight', true);
    } else {
      element.selectAll('.ml-annotation__text-rect').classed('ml-annotation__text-rect--blur', true);
      element.selectAll('.ml-annotation__text').classed('ml-annotation__text--blur', true);
      element.selectAll('.ml-annotation__rect').classed('ml-annotation__rect--blur', true);
    }
  });
}
function unhighlightFocusChartAnnotation() {
  const annotations = _d.default.selectAll('.ml-annotation');
  annotations.each(function () {
    // @ts-ignore
    const element = _d.default.select(this);
    element.selectAll('.ml-annotation__text-rect').classed('ml-annotation__text-rect--blur', false);
    element.selectAll('.ml-annotation__rect').classed('ml-annotation__rect--highlight', false).classed('ml-annotation__rect--blur', false);
    element.selectAll('.ml-annotation__text').classed('ml-annotation__text--blur', false);
  });
}