"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDimensionsRoute = exports.dimensionsRoutes = void 0;
var _zod = require("@kbn/zod");
var _tracedEsClient = require("@kbn/traced-es-client");
var _zodHelpers = require("@kbn/zod-helpers");
var _datemath = require("@kbn/datemath");
var _server = require("@kbn/data-plugin/server");
var _create_route = require("../create_route");
var _get_dimensions = require("./get_dimensions");
var _utils = require("../../lib/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getDimensionsRoute = exports.getDimensionsRoute = (0, _create_route.createRoute)({
  endpoint: 'GET /internal/metrics_experience/dimensions',
  security: {
    authz: {
      enabled: false,
      reason: 'Authorization provided by Elasticsearch'
    }
  },
  params: _zod.z.object({
    query: _zod.z.object({
      dimensions: _zod.z.string().transform(str => {
        try {
          const parsed = JSON.parse(str);
          return parsed;
        } catch {
          throw new Error('Invalid JSON');
        }
      }).pipe(_zod.z.array(_zod.z.string()).min(1).max(10)),
      indices: _zod.z.string().optional().transform(str => {
        if (!str) return ['metrics-*'];
        try {
          const parsed = JSON.parse(str);
          return parsed;
        } catch {
          throw new Error('Invalid JSON');
        }
      }).pipe(_zod.z.array(_zod.z.string())),
      to: _zod.z.string().datetime().default((0, _datemath.parse)('now').toISOString()).transform(_zodHelpers.isoToEpoch),
      from: _zod.z.string().datetime().default((0, _datemath.parse)('now-15m', {
        roundUp: true
      }).toISOString()).transform(_zodHelpers.isoToEpoch)
    })
  }),
  handler: async ({
    context,
    params,
    logger,
    request
  }) => {
    const {
      elasticsearch,
      featureFlags
    } = await context.core;
    await (0, _utils.throwNotFoundIfMetricsExperienceDisabled)(featureFlags);
    const {
      dimensions,
      indices,
      from,
      to
    } = params.query;
    const esClient = elasticsearch.client.asCurrentUser;
    const values = await (0, _get_dimensions.getDimensions)({
      esClient: (0, _tracedEsClient.createTracedEsClient)({
        client: esClient,
        logger,
        plugin: 'metrics_experience',
        abortSignal: (0, _server.getRequestAbortedSignal)(request.events.aborted$)
      }),
      dimensions,
      indices,
      from,
      to,
      logger
    });
    return {
      values
    };
  }
});
const dimensionsRoutes = exports.dimensionsRoutes = {
  ...getDimensionsRoute
};