"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MAX_NUM_OF_COLUMNS = void 0;
exports.canColumnBeDroppedInMetricDimension = canColumnBeDroppedInMetricDimension;
exports.canColumnBeUsedBeInMetricDimension = canColumnBeUsedBeInMetricDimension;
exports.getAllColumns = void 0;
exports.getIndexPatternFromTextBasedQuery = getIndexPatternFromTextBasedQuery;
exports.getStateFromAggregateQuery = getStateFromAggregateQuery;
exports.isNumeric = exports.isNotNumeric = void 0;
exports.loadIndexPatternRefs = loadIndexPatternRefs;
exports.mergeLayer = mergeLayer;
exports.updateColumnFormat = updateColumnFormat;
exports.updateColumnLabel = updateColumnLabel;
var _esqlUtils = require("@kbn/esql-utils");
var _id_generator = require("../../../id_generator");
var _fetch_data_from_aggregate_query = require("./fetch_data_from_aggregate_query");
var _fieldlist_cache = require("./fieldlist_cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_NUM_OF_COLUMNS = exports.MAX_NUM_OF_COLUMNS = 5;
async function loadIndexPatternRefs(indexPatternsService) {
  const indexPatterns = await indexPatternsService.getIdsWithTitle();
  const timefields = await Promise.all(indexPatterns.map(p => indexPatternsService.get(p.id).then(pat => pat.timeFieldName)));
  return indexPatterns.map((p, i) => ({
    ...p,
    timeField: timefields[i]
  })).sort((a, b) => {
    return a.title.localeCompare(b.title);
  });
}
const getAllColumns = (existingColumns, columnsFromQuery) => {
  // filter out columns that do not exist on the query
  const columns = existingColumns.filter(c => {
    const columnExists = columnsFromQuery === null || columnsFromQuery === void 0 ? void 0 : columnsFromQuery.some(f => f.name === (c === null || c === void 0 ? void 0 : c.fieldName));
    if (columnExists) return c;
  });
  const allCols = [...columns, ...columnsFromQuery.map(c => ({
    columnId: c.id,
    fieldName: c.id,
    label: c.name,
    meta: c.meta,
    ...(c.variable ? {
      variable: c.variable
    } : {})
  }))];
  const uniqueIds = [];
  return allCols.filter(col => {
    const isDuplicate = uniqueIds.includes(col.columnId);
    if (!isDuplicate) {
      uniqueIds.push(col.columnId);
      return true;
    }
    return false;
  });
};
exports.getAllColumns = getAllColumns;
async function getStateFromAggregateQuery(state, query, dataViews, data, expressions, http, frameDataViews) {
  var _indexPatternRefs$fin, _indexPatternRefs$fin2, _state$layers$newLaye;
  let indexPatternRefs = frameDataViews !== null && frameDataViews !== void 0 && frameDataViews.indexPatternRefs.length ? frameDataViews.indexPatternRefs : await loadIndexPatternRefs(dataViews);
  const errors = [];
  const layerIds = Object.keys(state.layers);
  const context = state.initialContext;
  const newLayerId = layerIds.length > 0 ? layerIds[0] : (0, _id_generator.generateId)();
  // fetch the pattern from the query
  const indexPattern = getIndexPatternFromTextBasedQuery(query);
  // get the id of the dataview
  let dataViewId = (_indexPatternRefs$fin = (_indexPatternRefs$fin2 = indexPatternRefs.find(r => r.title === indexPattern)) === null || _indexPatternRefs$fin2 === void 0 ? void 0 : _indexPatternRefs$fin2.id) !== null && _indexPatternRefs$fin !== void 0 ? _indexPatternRefs$fin : '';
  let columnsFromQuery = [];
  let timeFieldName;
  try {
    var _table$columns;
    const dataView = await (0, _esqlUtils.getESQLAdHocDataview)({
      dataViewsService: dataViews,
      query: query.esql,
      options: {
        skipFetchFields: true
      },
      http
    });
    if (dataView && dataView.id) {
      dataViewId = dataView === null || dataView === void 0 ? void 0 : dataView.id;
      indexPatternRefs = [...indexPatternRefs, {
        id: dataView.id,
        title: dataView.name,
        timeField: dataView.timeFieldName
      }];
    }
    timeFieldName = dataView.timeFieldName;
    const table = await (0, _fetch_data_from_aggregate_query.fetchDataFromAggregateQuery)(query, dataView, data, expressions);
    columnsFromQuery = (_table$columns = table === null || table === void 0 ? void 0 : table.columns) !== null && _table$columns !== void 0 ? _table$columns : [];
    (0, _fieldlist_cache.addColumnsToCache)(query, columnsFromQuery);
  } catch (e) {
    errors.push(e);
  }
  const tempState = {
    layers: {
      [newLayerId]: {
        index: dataViewId,
        query,
        columns: (_state$layers$newLaye = state.layers[newLayerId].columns) !== null && _state$layers$newLaye !== void 0 ? _state$layers$newLaye : [],
        timeField: timeFieldName,
        errors
      }
    }
  };
  return {
    ...tempState,
    indexPatternRefs,
    initialContext: context
  };
}
function getIndexPatternFromTextBasedQuery(query) {
  return (0, _esqlUtils.getIndexPatternFromESQLQuery)(query.esql);
}
const isNumeric = column => {
  var _column$meta;
  return (column === null || column === void 0 ? void 0 : (_column$meta = column.meta) === null || _column$meta === void 0 ? void 0 : _column$meta.type) === 'number';
};
exports.isNumeric = isNumeric;
const isNotNumeric = column => !isNumeric(column);
exports.isNotNumeric = isNotNumeric;
function canColumnBeDroppedInMetricDimension(columns, selectedColumnType) {
  // check if at least one numeric field exists
  const hasNumberTypeColumns = columns === null || columns === void 0 ? void 0 : columns.some(isNumeric);
  return !hasNumberTypeColumns || hasNumberTypeColumns && selectedColumnType === 'number';
}
function canColumnBeUsedBeInMetricDimension(columns, selectedColumnType) {
  // check if at least one numeric field exists
  const hasNumberTypeColumns = columns === null || columns === void 0 ? void 0 : columns.some(isNumeric);
  return !hasNumberTypeColumns || columns.length >= MAX_NUM_OF_COLUMNS || hasNumberTypeColumns && selectedColumnType === 'number';
}
function mergeLayer({
  state,
  layerId,
  newLayer
}) {
  return {
    ...state,
    layers: {
      ...state.layers,
      [layerId]: {
        ...state.layers[layerId],
        ...newLayer
      }
    }
  };
}
function updateColumnLabel({
  layer,
  columnId,
  value
}) {
  const currentColumnIndex = layer.columns.findIndex(c => c.columnId === columnId);
  const currentColumn = layer.columns[currentColumnIndex];
  return {
    ...layer,
    columns: [...layer.columns.slice(0, currentColumnIndex), {
      ...currentColumn,
      label: value,
      customLabel: !!value
    }, ...layer.columns.slice(currentColumnIndex + 1)]
  };
}
function updateColumnFormat({
  layer,
  columnId,
  value
}) {
  const currentColumnIndex = layer.columns.findIndex(c => c.columnId === columnId);
  const currentColumn = layer.columns[currentColumnIndex];
  return {
    ...layer,
    columns: [...layer.columns.slice(0, currentColumnIndex), {
      ...currentColumn,
      params: {
        format: value
      }
    }, ...layer.columns.slice(currentColumnIndex + 1)]
  };
}