"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
var _exportNames = {
  unpackBufferToAssetsMap: true,
  unpackArchiveEntriesIntoMemory: true,
  traverseArchiveEntries: true,
  deletePackageCache: true,
  getPathParts: true,
  getAssetFromAssetsMap: true,
  getBufferExtractor: true,
  untarBuffer: true,
  unzipBuffer: true,
  generatePackageInfoFromArchiveBuffer: true
};
exports.deletePackageCache = void 0;
Object.defineProperty(exports, "generatePackageInfoFromArchiveBuffer", {
  enumerable: true,
  get: function () {
    return _parse.generatePackageInfoFromArchiveBuffer;
  }
});
exports.getAssetFromAssetsMap = getAssetFromAssetsMap;
Object.defineProperty(exports, "getBufferExtractor", {
  enumerable: true,
  get: function () {
    return _extract.getBufferExtractor;
  }
});
exports.getPathParts = getPathParts;
exports.traverseArchiveEntries = traverseArchiveEntries;
exports.unpackArchiveEntriesIntoMemory = unpackArchiveEntriesIntoMemory;
exports.unpackBufferToAssetsMap = unpackBufferToAssetsMap;
Object.defineProperty(exports, "untarBuffer", {
  enumerable: true,
  get: function () {
    return _extract.untarBuffer;
  }
});
Object.defineProperty(exports, "unzipBuffer", {
  enumerable: true,
  get: function () {
    return _extract.unzipBuffer;
  }
});
var _errors = require("../../../errors");
var _archive_iterator = require("./archive_iterator");
var _cache = require("./cache");
Object.keys(_cache).forEach(function (key) {
  if (key === "default" || key === "__esModule") return;
  if (Object.prototype.hasOwnProperty.call(_exportNames, key)) return;
  if (key in exports && exports[key] === _cache[key]) return;
  Object.defineProperty(exports, key, {
    enumerable: true,
    get: function () {
      return _cache[key];
    }
  });
});
var _extract = require("./extract");
var _parse = require("./parse");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function unpackBufferToAssetsMap({
  contentType,
  archiveBuffer,
  useStreaming
}) {
  const archiveIterator = (0, _archive_iterator.createArchiveIterator)(archiveBuffer, contentType);
  let paths = [];
  let assetsMap = new Map();
  if (useStreaming) {
    paths = await archiveIterator.getPaths();
    // We keep the assetsMap empty as we don't want to load all the assets in memory
    assetsMap = new Map();
  } else {
    const entries = await unpackArchiveEntriesIntoMemory(archiveBuffer, contentType);
    entries.forEach(entry => {
      const {
        path,
        buffer
      } = entry;
      if (buffer) {
        assetsMap.set(path, buffer);
        paths.push(path);
      }
    });
  }
  return {
    paths,
    assetsMap,
    archiveIterator
  };
}

/**
 * This function extracts all archive entries into memory.
 *
 * NOTE: This is potentially dangerous for large archives and can cause OOM
 * errors. Use 'traverseArchiveEntries' instead to iterate over the entries
 * without storing them all in memory at once.
 *
 * @param archiveBuffer
 * @param contentType
 * @returns All the entries in the archive buffer
 */
async function unpackArchiveEntriesIntoMemory(archiveBuffer, contentType) {
  const entries = [];
  const addToEntries = async entry => void entries.push(entry);
  await traverseArchiveEntries(archiveBuffer, contentType, addToEntries);

  // While unpacking a tar.gz file with unzipBuffer() will result in a thrown
  // error, unpacking a zip file with untarBuffer() just results in nothing.
  if (entries.length === 0) {
    throw new _errors.PackageInvalidArchiveError(`Archive seems empty. Assumed content type was ${contentType}, check if this matches the archive type.`);
  }
  return entries;
}
async function traverseArchiveEntries(archiveBuffer, contentType, onEntry, readBuffer) {
  const bufferExtractor = (0, _extract.getBufferExtractor)({
    contentType
  });
  if (!bufferExtractor) {
    throw new _errors.PackageUnsupportedMediaTypeError(`Unsupported media type ${contentType}. Please use 'application/gzip' or 'application/zip'`);
  }
  try {
    const onlyFiles = ({
      path
    }) => !path.endsWith('/');
    await bufferExtractor(archiveBuffer, onlyFiles, onEntry, readBuffer);
  } catch (error) {
    throw new _errors.PackageInvalidArchiveError(`Error during extraction of package: ${error}. Assumed content type was ${contentType}, check if this matches the archive type.`);
  }
}
const deletePackageCache = ({
  name,
  version
}) => {
  (0, _cache.deletePackageInfo)({
    name,
    version
  });
};
exports.deletePackageCache = deletePackageCache;
function getPathParts(path) {
  let dataset;
  let [pkgkey, service, type, file] = path.split('/');

  // if it's a data stream
  if (service === 'data_stream') {
    // save the dataset name
    dataset = type;
    // drop the `data_stream/dataset-name` portion & re-parse
    [pkgkey, service, type, file] = path.replace(`data_stream/${dataset}/`, '').split('/');
  }

  // To support the NOTICE asset at the root level
  if (service === 'NOTICE.txt') {
    file = service;
    type = 'notice';
    service = '';
  }

  // To support the LICENSE asset at the root level
  if (service === 'LICENSE.txt') {
    file = service;
    type = 'license';
    service = '';
  }

  // This is to cover for the fields.yml files inside the "fields" directory
  if (file === undefined) {
    file = type;
    type = 'fields';
    service = '';
  }
  return {
    pkgkey,
    service,
    type,
    file,
    dataset,
    path
  };
}
function getAssetFromAssetsMap(assetsMap, key) {
  const buffer = assetsMap.get(key);
  if (buffer === undefined) throw new _errors.PackageNotFoundError(`Cannot find asset ${key}`);
  return buffer;
}