"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postBulkAgentsUpgradeHandler = exports.postAgentUpgradeHandler = exports.checkKibanaVersion = void 0;
var _coerce = _interopRequireDefault(require("semver/functions/coerce"));
var _gt = _interopRequireDefault(require("semver/functions/gt"));
var _major = _interopRequireDefault(require("semver/functions/major"));
var _minor = _interopRequireDefault(require("semver/functions/minor"));
var _moment = _interopRequireDefault(require("moment"));
var _agents = _interopRequireWildcard(require("../../services/agents"));
var AgentService = _agents;
var _services = require("../../services");
var _errors = require("../../errors");
var _services2 = require("../../../common/services");
var _check_fleet_server_versions = require("../../../common/services/check_fleet_server_versions");
var _get_all_fleet_server_agents = require("../../collectors/get_all_fleet_server_agents");
var _versions = require("../../services/agents/versions");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const postAgentUpgradeHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const {
    version,
    source_uri: sourceUri,
    force,
    skipRateLimitCheck
  } = request.body;
  const kibanaVersion = _services.appContextService.getKibanaVersion();
  const latestAgentVersion = await (0, _versions.getLatestAvailableAgentVersion)();
  try {
    checkKibanaVersion(version, kibanaVersion, force);
  } catch (err) {
    return response.customError({
      statusCode: 400,
      body: {
        message: err.message
      }
    });
  }
  const agent = await (0, _agents.getAgentById)(esClient, soClient, request.params.agentId);
  const fleetServerAgents = await (0, _get_all_fleet_server_agents.getAllFleetServerAgents)(soClient, esClient);
  const agentIsFleetServer = fleetServerAgents.some(fleetServerAgent => fleetServerAgent.id === agent.id);
  if (!agentIsFleetServer) {
    try {
      (0, _check_fleet_server_versions.checkFleetServerVersion)(version, fleetServerAgents);
    } catch (err) {
      return response.customError({
        statusCode: 400,
        body: {
          message: err.message
        }
      });
    }
  }
  const {
    hasBeenUpgradedRecently,
    timeToWaitMs
  } = (0, _services2.getRecentUpgradeInfoForAgent)(agent);
  const timeToWaitString = _moment.default.utc(_moment.default.duration(timeToWaitMs).asMilliseconds()).format('mm[m]ss[s]');
  if (!skipRateLimitCheck && hasBeenUpgradedRecently) {
    return response.customError({
      statusCode: 429,
      body: {
        message: `agent ${request.params.agentId} was upgraded less than ${_services2.AGENT_UPGRADE_COOLDOWN_IN_MIN} minutes ago. Please wait ${timeToWaitString} before trying again to ensure the upgrade will not be rolled back.`
      },
      headers: {
        // retry-after expects seconds
        'retry-after': Math.ceil(timeToWaitMs / 1000).toString()
      }
    });
  }
  if (agent.unenrollment_started_at || agent.unenrolled_at) {
    return response.customError({
      statusCode: 400,
      body: {
        message: 'cannot upgrade an unenrolling or unenrolled agent'
      }
    });
  }
  if (!force && (0, _services2.isAgentUpgrading)(agent)) {
    return response.customError({
      statusCode: 400,
      body: {
        message: `agent ${request.params.agentId} is already upgrading`
      }
    });
  }
  if (!force && !skipRateLimitCheck && !(0, _services2.isAgentUpgradeableToVersion)(agent, version)) {
    return response.customError({
      statusCode: 400,
      body: {
        message: `Agent ${request.params.agentId} is not upgradeable: ${(0, _services2.getNotUpgradeableMessage)(agent, latestAgentVersion, version)}`
      }
    });
  }
  await AgentService.sendUpgradeAgentAction({
    soClient,
    esClient,
    agentId: request.params.agentId,
    version,
    sourceUri
  });
  const body = {};
  return response.ok({
    body
  });
};
exports.postAgentUpgradeHandler = postAgentUpgradeHandler;
const postBulkAgentsUpgradeHandler = async (context, request, response) => {
  const coreContext = await context.core;
  const soClient = coreContext.savedObjects.client;
  const esClient = coreContext.elasticsearch.client.asInternalUser;
  const {
    version,
    source_uri: sourceUri,
    agents,
    force,
    skipRateLimitCheck,
    rollout_duration_seconds: upgradeDurationSeconds,
    start_time: startTime,
    batchSize
  } = request.body;
  const kibanaVersion = _services.appContextService.getKibanaVersion();
  try {
    checkKibanaVersion(version, kibanaVersion, force);
    const fleetServerAgents = await (0, _get_all_fleet_server_agents.getAllFleetServerAgents)(soClient, esClient);
    (0, _check_fleet_server_versions.checkFleetServerVersion)(version, fleetServerAgents, force);
  } catch (err) {
    return response.customError({
      statusCode: 400,
      body: {
        message: err.message
      }
    });
  }
  const agentOptions = Array.isArray(agents) ? {
    agentIds: agents
  } : {
    kuery: agents,
    showInactive: request.body.includeInactive
  };
  const upgradeOptions = {
    ...agentOptions,
    sourceUri,
    version,
    force,
    skipRateLimitCheck,
    upgradeDurationSeconds,
    startTime,
    batchSize
  };
  const results = await AgentService.sendUpgradeAgentsActions(soClient, esClient, upgradeOptions);
  return response.ok({
    body: {
      actionId: results.actionId
    }
  });
};
exports.postBulkAgentsUpgradeHandler = postBulkAgentsUpgradeHandler;
const checkKibanaVersion = (version, kibanaVersion, force = false) => {
  var _semverCoerce, _semverCoerce2;
  // get version number only in case "-SNAPSHOT" is in it
  const kibanaVersionNumber = (_semverCoerce = (0, _coerce.default)(kibanaVersion)) === null || _semverCoerce === void 0 ? void 0 : _semverCoerce.version;
  if (!kibanaVersionNumber) throw new _errors.AgentRequestInvalidError(`KibanaVersion ${kibanaVersionNumber} is not valid`);
  const versionToUpgradeNumber = (_semverCoerce2 = (0, _coerce.default)(version)) === null || _semverCoerce2 === void 0 ? void 0 : _semverCoerce2.version;
  if (!versionToUpgradeNumber) throw new _errors.AgentRequestInvalidError(`Version to upgrade ${versionToUpgradeNumber} is not valid`);
  if (!force && (0, _gt.default)(versionToUpgradeNumber, kibanaVersionNumber) && !(0, _services2.differsOnlyInPatch)(versionToUpgradeNumber, kibanaVersionNumber)) {
    throw new _errors.AgentRequestInvalidError(`Cannot upgrade agent to ${versionToUpgradeNumber} because it is higher than the installed kibana version ${kibanaVersionNumber}`);
  }
  const kibanaMajorGt = (0, _major.default)(kibanaVersionNumber) > (0, _major.default)(versionToUpgradeNumber);
  const kibanaMajorEqMinorGte = (0, _major.default)(kibanaVersionNumber) === (0, _major.default)(versionToUpgradeNumber) && (0, _minor.default)(kibanaVersionNumber) >= (0, _minor.default)(versionToUpgradeNumber);

  // When force is enabled, only the major and minor versions are checked
  if (force && !(kibanaMajorGt || kibanaMajorEqMinorGte)) {
    throw new _errors.AgentRequestInvalidError(`Cannot force upgrade agent to ${versionToUpgradeNumber} because it does not satisfy the major and minor of the installed kibana version ${kibanaVersionNumber}`);
  }
};
exports.checkKibanaVersion = checkKibanaVersion;