"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.QueryColumns = exports.NOT_SUGGESTED_TYPES = void 0;
exports.getFromCommandHelper = getFromCommandHelper;
exports.getPolicyHelper = getPolicyHelper;
exports.getSourcesHelper = getSourcesHelper;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _esqlAst = require("@kbn/esql-ast");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const NOT_SUGGESTED_TYPES = exports.NOT_SUGGESTED_TYPES = ['unsupported'];

/**
 * Responsible for efficiently computing the list of columns generated by a specific ESQL query.
 */
class QueryColumns {
  /**
   * Retrieves from cache the columns for a given query, ignoring case.
   * @param query
   * @returns
   */
  static fromCache(query) {
    for (const key of QueryColumns.cache.keys()) {
      if (key.toLowerCase() === query.toLowerCase()) {
        return QueryColumns.cache.get(key);
      }
    }
    return undefined;
  }

  // once computed, the columns for the query will be cached here

  constructor(query, originalQueryText, resourceRetriever, options) {
    (0, _defineProperty2.default)(this, "fullQueryCacheKey", void 0);
    this.query = query;
    this.originalQueryText = originalQueryText;
    this.resourceRetriever = resourceRetriever;
    this.options = options;
    this.fullQueryCacheKey = _esqlAst.BasicPrettyPrinter.print(this.query, {
      skipHeader: true
    });
  }

  /**
   * Returns columns for this query, filtered by type and optionally ignoring some names.
   */
  async byType(expectedType = 'any', ignored = []) {
    const types = Array.isArray(expectedType) ? expectedType : [expectedType];
    await this.buildCache();
    const cachedFields = QueryColumns.fromCache(this.fullQueryCacheKey);
    return (cachedFields === null || cachedFields === void 0 ? void 0 : cachedFields.filter(({
      name,
      type
    }) => {
      const ts = Array.isArray(type) ? type : [type];
      return !ignored.includes(name) && (types[0] === 'any' || ts.some(t => types.includes(t))) && !NOT_SUGGESTED_TYPES.includes(type);
    })) || [];
  }

  /**
   * Returns a map of column name to column metadata for this query.
   */
  async asMap() {
    await this.buildCache();
    const cachedFields = QueryColumns.fromCache(this.fullQueryCacheKey);
    const cacheCopy = new Map();
    cachedFields === null || cachedFields === void 0 ? void 0 : cachedFields.forEach(field => cacheCopy.set(field.name, field));
    return cacheCopy;
  }

  /**
   * Ensures the cache is populated for all subqueries of this query context.
   */
  async buildCache() {
    if (!this.fullQueryCacheKey) return;
    const getFields = async queryToES => {
      var _this$options;
      if (!((_this$options = this.options) !== null && _this$options !== void 0 && _this$options.invalidateColumnsCache)) {
        const cached = QueryColumns.fromCache(queryToES);
        if (cached) {
          return cached;
        }
      }
      const fields = await (0, _helpers.getFieldsFromES)(queryToES, this.resourceRetriever);
      QueryColumns.cache.set(queryToES, fields);
      return fields;
    };
    const subqueries = [];
    for (let i = 0; i < this.query.commands.length; i++) {
      subqueries.push(_esqlAst.Builder.expression.query(this.query.commands.slice(0, i + 1)));
    }
    const getPolicies = async () => {
      var _await$this$resourceR, _this$resourceRetriev, _this$resourceRetriev2;
      const policies = (_await$this$resourceR = await ((_this$resourceRetriev = this.resourceRetriever) === null || _this$resourceRetriev === void 0 ? void 0 : (_this$resourceRetriev2 = _this$resourceRetriev.getPolicies) === null || _this$resourceRetriev2 === void 0 ? void 0 : _this$resourceRetriev2.call(_this$resourceRetriev))) !== null && _await$this$resourceR !== void 0 ? _await$this$resourceR : [];
      return new Map(policies.map(p => [p.name, p]));
    };
    for (const subquery of subqueries) {
      await this.cacheColumnsForQuery(subquery, getFields, getPolicies);
    }
  }

  /**
   * Caches the columns for a given query if not already cached.
   * @param query
   * @param fetchFields
   * @param getPolicies
   */
  async cacheColumnsForQuery(query, fetchFields, getPolicies) {
    var _this$options2, _QueryColumns$fromCac;
    let cacheKey;
    try {
      cacheKey = _esqlAst.BasicPrettyPrinter.print(query);
    } catch {
      // for some syntactically incorrect queries
      // the printer will throw. They're incorrect
      // anyways, so just move on — ANTLR errors will
      // be reported.
      return;
    }
    if (!((_this$options2 = this.options) !== null && _this$options2 !== void 0 && _this$options2.invalidateColumnsCache)) {
      const existsInCache = Boolean(QueryColumns.fromCache(cacheKey));
      if (existsInCache) {
        // this is already in the cache
        return;
      }
    }
    const queryBeforeCurrentCommand = _esqlAst.BasicPrettyPrinter.print({
      ...query,
      commands: query.commands.slice(0, -1)
    });
    const fieldsAvailableAfterPreviousCommand = (_QueryColumns$fromCac = QueryColumns.fromCache(queryBeforeCurrentCommand)) !== null && _QueryColumns$fromCac !== void 0 ? _QueryColumns$fromCac : [];
    const availableFields = await (0, _helpers.getCurrentQueryAvailableColumns)(query.commands, fieldsAvailableAfterPreviousCommand, fetchFields, getPolicies, this.originalQueryText);
    QueryColumns.cache.set(cacheKey, availableFields);
  }
}
exports.QueryColumns = QueryColumns;
(0, _defineProperty2.default)(QueryColumns, "cache", new Map());
function getPolicyHelper(resourceRetriever) {
  const getPolicies = async () => {
    var _resourceRetriever$ge;
    return (await (resourceRetriever === null || resourceRetriever === void 0 ? void 0 : (_resourceRetriever$ge = resourceRetriever.getPolicies) === null || _resourceRetriever$ge === void 0 ? void 0 : _resourceRetriever$ge.call(resourceRetriever))) || [];
  };
  return {
    getPolicies: async () => {
      const policies = await getPolicies();
      return policies;
    },
    getPolicyMetadata: async policyName => {
      const policies = await getPolicies();
      return policies.find(({
        name
      }) => name === policyName);
    }
  };
}
function getSourcesHelper(resourceRetriever) {
  return async () => {
    var _resourceRetriever$ge2;
    return (await (resourceRetriever === null || resourceRetriever === void 0 ? void 0 : (_resourceRetriever$ge2 = resourceRetriever.getSources) === null || _resourceRetriever$ge2 === void 0 ? void 0 : _resourceRetriever$ge2.call(resourceRetriever))) || [];
  };
}
async function getFromCommandHelper(resourceRetriever) {
  const getSources = getSourcesHelper(resourceRetriever);
  const sources = await (getSources === null || getSources === void 0 ? void 0 : getSources());
  const visibleSources = sources.filter(source => !source.hidden) || [];
  if (visibleSources.find(source => source.name.startsWith('logs'))) {
    return 'FROM logs*';
  }
  if (visibleSources.length > 0) {
    return `FROM ${visibleSources[0].name}`;
  }
  return '';
}