"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ParameterReplacer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _esqlAst = require("@kbn/esql-ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class ParameterReplacer {
  constructor(params) {
    (0, _defineProperty2.default)(this, "parametersMap", {});
    (0, _defineProperty2.default)(this, "positionalIndex", 0);
    this.parametersMap = this.buildParametersMap(params);
  }
  buildParametersMap(params) {
    const map = {};
    if (!params) {
      return map;
    }
    let index = 0;
    const list = Array.isArray(params) ? params : [params];
    for (const param of list) {
      if (typeof param === 'object' && param !== null) {
        for (const [key, value] of Object.entries(param)) {
          map[key.toLowerCase()] = value;
        }
      } else {
        map[String(index++)] = param;
      }
    }
    return map;
  }
  shouldReplaceNode(node) {
    if ((0, _esqlAst.isFunctionExpression)(node)) {
      return node.name.startsWith('?');
    }
    if ((0, _esqlAst.isColumn)(node)) {
      return node.args.some(arg => (0, _esqlAst.isParamLiteral)(arg));
    }
    return (0, _esqlAst.isParamLiteral)(node);
  }
  replace(node, parent) {
    if (!this.shouldReplaceNode(node)) {
      return node;
    }
    if ((0, _esqlAst.isFunctionExpression)(node)) {
      return this.replaceFunctionExpression(node);
    }
    if ((0, _esqlAst.isColumn)(node)) {
      return this.replaceColumnExpression(node);
    }
    return this.buildReplacementAstNode(node, parent);
  }
  replaceFunctionExpression(node) {
    const funcName = this.resolveFunctionName(node);
    return funcName ? _esqlAst.Builder.expression.func.node({
      ...node,
      name: funcName
    }) : node;
  }
  replaceColumnExpression(node, parent) {
    return _esqlAst.Builder.expression.column({
      ...node,
      args: node.args.flatMap(arg => (0, _esqlAst.isParamLiteral)(arg) ? this.buildReplacementAstNode(arg, parent) : arg)
    });
  }
  resolveParamValue(node) {
    switch (node.paramType) {
      case 'named':
      case 'positional':
        return this.parametersMap[String(node.value).toLowerCase()];
      default:
        return this.parametersMap[this.positionalIndex++];
    }
  }
  buildReplacementAstNode(node, parent) {
    if (!(0, _esqlAst.isParamLiteral)(node)) {
      return node;
    }
    const value = this.resolveParamValue(node);
    if (value === null || value === undefined) {
      return node;
    }
    if (node.paramKind === '??') {
      return String(value).split('.').map(name => _esqlAst.Builder.identifier({
        name
      }));
    }
    if (parent && node.paramKind === '?' && (0, _esqlAst.isFunctionExpression)(parent) && parent.subtype === 'variadic-call') {
      if (parent.name === 'bucket' && node.type === 'literal') {
        return _esqlAst.Builder.expression.literal.string(String(value), {
          unquoted: String(value).match(/^now\(\)/i) !== null
        });
      }
      return _esqlAst.Builder.identifier(String(value));
    }
    return this.buildLiteral(value);
  }
  resolveFunctionName(node) {
    const paramKey = node.name.replace(/^\?{1,2}/, '').toLocaleLowerCase();
    const paramValue = this.parametersMap[paramKey];
    return paramValue !== undefined ? String(paramValue) : node.name;
  }
  buildLiteral(value) {
    if (value === null) {
      return _esqlAst.Builder.expression.literal.nil();
    }
    switch (typeof value) {
      case 'number':
        return _esqlAst.Builder.expression.literal.numeric({
          value,
          literalType: Number.isInteger(value) ? 'integer' : 'double'
        });
      default:
        return _esqlAst.Builder.expression.literal.string(String(value));
    }
  }
}
exports.ParameterReplacer = ParameterReplacer;