"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toNavigationItems = void 0;
var _classnames = _interopRequireDefault(require("classnames"));
var _known_icons_mappings = require("./known_icons_mappings");
var _is_active_from_url = require("./utils/is_active_from_url");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const SKIP_WARNINGS = process.env.NODE_ENV === 'production';
/**
 * Converts the navigation tree definition and nav links into a format for new navigation.
 *
 * @remarks
 *
 * Structural Assumptions and Mapping
 *
 * - Root node (1st level) is used for the "logo" item and application branding
 * - 2nd level nodes are transformed into primary navigation items:
 *   - Accordion nodes are flattened (not supported) - their children become primary items
 *   - Nodes without links that aren't panel openers are treated as section dividers and not supported in new nav - their children are flattened
 *   - panelOpener nodes create flyout secondary navigation panels, they can't have links directly, but can have sections with links
 * - 3rd level is used for secondary navigation (children of panelOpener):
 *   - If all 3rd level items have links, they're treated as menu items and wrapped in a single section
 *   - If some don't have links, they're treated as section headers with their children becoming menu items
 * - Footer is limited to 5 items maximum (extras are dropped with warning)
 *
 * @param navigationTree
 * @param navLinks
 * @param activeNodes
 * @param panelStateManager - Manager for panel opener state
 */
const toNavigationItems = (navigationTree, activeNodes, panelStateManager) => {
  var _navigationTree$foote;
  // HACK: extract the logo, primary and footer nodes from the navigation tree
  let logoNode = null;
  let primaryNodes = [];
  let footerNodes = [];
  let deepestActiveItemId;
  let currentActiveItemIdLevel = -1;
  const isActive = navNode => (0, _is_active_from_url.isActiveFromUrl)(navNode.path, activeNodes, false);
  const maybeMarkActive = (navNode, level, parentNode) => {
    if (deepestActiveItemId == null || currentActiveItemIdLevel < level) {
      if (isActive(navNode)) {
        deepestActiveItemId = navNode.id;
        currentActiveItemIdLevel = level;
        if (parentNode !== null && parentNode !== void 0 && parentNode.id) {
          panelStateManager.setPanelLastActive(parentNode.id, navNode.id);
        }
      }
    }
  };
  const getTestSubj = (navNode, append = []) => {
    const {
      id,
      path,
      deepLink
    } = navNode;
    return (0, _classnames.default)(`nav-item`, `nav-item-${path}`, {
      [`nav-item-deepLinkId-${deepLink === null || deepLink === void 0 ? void 0 : deepLink.id}`]: !!deepLink,
      [`nav-item-id-${id}`]: id,
      [`nav-item-isActive`]: isActive(navNode)
    }, ...append);
  };
  if (navigationTree.body.length === 1) {
    const firstNode = navigationTree.body[0];
    if (!isRecentlyAccessedDefinition(firstNode)) {
      var _firstNode$children;
      primaryNodes = (_firstNode$children = firstNode.children) !== null && _firstNode$children !== void 0 ? _firstNode$children : [];
      const homeNodeIndex = primaryNodes.findIndex(node => node.renderAs === 'home');
      if (homeNodeIndex !== -1) {
        logoNode = primaryNodes[homeNodeIndex];
        primaryNodes = primaryNodes.filter((_, index) => index !== homeNodeIndex); // Remove the logo node from primary items
        maybeMarkActive(logoNode, 0);
      } else {
        warnOnce(`No "home" node found in primary nodes. There should be a logo node with solution logo, name and home page href. renderAs: "home" is expected.`);
      }
    }
  } else {
    warnOnce(`Navigation tree body has multiple root nodes. First level should have a single node. It is not used and shall be removed later after we fully migrate to the new nav.`);
  }
  if (((_navigationTree$foote = navigationTree.footer) === null || _navigationTree$foote === void 0 ? void 0 : _navigationTree$foote.length) === 1) {
    const firstNode = navigationTree.footer[0];
    if (!isRecentlyAccessedDefinition(firstNode)) {
      var _firstNode$children2;
      footerNodes = (_firstNode$children2 = firstNode.children) !== null && _firstNode$children2 !== void 0 ? _firstNode$children2 : [];
    }
  } else {
    warnOnce(`Navigation tree footer has multiple root nodes. Footer should have a single node for the footer links.`);
  }
  const logoItem = {
    href: warnIfMissing(logoNode, 'href', '/missing-href-😭'),
    iconType: getIcon(logoNode),
    id: warnIfMissing(logoNode, 'id', 'kibana'),
    label: warnIfMissing(logoNode, 'title', 'Kibana'),
    'data-test-subj': logoNode ? getTestSubj(logoNode, ['nav-item-home']) : undefined
  };
  const toMenuItem = navNode => {
    var _secondarySections;
    if (!navNode) return null;
    if (isRecentlyAccessedDefinition(navNode)) {
      warnOnce(`Recently accessed node "${navNode.id}" is not supported in the new navigation. Ignoring it.`);
      return null;
    }
    if (navNode.sideNavStatus === 'hidden' || navNode.sideNavVersion === 'v1') {
      return null;
    }

    // Flatten accordion items into a single level with links.
    // This is because the new navigation does not support accordion items.
    if (navNode.renderAs === 'accordion') {
      var _navNode$children;
      if (!((_navNode$children = navNode.children) !== null && _navNode$children !== void 0 && _navNode$children.length)) {
        warnOnce(`Accordion node "${navNode.id}" has no children. Ignoring it.`);
        return null;
      }
      const items = filterEmpty(navNode.children.flatMap(toMenuItem));
      warnOnce(`Accordion items are not supported in the new navigation. Flattening them "${items.map(i => i.id).join(', ')}" and dropping accordion node "${navNode.id}".`);
      return items;
    }

    // This was like a sub-section title without a link in the old navigation.
    // In the new navigation, just flatten it into its children, since we must have links in the primary items.
    if (navNode.renderAs !== 'panelOpener' && !navNode.href) {
      var _navNode$children2, _navNode$children3;
      warnOnce(`Navigation node "${navNode.id}${navNode.title ? ` (${navNode.title})` : ''}" is missing href and is not a panel opener. This node was likely used as a sub-section. Ignoring this node and flattening its children: ${(_navNode$children2 = navNode.children) === null || _navNode$children2 === void 0 ? void 0 : _navNode$children2.map(c => c.id).join(', ')}.`);
      if (!((_navNode$children3 = navNode.children) !== null && _navNode$children3 !== void 0 && _navNode$children3.length)) return null;
      return filterEmpty(navNode.children.flatMap(toMenuItem));
    }
    let secondarySections;

    // Helper function to filter out hidden and custom render items
    const filterValidSecondaryChildren = children => {
      return children.filter(child => child.sideNavStatus !== 'hidden' && child.sideNavVersion !== 'v1').filter(child => {
        const isCustomRender = typeof child.renderItem === 'function';
        if (isCustomRender) {
          warnOnce(`Custom renderItem is not supported in the new navigation. Ignoring it for node "${child.id}".`);
        }
        return !isCustomRender;
      });
    };

    // Helper function to convert a node to a secondary menu item
    const createSecondaryMenuItem = child => {
      warnUnsupportedNavNodeOptions(child);
      maybeMarkActive(child, 2, navNode);
      return {
        id: child.id,
        label: warnIfMissing(child, 'title', 'Missing Title 😭'),
        href: warnIfMissing(child, 'href', 'Missing Href 😭'),
        isExternal: child.isExternalLink,
        'data-test-subj': getTestSubj(child),
        badgeType: child.badgeTypeV2
      };
    };
    if (navNode.renderAs === 'panelOpener') {
      var _navNode$children4;
      if (!((_navNode$children4 = navNode.children) !== null && _navNode$children4 !== void 0 && _navNode$children4.length)) {
        warnOnce(`Panel opener node "${navNode.id}" has no children. Ignoring it.`);
        return null;
      }
      const noSubSections = navNode.children.every(l => l.href);
      if (noSubSections) {
        var _navNode$children5;
        warnOnce(`Panel opener node "${navNode.id}" should contain panel sections, not direct links. Flattening links "${(_navNode$children5 = navNode.children) === null || _navNode$children5 === void 0 ? void 0 : _navNode$children5.map(c => c.id).join(', ')}" into secondary items and creating a placeholder section for these links.`);

        // If all children have hrefs, we can treat them as secondary items
        const validChildren = filterValidSecondaryChildren(navNode.children);
        secondarySections = [{
          id: `${navNode.id}-section`,
          items: validChildren.map(createSecondaryMenuItem)
        }];
      } else {
        // Otherwise, we need to create sections for each child
        secondarySections = filterEmpty(navNode.children.map(child => {
          var _child$children;
          if (child.sideNavStatus === 'hidden' || child.sideNavVersion === 'v1') return null;
          if (!((_child$children = child.children) !== null && _child$children !== void 0 && _child$children.length)) return null;
          warnUnsupportedNavNodeOptions(child);
          const validChildren = filterValidSecondaryChildren(child.children);
          const secondaryItems = validChildren.map(createSecondaryMenuItem);
          if (child.href) {
            warnOnce(`Secondary menu item node "${child.id}" has a href "${child.href}", but it should not. We're using it as a section title that doesn't have a link.`);
          }
          return {
            id: child.id,
            label: child.title,
            items: secondaryItems
          };
        })).filter(section => section.items.length > 0); // Filter out empty sections;
      }

      // If after all filtering there are no sections, we skip this menu item
      if (secondarySections.length === 0) {
        return null;
      }
    }
    warnUnsupportedNavNodeOptions(navNode);

    // for primary menu items there should always be a href
    // if it's a panel opener, we use the last opened panel or the first link inside the section as the href
    // if there are no sections, we use the href directly
    const itemHref = (_secondarySections = secondarySections) !== null && _secondarySections !== void 0 && _secondarySections.length ? getPanelOpenerHref(navNode, secondarySections, panelStateManager) : warnIfMissing(navNode, 'href', 'missing-href-😭');
    maybeMarkActive(navNode, 1);
    return {
      id: navNode.id,
      label: warnIfMissing(navNode, 'title', 'Missing Title 😭'),
      iconType: getIcon(navNode),
      href: itemHref,
      sections: secondarySections,
      'data-test-subj': getTestSubj(navNode),
      badgeType: navNode.badgeTypeV2
    };
  };
  const primaryItems = filterEmpty(primaryNodes.flatMap(toMenuItem));
  const footerItems = filterEmpty(footerNodes.flatMap(toMenuItem));
  if (footerItems.length > 5) {
    warnOnce(`Footer items should not exceed 5. Found ${footerItems.length}. Only the first 5 will be displayed. Dropped items: ${footerItems.slice(5).map(item => item.id).join(', ')}`);
  }
  if (!SKIP_WARNINGS) {
    warnAboutDuplicateIds(logoItem, primaryItems, footerItems);
    warnAboutDuplicateIcons(logoItem, primaryItems, footerItems);
  }
  return {
    logoItem,
    navItems: {
      primaryItems,
      footerItems
    },
    activeItemId: deepestActiveItemId,
    solutionId: navigationTree.id
  };
};

// =====================
// Utilities & Helpers
// =====================
exports.toNavigationItems = toNavigationItems;
function warnIfMissing(obj, key, fallback) {
  const keys = Array.isArray(key) ? key : [key];

  // Helper function to create warning message
  const createWarningMessage = reason => `Navigation item${obj !== null && obj !== void 0 && obj.id ? ` "${obj.id}"` : ''} ${reason}. Using fallback value: "${String(fallback)}".`;
  if (!obj) {
    warnOnce(createWarningMessage(`is missing`));
    return fallback;
  }

  // Try each key in order until we find a value
  for (const k of keys) {
    const value = obj[k];
    if (value !== undefined && value !== null) {
      return value;
    }
  }

  // None of the keys had values, warn and use fallback
  const missingKeysMessage = keys.length === 1 ? `is missing a "${String(keys[0])}"` : `is missing all of "${keys.join(', ')}"`;
  warnOnce(createWarningMessage(missingKeysMessage));
  return fallback;
}
const warnedMessages = new Set();
let lastWarning = '';
function warnOnce(message) {
  if (SKIP_WARNINGS) return;
  if (!warnedMessages.has(message)) {
    warnedMessages.add(message);
  }
  setTimeout(() => {
    const header = '\n=== Navigation Warnings ===\n';
    const warning = header + Array.from(warnedMessages.values()).map(msg => `• ${msg}`).join('\n');
    if (warning !== lastWarning) {
      // eslint-disable-next-line no-console
      console.warn(warning);
      lastWarning = warning;
    }
  }, 0);
}
function warnUnsupportedNavNodeOptions(navNode) {
  if (navNode.spaceBefore) {
    warnOnce(`Space before is not supported in the new navigation. Ignoring it for node "${navNode.id}".`);
  }
  if (navNode.badgeOptions || navNode.withBadge) {
    warnOnce(`Badge options are not supported in the new navigation. Ignoring them for node "${navNode.id}".`);
  }
  if (navNode.openInNewTab) {
    warnOnce(`Open in new tab is not supported in the new navigation. Ignoring it for node "${navNode.id}".`);
  }
  if (navNode.renderItem) {
    warnOnce(`Custom renderItem is not supported in the new navigation. Ignoring it for node "${navNode.id}".`);
  }
}
const isRecentlyAccessedDefinition = item => {
  return item.type === 'recentlyAccessed';
};
const filterEmpty = arr => arr.filter(item => item !== null && item !== undefined);

/**
 * Generic function to detect and warn about duplicate values in navigation items.
 * @param values - Array of values to check for duplicates
 * @param formatWarning - Function to format the warning message for duplicates
 */
function warnAboutDuplicates(values, formatWarning) {
  const valueGroups = new Map();
  values.forEach(value => {
    valueGroups.set(value, (valueGroups.get(value) || 0) + 1);
  });
  valueGroups.forEach((count, value) => {
    if (count > 1) {
      warnOnce(formatWarning(value, count));
    }
  });
}
function warnAboutDuplicateIcons(logoItem, primaryItems, footerItems) {
  if (SKIP_WARNINGS) return;
  // Collect all items with icons (only logo + primary items, excluding fallback)
  const icons = [logoItem, ...primaryItems, ...footerItems].filter(item => item.iconType && item.iconType !== FALLBACK_ICON && typeof item.iconType === 'string').map(item => String(item.iconType));
  warnAboutDuplicates(icons, icon => `Icon "${icon}" is used by multiple navigation items. Consider using unique icons for better UX.`);
}
function warnAboutDuplicateIds(logoItem, primaryItems, footerItems) {
  if (SKIP_WARNINGS) return;
  // Collect all IDs from all items, including secondary menu items
  let allIds = [logoItem.id];

  // Helper to extract IDs from menu items including their secondary sections
  const collectIds = items => {
    items.forEach(item => {
      allIds.push(item.id);
      if (item.sections) {
        item.sections.forEach(section => {
          allIds.push(section.id);
          section.items.forEach(secondaryItem => {
            allIds.push(secondaryItem.id);
          });
        });
      }
    });
  };
  collectIds(primaryItems);
  collectIds(footerItems);
  allIds = allIds.filter(id => !(id !== null && id !== void 0 && id.startsWith('node-'))); // Filter out auto-generated IDs

  warnAboutDuplicates(allIds, (id, count) => `ID "${id}" is used ${count} times in navigation items. Each navigation item must have a unique ID.`);
}
const FALLBACK_ICON = 'broom';
/**
 * Finds an item href based on the last active item history for a panel opener.
 * @param panelId - The panel opener node id
 * @param sections - The secondary menu sections to search in
 * @param panelStateManager - Manager for panel opener state
 * @returns The href of the last active item, or undefined if not found
 */
const findItemByLastActive = (panelId, sections, panelStateManager) => {
  const lastActiveItemId = panelStateManager.getPanelLastActive(panelId);
  if (!lastActiveItemId) return undefined;
  for (const section of sections) {
    const foundItem = section.items.find(item => item.id === lastActiveItemId);
    if (foundItem !== null && foundItem !== void 0 && foundItem.href) return foundItem.href;
  }
  return undefined;
};

/**
 * Finds the first available href from secondary menu sections.
 * @param sections - The secondary menu sections to search in
 * @returns The first available href, or undefined if none found
 */
const findFirstAvailableHref = sections => {
  for (const section of sections) {
    for (const item of section.items) {
      if (item.href && !item.isExternal) {
        return item.href;
      }
    }
  }
};

/**
 * Determines the appropriate href for a panel opener node.
 * Uses last active item history first, then falls back to first available href.
 * @param navNode - The navigation node (panel opener)
 * @param secondarySections - The secondary menu sections
 * @param panelStateManager - Manager for panel opener state
 * @returns The determined href for the panel opener
 */
const getPanelOpenerHref = (navNode, secondarySections, panelStateManager) => {
  // Try to use last active item first
  const lastActiveHref = findItemByLastActive(navNode.id, secondarySections, panelStateManager);
  if (lastActiveHref) return lastActiveHref;

  // Fall back to first available href
  const firstAvailableHref = findFirstAvailableHref(secondarySections);

  // Warn if panel opener has its own href (which it shouldn't)
  if (navNode.href) {
    warnOnce(`Panel opener node "${navNode.id}" has a href "${navNode.href}", but it should not. We're using it as a panel opener that contains sections with links and we use the first link inside the section as the href ${firstAvailableHref !== null && firstAvailableHref !== void 0 ? firstAvailableHref : 'missing-href-😭'}.`);
  }
  return firstAvailableHref !== null && firstAvailableHref !== void 0 ? firstAvailableHref : 'missing-href-😭';
};
const getIcon = node => {
  var _node$deepLink, _node$deepLink2;
  if (node !== null && node !== void 0 && node.iconV2) {
    return node.iconV2;
  }
  if (node !== null && node !== void 0 && node.icon) {
    return node.icon;
  }
  if (node && _known_icons_mappings.AppDeepLinkIdToIcon[node.id]) {
    return _known_icons_mappings.AppDeepLinkIdToIcon[node.id];
  }
  if (node !== null && node !== void 0 && (_node$deepLink = node.deepLink) !== null && _node$deepLink !== void 0 && _node$deepLink.euiIconType) {
    return node.deepLink.euiIconType;
  }
  if (node !== null && node !== void 0 && (_node$deepLink2 = node.deepLink) !== null && _node$deepLink2 !== void 0 && _node$deepLink2.icon) {
    return node.deepLink.icon;
  }
  warnOnce(`No icon found for node "${node === null || node === void 0 ? void 0 : node.id}". Expected iconV2, icon, deepLink.euiIconType, deepLink.icon or a known deep link id. Using fallback icon "broom".`);
  return FALLBACK_ICON;
};