"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerBulkDeleteRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const registerBulkDeleteRoute = (router, {
  config,
  coreUsageData,
  logger,
  access,
  deprecationInfo
}) => {
  const {
    allowHttpApiAccess
  } = config;
  router.post({
    path: '/_bulk_delete',
    options: {
      summary: `Delete saved objects`,
      tags: ['oas-tag:saved objects'],
      access,
      deprecated: deprecationInfo
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates authorization to the Saved Objects Client'
      }
    },
    validate: {
      body: _configSchema.schema.arrayOf(_configSchema.schema.object({
        type: _configSchema.schema.string(),
        id: _configSchema.schema.string()
      })),
      query: _configSchema.schema.object({
        force: _configSchema.schema.maybe(_configSchema.schema.boolean())
      })
    }
  }, (0, _utils.catchAndReturnBoomErrors)(async (context, request, response) => {
    (0, _utils.logWarnOnExternalRequest)({
      method: 'post',
      path: '/api/saved_objects/_bulk_delete',
      request,
      logger
    });
    const {
      force
    } = request.query;
    const types = [...new Set(request.body.map(({
      type
    }) => type))];
    const usageStatsClient = coreUsageData.getClient();
    usageStatsClient.incrementSavedObjectsBulkDelete({
      request,
      types
    }).catch(() => {});
    const {
      savedObjects
    } = await context.core;
    if (!allowHttpApiAccess) {
      (0, _utils.throwIfAnyTypeNotVisibleByAPI)(types, savedObjects.typeRegistry);
    }
    const statuses = await savedObjects.client.bulkDelete(request.body, {
      force
    });
    return response.ok({
      body: statuses
    });
  }));
};
exports.registerBulkDeleteRoute = registerBulkDeleteRoute;