"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createVisualizationTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _joiToJson = _interopRequireDefault(require("joi-to-json"));
var _metric = require("@kbn/lens-embeddable-utils/config_builder/schema/charts/metric");
var _onechatServer = require("@kbn/onechat-server");
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _guess_chart_type = require("./guess_chart_type");
var _graph_lens = require("./graph_lens");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const metricSchema = (0, _joiToJson.default)(_metric.esqlMetricState.getSchema());
const createVisualizationSchema = _zod.z.object({
  query: _zod.z.string().describe('A natural language query describing the desired visualization.'),
  existingConfig: _zod.z.string().optional().describe('An existing visualization configuration to modify.'),
  chartType: _zod.z.enum([_tool_result.SupportedChartType.Metric, _tool_result.SupportedChartType.Map]).optional().describe('(optional) The type of chart to create as indicated by the user. If not provided, the LLM will suggest the best chart type.'),
  esql: _zod.z.string().optional().describe('(optional) An ES|QL query. If not provided, tool with automatically generate the query. Only pass ES|QL queries from reliable sources (other tool calls or the user) and NEVER invent queries directly.')
});
const createVisualizationTool = () => {
  return {
    id: _onechatCommon.platformCoreTools.createVisualization,
    type: _onechatCommon.ToolType.builtin,
    description: `Create a visualization configuration based on a natural language description.

This tool will:
1. Determine the best chart type if not specified (from: ${Object.values(_tool_result.SupportedChartType).join(', ')})
2. Generate an ES|QL query if not provided
3. Generate a valid visualization configuration`,
    schema: createVisualizationSchema,
    availability: {
      cacheMode: 'space',
      handler: async ({
        uiSettings
      }) => {
        const enabled = await uiSettings.get(_managementSettingsIds.AGENT_BUILDER_DASHBOARD_TOOLS_SETTING_ID);
        return {
          status: enabled ? 'available' : 'unavailable'
        };
      }
    },
    tags: [],
    handler: async ({
      query: nlQuery,
      chartType,
      esql,
      existingConfig
    }, {
      esClient,
      modelProvider,
      logger,
      events
    }) => {
      try {
        // Step 1: Determine chart type if not provided
        let selectedChartType = chartType || _tool_result.SupportedChartType.Metric;
        const parsedExistingConfig = existingConfig ? JSON.parse(existingConfig) : null;
        if (!chartType) {
          logger.debug('Chart type not provided, using LLM to suggest one');
          selectedChartType = await (0, _guess_chart_type.guessChartType)(modelProvider, parsedExistingConfig === null || parsedExistingConfig === void 0 ? void 0 : parsedExistingConfig.type, nlQuery);
        }

        // Step 2: Generate visualization configuration using langgraph with validation retry
        const model = await modelProvider.getDefaultModel();
        const schema = metricSchema;

        // Create and invoke the validation retry graph
        const graph = (0, _graph_lens.createVisualizationGraph)(model, logger, events, esClient);
        const finalState = await graph.invoke({
          nlQuery,
          chartType: selectedChartType,
          schema,
          existingConfig,
          parsedExistingConfig,
          esqlQuery: esql || '',
          currentAttempt: 0,
          actions: [],
          validatedConfig: null,
          error: null
        });
        const {
          validatedConfig,
          error,
          currentAttempt,
          esqlQuery
        } = finalState;
        if (!validatedConfig) {
          throw new Error(`Failed to generate valid configuration after ${currentAttempt} attempts. Last error: ${error || 'Unknown error'}`);
        }
        return {
          results: [{
            type: _tool_result.ToolResultType.visualization,
            tool_result_id: (0, _onechatServer.getToolResultId)(),
            data: {
              query: nlQuery,
              visualization: validatedConfig,
              chart_type: selectedChartType,
              esql: esqlQuery
            }
          }]
        };
      } catch (error) {
        logger.error(`Error in create_visualization tool: ${error.message}`);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to create visualization: ${error.message}`,
              metadata: {
                nlQuery,
                esql,
                chartType
              }
            }
          }]
        };
      }
    }
  };
};
exports.createVisualizationTool = createVisualizationTool;