"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.browserToolsToLangchain = browserToolsToLangchain;
exports.createBrowserToolAdapter = createBrowserToolAdapter;
var _uuid = require("uuid");
var _tools = require("@langchain/core/tools");
var _onechatCommon = require("@kbn/onechat-common");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Create a browser tool adapter that registers browser tools as LLM tools
 */
function createBrowserToolAdapter({
  browserTool
}) {
  return (0, _tools.tool)(async () => {
    const callId = (0, _uuid.v4)();
    const result = {
      results: [{
        type: _onechatCommon.ToolResultType.other,
        data: {
          message: `Browser tool '${browserTool.id}' will be executed on client`,
          callId,
          executeOnClient: true
        }
      }]
    };
    return [JSON.stringify(result), result];
  }, {
    name: (0, _langchain.sanitizeToolId)(`browser_${browserTool.id}`),
    description: browserTool.description,
    schema: browserTool.schema,
    responseFormat: 'content_and_artifact'
  });
}

/**
 * Convert browser API tools to LLM-compatible tool definitions
 */
function browserToolsToLangchain({
  browserApiTools
}) {
  const tools = browserApiTools.map(tool => {
    return createBrowserToolAdapter({
      browserTool: tool
    });
  });
  const idMappings = new Map();
  browserApiTools.forEach(tool => {
    const toolId = (0, _langchain.sanitizeToolId)(`browser_${tool.id}`);
    idMappings.set(toolId, toolId);
  });
  return {
    tools,
    idMappings
  };
}