"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runDefaultAgentMode = void 0;
var _uuid = require("uuid");
var _rxjs = require("rxjs");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
var _utils = require("../utils");
var _capabilities = require("../utils/capabilities");
var _configuration = require("../utils/configuration");
var _graph = require("./graph");
var _convert_graph_events = require("./convert_graph_events");
var _browser_tool_adapter = require("../../../tools/browser_tool_adapter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const chatAgentGraphName = 'default-onechat-agent';
/**
 * Create the handler function for the default onechat agent.
 */
const runDefaultAgentMode = async ({
  nextInput,
  conversation,
  agentConfiguration,
  capabilities,
  runId = (0, _uuid.v4)(),
  agentId,
  abortSignal,
  browserApiTools,
  startTime = new Date()
}, {
  logger,
  request,
  modelProvider,
  toolProvider,
  attachments,
  events
}) => {
  var _conversation$rounds;
  const model = await modelProvider.getDefaultModel();
  const resolvedCapabilities = (0, _capabilities.resolveCapabilities)(capabilities);
  const resolvedConfiguration = (0, _configuration.resolveConfiguration)(agentConfiguration);
  logger.debug(`Running chat agent with connector: ${model.connector.name}, runId: ${runId}`);
  const manualEvents$ = new _rxjs.Subject();
  const eventEmitter = event => {
    manualEvents$.next(event);
  };
  const selectedTools = await (0, _utils.selectTools)({
    input: nextInput,
    conversation,
    toolProvider,
    agentConfiguration,
    attachmentsService: attachments,
    request
  });
  const {
    tools: langchainTools,
    idMappings: toolIdMapping
  } = await (0, _langchain.toolsToLangchain)({
    tools: selectedTools,
    logger,
    request,
    sendEvent: eventEmitter
  });
  let browserLangchainTools = [];
  let browserIdMappings = new Map();
  if (browserApiTools && browserApiTools.length > 0) {
    const browserToolResult = (0, _browser_tool_adapter.browserToolsToLangchain)({
      browserApiTools
    });
    browserLangchainTools = browserToolResult.tools;
    browserIdMappings = browserToolResult.idMappings;
  }
  const allTools = [...langchainTools, ...browserLangchainTools];
  const allToolIdMappings = new Map([...toolIdMapping, ...browserIdMappings]);
  const cycleLimit = 10;
  const graphRecursionLimit = getRecursionLimit(cycleLimit);
  const processedConversation = await (0, _utils.prepareConversation)({
    nextInput,
    previousRounds: (_conversation$rounds = conversation === null || conversation === void 0 ? void 0 : conversation.rounds) !== null && _conversation$rounds !== void 0 ? _conversation$rounds : [],
    attachmentsService: attachments
  });
  const initialMessages = (0, _utils.conversationToLangchainMessages)({
    conversation: processedConversation
  });
  const agentGraph = (0, _graph.createAgentGraph)({
    logger,
    events: {
      emit: eventEmitter
    },
    chatModel: model.chatModel,
    tools: allTools,
    configuration: resolvedConfiguration,
    capabilities: resolvedCapabilities
  });
  logger.debug(`Running chat agent with graph: ${chatAgentGraphName}, runId: ${runId}`);
  const eventStream = agentGraph.streamEvents({
    initialMessages,
    cycleLimit
  }, {
    version: 'v2',
    signal: abortSignal,
    runName: chatAgentGraphName,
    metadata: {
      graphName: chatAgentGraphName,
      agentId
    },
    recursionLimit: graphRecursionLimit,
    callbacks: []
  });
  const graphEvents$ = (0, _rxjs.from)(eventStream).pipe((0, _rxjs.filter)(_langchain.isStreamEvent), (0, _convert_graph_events.convertGraphEvents)({
    graphName: chatAgentGraphName,
    toolIdMapping: allToolIdMappings,
    logger,
    startTime
  }), (0, _rxjs.finalize)(() => manualEvents$.complete()));
  const processedInput = {
    message: processedConversation.nextInput.message,
    attachments: processedConversation.nextInput.attachments.map(a => a.attachment)
  };
  const events$ = (0, _rxjs.merge)(graphEvents$, manualEvents$).pipe((0, _utils.addRoundCompleteEvent)({
    userInput: processedInput,
    startTime,
    modelProvider
  }), (0, _rxjs.shareReplay)());
  events$.subscribe({
    next: event => events.emit(event),
    error: () => {
      // error will be handled by function return, we just need to trap here
    }
  });
  const round = await (0, _utils.extractRound)(events$);
  return {
    round
  };
};
exports.runDefaultAgentMode = runDefaultAgentMode;
const getRecursionLimit = cycleLimit => {
  // langchain's recursionLimit is basically the number of nodes we can traverse before hitting a recursion limit error
  // we have two steps per cycle (agent node + tool call node), and then a few other steps (prepare + answering), and some extra buffer
  return cycleLimit * 2 + 8;
};