"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isToolNotFoundError = exports.isRequestAbortedError = exports.isOnechatError = exports.isInternalError = exports.isConversationNotFoundError = exports.isBadRequestError = exports.isAgentNotFoundError = exports.isAgentExecutionError = exports.createToolNotFoundError = exports.createRequestAbortedError = exports.createOnechatError = exports.createInternalError = exports.createConversationNotFoundError = exports.createBadRequestError = exports.createAgentNotFoundError = exports.createAgentExecutionError = exports.OnechatErrorUtils = exports.OnechatErrorCode = void 0;
var _sseUtils = require("@kbn/sse-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/**
 * Code to identify onechat errors
 */
let OnechatErrorCode = exports.OnechatErrorCode = /*#__PURE__*/function (OnechatErrorCode) {
  OnechatErrorCode["internalError"] = "internalError";
  OnechatErrorCode["badRequest"] = "badRequest";
  OnechatErrorCode["toolNotFound"] = "toolNotFound";
  OnechatErrorCode["agentNotFound"] = "agentNotFound";
  OnechatErrorCode["conversationNotFound"] = "conversationNotFound";
  OnechatErrorCode["agentExecutionError"] = "agentExecutionError";
  OnechatErrorCode["requestAborted"] = "requestAborted";
  return OnechatErrorCode;
}({});
const OnechatError = _sseUtils.ServerSentEventError;

/**
 * Base error class used for all onechat errors.
 */

const isOnechatError = err => {
  return err instanceof OnechatError;
};
exports.isOnechatError = isOnechatError;
const createOnechatError = (errorCode, message, meta) => {
  return new OnechatError(errorCode, message, meta !== null && meta !== void 0 ? meta : {});
};

/**
 * Represents an internal error
 */
exports.createOnechatError = createOnechatError;
/**
 * Checks if the given error is a {@link OnechatInternalError}
 */
const isInternalError = err => {
  return isOnechatError(err) && err.code === OnechatErrorCode.internalError;
};
exports.isInternalError = isInternalError;
const createInternalError = (message, meta) => {
  return new OnechatError(OnechatErrorCode.internalError, message, meta !== null && meta !== void 0 ? meta : {});
};

/**
 * Represents a generic bad request error
 */
exports.createInternalError = createInternalError;
/**
 * Checks if the given error is a {@link OnechatInternalError}
 */
const isBadRequestError = err => {
  return isOnechatError(err) && err.code === OnechatErrorCode.badRequest;
};
exports.isBadRequestError = isBadRequestError;
const createBadRequestError = (message, meta = {}) => {
  return new OnechatError(OnechatErrorCode.badRequest, message, {
    ...meta,
    statusCode: 400
  });
};

/**
 * Error thrown when trying to retrieve or execute a tool not present or available in the current context.
 */
exports.createBadRequestError = createBadRequestError;
/**
 * Checks if the given error is a {@link OnechatToolNotFoundError}
 */
const isToolNotFoundError = err => {
  return isOnechatError(err) && err.code === OnechatErrorCode.toolNotFound;
};
exports.isToolNotFoundError = isToolNotFoundError;
const createToolNotFoundError = ({
  toolId,
  customMessage,
  meta = {}
}) => {
  return new OnechatError(OnechatErrorCode.toolNotFound, customMessage !== null && customMessage !== void 0 ? customMessage : `Tool ${toolId} not found`, {
    ...meta,
    toolId,
    statusCode: 404
  });
};

/**
 * Error thrown when trying to retrieve or execute a tool not present or available in the current context.
 */
exports.createToolNotFoundError = createToolNotFoundError;
/**
 * Checks if the given error is a {@link OnechatInternalError}
 */
const isAgentNotFoundError = err => {
  return isOnechatError(err) && err.code === OnechatErrorCode.agentNotFound;
};
exports.isAgentNotFoundError = isAgentNotFoundError;
const createAgentNotFoundError = ({
  agentId,
  customMessage,
  meta = {}
}) => {
  return new OnechatError(OnechatErrorCode.agentNotFound, customMessage !== null && customMessage !== void 0 ? customMessage : `Agent ${agentId} not found`, {
    ...meta,
    agentId,
    statusCode: 404
  });
};

/**
 * Error thrown when trying to retrieve or execute a tool not present or available in the current context.
 */
exports.createAgentNotFoundError = createAgentNotFoundError;
/**
 * Checks if the given error is a {@link OnechatConversationNotFoundError}
 */
const isConversationNotFoundError = err => {
  return isOnechatError(err) && err.code === OnechatErrorCode.conversationNotFound;
};
exports.isConversationNotFoundError = isConversationNotFoundError;
const createConversationNotFoundError = ({
  conversationId,
  customMessage,
  meta = {}
}) => {
  return new OnechatError(OnechatErrorCode.conversationNotFound, customMessage !== null && customMessage !== void 0 ? customMessage : `Conversation ${conversationId} not found`, {
    ...meta,
    conversationId,
    statusCode: 404
  });
};

/**
 * Represents an internal error
 */
exports.createConversationNotFoundError = createConversationNotFoundError;
/**
 * Checks if the given error is a {@link OnechatRequestAbortedError}
 */
const isRequestAbortedError = err => {
  return isOnechatError(err) && err.code === OnechatErrorCode.requestAborted;
};
exports.isRequestAbortedError = isRequestAbortedError;
const createRequestAbortedError = (message, meta) => {
  return new OnechatError(OnechatErrorCode.requestAborted, message, meta !== null && meta !== void 0 ? meta : {});
};

/**
 * Represents an error related to agent execution
 */
exports.createRequestAbortedError = createRequestAbortedError;
/**
 * Checks if the given error is a {@link OnechatInternalError}
 */
const isAgentExecutionError = err => {
  return isOnechatError(err) && err.code === OnechatErrorCode.agentExecutionError;
};
exports.isAgentExecutionError = isAgentExecutionError;
const createAgentExecutionError = (message, code, meta) => {
  return new OnechatError(OnechatErrorCode.agentExecutionError, message, {
    ...meta,
    errCode: code
  });
};

/**
 * Global utility exposing all error utilities from a single export.
 */
exports.createAgentExecutionError = createAgentExecutionError;
const OnechatErrorUtils = exports.OnechatErrorUtils = {
  isOnechatError,
  isInternalError,
  isToolNotFoundError,
  isAgentNotFoundError,
  isConversationNotFoundError,
  isAgentExecutionError,
  createInternalError,
  createToolNotFoundError,
  createAgentNotFoundError,
  createConversationNotFoundError,
  createAgentExecutionError
};