"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.canDeployCustomPackageAsAgentlessOrThrow = canDeployCustomPackageAsAgentlessOrThrow;
exports.canUseMultipleAgentPolicies = canUseMultipleAgentPolicies;
exports.canUseOutputForIntegration = canUseOutputForIntegration;
exports.mapPackagePolicySavedObjectToPackagePolicy = void 0;
exports.preflightCheckPackagePolicy = preflightCheckPackagePolicy;
var _constants = require("../../../common/constants");
var _output_helpers = require("../../../common/services/output_helpers");
var _ = require("..");
var _errors = require("../../errors");
var _license = require("../license");
var _output = require("../output");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const mapPackagePolicySavedObjectToPackagePolicy = ({
  id,
  version,
  attributes,
  namespaces
}) => {
  const {
    bump_agent_policy_revision: bumpAgentPolicyRevision,
    latest_revision: latestRevision,
    ...restAttributes
  } = attributes;
  return {
    id,
    version,
    ...(namespaces ? {
      spaceIds: namespaces
    } : {}),
    ...restAttributes
  };
};
exports.mapPackagePolicySavedObjectToPackagePolicy = mapPackagePolicySavedObjectToPackagePolicy;
async function preflightCheckPackagePolicy(soClient, packagePolicy, packageInfo) {
  // Package policies cannot be created for content type packages
  if ((packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.type) === 'content') {
    throw new _errors.PackagePolicyContentPackageError('Cannot create policy for content only packages');
  }

  // If package policy has multiple agent policies IDs, or no agent policies (orphaned integration policy)
  // check if user can use multiple agent policies feature
  const {
    canUseReusablePolicies,
    errorMessage: canUseMultipleAgentPoliciesErrorMessage
  } = canUseMultipleAgentPolicies();
  if (!canUseReusablePolicies && packagePolicy.policy_ids.length !== 1) {
    throw new _errors.PackagePolicyMultipleAgentPoliciesError(canUseMultipleAgentPoliciesErrorMessage);
  }

  // If package policy has an output_id, check if it can be used
  if (packagePolicy.output_id && packagePolicy.package) {
    const {
      canUseOutputForIntegrationResult,
      errorMessage: outputForIntegrationErrorMessage
    } = await canUseOutputForIntegration(soClient, packagePolicy);
    if (!canUseOutputForIntegrationResult && outputForIntegrationErrorMessage) {
      throw new _errors.PackagePolicyOutputError(outputForIntegrationErrorMessage);
    }
  }
}
function canUseMultipleAgentPolicies() {
  const hasEnterpriseLicence = _license.licenseService.hasAtLeast(_constants.LICENCE_FOR_MULTIPLE_AGENT_POLICIES);
  return {
    canUseReusablePolicies: hasEnterpriseLicence,
    errorMessage: !hasEnterpriseLicence ? 'Reusable integration policies are only available with an Enterprise license' : 'Reusable integration policies are not supported'
  };
}
async function canUseOutputForIntegration(soClient, packagePolicy) {
  var _packagePolicy$packag;
  const outputId = packagePolicy.output_id;
  const packageName = (_packagePolicy$packag = packagePolicy.package) === null || _packagePolicy$packag === void 0 ? void 0 : _packagePolicy$packag.name;
  if (outputId) {
    const hasAllowedLicense = _license.licenseService.hasAtLeast(_constants.LICENCE_FOR_OUTPUT_PER_INTEGRATION);
    if (!hasAllowedLicense) {
      return {
        canUseOutputForIntegrationResult: false,
        errorMessage: `Output per integration is only available with an ${_constants.LICENCE_FOR_OUTPUT_PER_INTEGRATION} license`
      };
    }
    const allowedOutputTypesForIntegration = (0, _output_helpers.getAllowedOutputTypesForIntegration)(packageName);
    const allowedOutputTypesForPackagePolicy = (0, _output_helpers.getAllowedOutputTypesForPackagePolicy)(packagePolicy);
    const allowedOutputTypes = allowedOutputTypesForIntegration.filter(type => allowedOutputTypesForPackagePolicy.includes(type));
    const output = await _output.outputService.get(outputId);
    if (!allowedOutputTypes.includes(output.type)) {
      return {
        canUseOutputForIntegrationResult: false,
        errorMessage: `Output type "${output.type}" is not usable with package "${packageName}"`
      };
    }
  }
  return {
    canUseOutputForIntegrationResult: true,
    errorMessage: null
  };
}
function canDeployCustomPackageAsAgentlessOrThrow(packagePolicy, packageInfo) {
  var _packageInfo$savedObj, _appContextService$ge, _appContextService$ge2, _appContextService$ge3;
  const installSource = packageInfo && 'savedObject' in packageInfo && ((_packageInfo$savedObj = packageInfo.savedObject) === null || _packageInfo$savedObj === void 0 ? void 0 : _packageInfo$savedObj.attributes.install_source);
  const isCustom = installSource === 'custom' || installSource === 'upload';
  const isCustomAgentlessAllowed = (_appContextService$ge = _.appContextService.getConfig()) === null || _appContextService$ge === void 0 ? void 0 : (_appContextService$ge2 = _appContextService$ge.agentless) === null || _appContextService$ge2 === void 0 ? void 0 : (_appContextService$ge3 = _appContextService$ge2.customIntegrations) === null || _appContextService$ge3 === void 0 ? void 0 : _appContextService$ge3.enabled;
  if (packagePolicy.supports_agentless && isCustom && !isCustomAgentlessAllowed) {
    throw new _errors.CustomPackagePolicyNotAllowedForAgentlessError();
  }
  return true;
}