"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.areFieldAndUserDefinedColumnTypesCompatible = areFieldAndUserDefinedColumnTypesCompatible;
exports.createMapFromList = createMapFromList;
exports.getCurrentQueryAvailableColumns = getCurrentQueryAvailableColumns;
exports.getFieldsFromES = getFieldsFromES;
exports.getParamAtPosition = getParamAtPosition;
exports.isHeaderCommandSuggestion = isHeaderCommandSuggestion;
exports.isParam = void 0;
exports.isSourceCommandSuggestion = isSourceCommandSuggestion;
exports.noCaseCompare = void 0;
exports.nonNullable = nonNullable;
var _esqlAst = require("@kbn/esql-ast");
var _ecs_metadata_helper = require("../autocomplete/utils/ecs_metadata_helper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function nonNullable(v) {
  return v != null;
}
function isSourceCommandSuggestion({
  label
}) {
  return ['FROM', 'ROW', 'SHOW', 'TS'].includes(label);
}
function isHeaderCommandSuggestion({
  label
}) {
  return label === 'SET';
}
function createMapFromList(arr) {
  const arrMap = new Map();
  for (const item of arr) {
    arrMap.set(item.name, item);
  }
  return arrMap;
}
function areFieldAndUserDefinedColumnTypesCompatible(fieldType, userColumnType) {
  if (fieldType == null) {
    return false;
  }
  return fieldType === userColumnType;
}
const isParam = x => !!x && typeof x === 'object' && x.type === 'literal' && x.literalType === 'param';

/**
 * Compares two strings in a case-insensitive manner
 */
exports.isParam = isParam;
const noCaseCompare = (a, b) => a.toLowerCase() === b.toLowerCase();

/**
 * Given a function signature, returns the parameter at the given position.
 *
 * Takes into account variadic functions (minParams), returning the last
 * parameter if the position is greater than the number of parameters.
 *
 * @param signature
 * @param position
 * @returns
 */
exports.noCaseCompare = noCaseCompare;
function getParamAtPosition({
  params,
  minParams
}, position) {
  return params.length > position ? params[position] : minParams ? params[params.length - 1] : null;
}
async function getEcsMetadata(resourceRetriever) {
  if (!(resourceRetriever !== null && resourceRetriever !== void 0 && resourceRetriever.getFieldsMetadata)) {
    return undefined;
  }
  const client = await (resourceRetriever === null || resourceRetriever === void 0 ? void 0 : resourceRetriever.getFieldsMetadata);
  if (client.find) {
    // Fetch full list of ECS field
    // This list should be cached already by fieldsMetadataClient
    const results = await client.find({
      attributes: ['type']
    });
    return results === null || results === void 0 ? void 0 : results.fields;
  }
}
// Get the fields from the FROM clause, enrich them with ECS metadata
async function getFieldsFromES(query, resourceRetriever) {
  var _resourceRetriever$ge;
  const metadata = await getEcsMetadata();
  const fieldsOfType = await (resourceRetriever === null || resourceRetriever === void 0 ? void 0 : (_resourceRetriever$ge = resourceRetriever.getColumnsFor) === null || _resourceRetriever$ge === void 0 ? void 0 : _resourceRetriever$ge.call(resourceRetriever, {
    query
  }));
  const fieldsWithMetadata = (0, _ecs_metadata_helper.enrichFieldsWithECSInfo)(fieldsOfType || [], metadata);
  return fieldsWithMetadata;
}

/**
 * @param query, the ES|QL query
 * @param commands, the AST commands
 * @param previousPipeFields, the fields from the previous pipe
 * @returns a list of fields that are available for the current pipe
 */
async function getCurrentQueryAvailableColumns(commands, previousPipeFields, fetchFields, getPolicies, originalQueryText) {
  const lastCommand = commands[commands.length - 1];
  const commandDef = _esqlAst.esqlCommandRegistry.getCommandByName(lastCommand.name);
  const getJoinFields = command => {
    const joinSummary = _esqlAst.mutate.commands.join.summarize({
      type: 'query',
      commands: [command]
    });
    const joinIndices = joinSummary.map(({
      target: {
        index
      }
    }) => index);
    if (joinIndices.length > 0) {
      const joinFieldQuery = _esqlAst.synth.cmd`FROM ${joinIndices}`.toString();
      return fetchFields(joinFieldQuery);
    }
    return Promise.resolve([]);
  };
  const getEnrichFields = async command => {
    if (!(0, _esqlAst.isSource)(command.args[0])) {
      return [];
    }
    const policyName = command.args[0].name;
    const policies = await getPolicies();
    const policy = policies.get(policyName);
    if (policy) {
      const fieldsQuery = `FROM ${policy.sourceIndices.join(', ')} | KEEP ${policy.enrichFields.join(', ')}`;
      return fetchFields(fieldsQuery);
    }
    return [];
  };
  const getFromFields = command => {
    return fetchFields(_esqlAst.BasicPrettyPrinter.command(command));
  };
  const additionalFields = {
    fromJoin: getJoinFields,
    fromEnrich: getEnrichFields,
    fromFrom: getFromFields
  };
  if (commandDef !== null && commandDef !== void 0 && commandDef.methods.columnsAfter) {
    return commandDef.methods.columnsAfter(lastCommand, previousPipeFields, originalQueryText, additionalFields);
  }
  return previousPipeFields;
}