"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildSignatureTypes = void 0;
exports.createMessage = createMessage;
exports.errors = void 0;
exports.getMessageFromId = getMessageFromId;
exports.tagSemanticError = tagSemanticError;
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function getMessageAndTypeFromId({
  messageId,
  values
}) {
  // Use a less strict type instead of doing a typecast on each message type
  const out = values;
  // i18n validation wants to the values prop to be declared inline, so need to unpack and redeclare again all props
  switch (messageId) {
    case 'unknownColumn':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unknownColumn', {
          defaultMessage: 'Unknown column "{name}"',
          values: {
            name: out.name
          }
        })
      };
    case 'unknownIndex':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unknownIndex', {
          defaultMessage: 'Unknown index "{name}"',
          values: {
            name: out.name
          }
        })
      };
    case 'unknownFunction':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.missingFunction', {
          defaultMessage: 'Unknown function {name}',
          values: {
            name: out.name.toUpperCase()
          }
        })
      };
    case 'unknownSetting':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unknownSetting', {
          defaultMessage: 'Unknown setting {name}',
          values: {
            name: out.name
          }
        })
      };
    case 'noMatchingCallSignature':
      const signatureList = out.validSignatures.map(sig => `- (${sig})`).join('\n  ');
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.noMatchingCallSignatures', {
          defaultMessage: `Invalid input types for {functionName}.

Received ({argTypes}).

Expected one of:
  {validSignatures}`,
          values: {
            functionName: out.functionName.toUpperCase(),
            argTypes: out.argTypes,
            validSignatures: signatureList
          }
        })
      };
    case 'wrongNumberArgsVariadic':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.wrongNumberArgsVariadic', {
          defaultMessage: '{fn} expected {validArgCounts} arguments, but got {actual}.',
          values: {
            fn: out.fn.toUpperCase(),
            validArgCounts: _i18n.i18n.formatList('disjunction', Array.from(out.validArgCounts).map(String)),
            actual: out.actual
          }
        })
      };
    case 'wrongNumberArgsExact':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.wrongNumberArgsExact', {
          defaultMessage: '{fn} expected {expected, plural, one {one argument} other {{expected} arguments}}, but got {actual}.',
          values: {
            fn: out.fn.toUpperCase(),
            expected: out.expected,
            actual: out.actual
          }
        })
      };
    case 'wrongNumberArgsAtLeast':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.wrongNumberArgsAtLeast', {
          defaultMessage: '{fn} expected at least {minArgs, plural, one {one argument} other {{minArgs} arguments}}, but got {actual}.',
          values: {
            fn: out.fn.toUpperCase(),
            minArgs: out.minArgs,
            actual: out.actual
          }
        })
      };
    case 'unsupportedColumnTypeForCommand':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unsupportedColumnTypeForCommand', {
          defaultMessage: '{command} only supports values of type {type}. Found "{column}" of type {givenType}',
          values: {
            command: out.command.toUpperCase(),
            type: out.type,
            column: out.column,
            givenType: out.givenType
          }
        })
      };
    case 'unknownDissectKeyword':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unknownDissectKeyword', {
          defaultMessage: 'Expected [APPEND_SEPARATOR] in [DISSECT] but found [{keyword}]',
          values: {
            keyword: out.keyword
          }
        })
      };
    case 'functionNotAllowedHere':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.functionNotAvailableInLocation', {
          defaultMessage: 'Function {name} not allowed in {locationName}',
          values: {
            locationName: out.locationName.toUpperCase(),
            name: out.name.toUpperCase()
          }
        })
      };
    case 'unknownInterval':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unknownInterval', {
          defaultMessage: `Unexpected time interval qualifier: ''{value}''`,
          values: {
            value: out.value
          }
        })
      };
    case 'unknownPolicy':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unknownPolicy', {
          defaultMessage: 'Unknown policy "{name}"',
          values: {
            name: out.name
          }
        })
      };
    case 'nestedAggFunction':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.nestedAggFunction', {
          defaultMessage: 'Aggregation functions cannot be nested. Found {name} in {parentName}.',
          values: {
            parentName: out.parentName.toUpperCase(),
            name: out.name.toUpperCase()
          }
        })
      };
    case 'unknownAggregateFunction':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unknowAggregateFunction', {
          defaultMessage: 'Expected an aggregate function or group but got "{value}" of type {type}',
          values: {
            type: out.type,
            value: out.value
          }
        })
      };
    case 'unsupportedFieldType':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unsupportedFieldType', {
          defaultMessage: 'Field "{field}" cannot be retrieved, it is unsupported or not indexed; returning null',
          values: {
            field: out.field
          }
        }),
        type: 'warning'
      };
    case 'unsupportedMode':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unsupportedSettingValue', {
          defaultMessage: 'Unrecognized value "{value}" for {command}, mode needs to be one of [{expected}]',
          values: {
            expected: out.expected,
            value: out.value,
            command: out.command.toUpperCase()
          }
        }),
        type: 'error'
      };
    case 'metadataBracketsDeprecation':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.metadataBracketsDeprecation', {
          defaultMessage: "Square brackets '[]' need to be removed from FROM METADATA declaration"
        }),
        type: 'warning'
      };
    case 'unknownMetadataField':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.wrongMetadataArgumentType', {
          defaultMessage: 'Metadata field "{value}" is not available. Available metadata fields are: [{availableFields}]',
          values: {
            value: out.value,
            availableFields: out.availableFields
          }
        }),
        type: 'error'
      };
    case 'wrongDissectOptionArgumentType':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.wrongDissectOptionArgumentType', {
          defaultMessage: 'Invalid value for DISSECT APPEND_SEPARATOR: expected a string, but was [{value}]',
          values: {
            value: out.value
          }
        }),
        type: 'error'
      };
    case 'invalidJoinIndex':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.invalidJoinIndex', {
          defaultMessage: '"{identifier}" is not a valid JOIN index. Please use a "lookup" mode index.',
          values: {
            identifier: out.identifier
          }
        })
      };
    case 'joinOnSingleExpression':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.joinOnSingleExpression', {
          defaultMessage: 'JOIN ON clause must be a comma separated list of fields or a single expression'
        }),
        type: 'error'
      };
    case 'tooManyForks':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.tooManyForks', {
          defaultMessage: '[FORK] a query cannot have more than one FORK command.'
        })
      };
    case 'licenseRequired':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.licenseRequired', {
          defaultMessage: '{name} requires a {requiredLicense} license.',
          values: {
            name: out.name.toUpperCase(),
            requiredLicense: out.requiredLicense.toUpperCase()
          }
        })
      };
    case 'licenseRequiredForSignature':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.licenseRequiredForSignature', {
          defaultMessage: '{name} with {signatureDescription} requires a {requiredLicense} license.',
          values: {
            name: out.name.toUpperCase(),
            signatureDescription: out.signatureDescription,
            requiredLicense: out.requiredLicense.toUpperCase()
          }
        })
      };
    case 'changePointWrongFieldType':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.changePointWrongFieldType', {
          defaultMessage: 'CHANGE_POINT only supports numeric values, found "{columnName}" of type {givenType}',
          values: {
            columnName: out.columnName,
            givenType: out.givenType
          }
        }),
        type: 'error'
      };
    case 'dropTimestampWarning':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.dropTimestampWarning', {
          defaultMessage: 'Dropping "@timestamp" prevents the time range from being applied.'
        }),
        type: 'warning'
      };
    case 'inferenceIdRequired':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.inferenceIdRequired', {
          defaultMessage: '"inference_id" parameter is required for {command}.',
          values: {
            command: out.command.toUpperCase()
          }
        }),
        type: 'error'
      };
    case 'unsupportedQueryType':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.unsupportedQueryType', {
          defaultMessage: '{command} query must be of type text. Found {expressionType}',
          values: {
            command: out.command.toUpperCase(),
            expressionType: out.expressionType
          }
        }),
        type: 'error'
      };
    case 'forkTooManyBranches':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.forkTooManyBranches', {
          defaultMessage: '[FORK] Supports a maximum of 8 branches.'
        }),
        type: 'error'
      };
    case 'forkTooFewBranches':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.forkTooFewBranches', {
          defaultMessage: '[FORK] Must include at least two branches.'
        }),
        type: 'error'
      };
    case 'forkNotAllowedWithSubqueries':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.forkNotAllowedWithSubqueries', {
          defaultMessage: '[FORK] Command is not allowed inside a subquery.'
        }),
        type: 'error'
      };
    case 'inlineStatsNotAllowedAfterLimit':
      return {
        message: _i18n.i18n.translate('kbn-esql-ast.esql.validation.inlineStatsNotAllowedAfterLimit', {
          defaultMessage: '[INLINE STATS] Command is not allowed at the root level when the query contains subqueries.'
        }),
        type: 'error'
      };
  }
  return {
    message: ''
  };
}
function getMessageFromId({
  locations,
  ...payload
}) {
  const {
    message,
    type = 'error'
  } = getMessageAndTypeFromId(payload);
  return createMessage(type, message, locations, payload.messageId);
}
function createMessage(type, message, location, messageId) {
  return {
    type,
    text: message,
    location,
    code: messageId
  };
}
const createError = (messageId, location, message = '') => createMessage('error', message, location, messageId);

/**
 * Tags an error as semantic, indicating it requires runtime data to validate.
 *
 * Semantic errors depend on external data (indices, columns, policies) that can only
 * be determined at runtime. When the required callback is not available, these errors
 * will be filtered out during validation to avoid false positives.
 *
 * This is the core mechanism of the Error Tagging system, which eliminates the need
 * for manual maintenance of error-to-callback mappings.
 *
 * @param error - The base error message to tag
 * @param requiresCallback - The name of the callback required to validate this error
 *                          Common values: 'getColumnsFor', 'getSources', 'getPolicies', 'getJoinIndices'
 * @returns The error with semantic metadata attached
 *
 * @example
 * ```typescript
 * // Error that requires column information
 * unknownColumn: (column) =>
 *   tagSemanticError(
 *     errors.byId('unknownColumn', column.location, { name: column.name }),
 *     'getColumnsFor'  // Will be filtered if getColumnsFor callback is missing
 *   )
 * ```
 */
function tagSemanticError(error, requiresCallback) {
  return {
    ...error,
    errorType: 'semantic',
    requiresCallback
  };
}
const errors = exports.errors = {
  unexpected: (location, message = _i18n.i18n.translate('kbn-esql-ast.esql.validation.errors.unexpected.message', {
    defaultMessage: 'Unexpected error, this should never happen.'
  })) => {
    return createError('unexpected', location, message);
  },
  byId: (id, location, values) => getMessageFromId({
    messageId: id,
    values,
    locations: location
  }),
  unknownFunction: fn => errors.byId('unknownFunction', fn.location, fn),
  unknownColumn: column => tagSemanticError(errors.byId('unknownColumn', column.location, {
    name: column.name
  }), 'getColumnsFor'),
  unknownIndex: source => tagSemanticError(errors.byId('unknownIndex', source.location, {
    name: source.name
  }), 'getSources'),
  unknownPolicy: (policyName, location) => tagSemanticError(errors.byId('unknownPolicy', location, {
    name: policyName
  }), 'getPolicies'),
  tooManyForks: command => errors.byId('tooManyForks', command.location, {}),
  nestedAggFunction: (fn, parentName) => errors.byId('nestedAggFunction', fn.location, {
    name: fn.name,
    parentName
  }),
  unknownAggFunction: (node, type = 'FieldAttribute') => errors.byId('unknownAggregateFunction', node.location, {
    value: node.name,
    type
  }),
  invalidJoinIndex: identifier => tagSemanticError(errors.byId('invalidJoinIndex', identifier.location, {
    identifier: identifier.name
  }), 'getJoinIndices'),
  joinOnSingleExpression: location => errors.byId('joinOnSingleExpression', location, {}),
  noMatchingCallSignature: (fn, definition, argTypes) => {
    const validSignatures = definition.signatures.toSorted((a, b) => a.params.length - b.params.length).map(sig => {
      const definitionArgTypes = buildSignatureTypes(sig);
      return `${definitionArgTypes}`;
    });
    return tagSemanticError(errors.byId('noMatchingCallSignature', fn.location, {
      functionName: fn.name,
      argTypes: argTypes.join(', '),
      validSignatures
    }), 'getColumnsFor');
  },
  licenseRequired: (fn, license) => errors.byId('licenseRequired', fn.location, {
    name: fn.name,
    requiredLicense: license
  }),
  licenseRequiredForSignature: (fn, signature) => {
    const signatureDescription = signature.params.map(param => `'${param.name}' of type '${param.type}'`) // TODO this isn't well i18n'd
    .join(', ');
    return errors.byId('licenseRequiredForSignature', fn.location, {
      name: fn.name,
      signatureDescription,
      requiredLicense: signature.license
    });
  },
  functionNotAllowedHere: (fn, locationName) => errors.byId('functionNotAllowedHere', fn.location, {
    name: fn.name,
    locationName
  }),
  wrongNumberArgs: (fn, definition) => {
    const validArgCounts = new Set();
    let minParams;
    for (const sig of definition.signatures) {
      if (sig.minParams) {
        minParams = sig.minParams;
        break;
      }
      validArgCounts.add(sig.params.length);
      validArgCounts.add(sig.params.filter(p => !p.optional).length);
    }
    const arity = fn.args.length;
    if (minParams !== undefined) {
      return errors.byId('wrongNumberArgsAtLeast', fn.location, {
        fn: fn.name,
        minArgs: minParams,
        actual: arity
      });
    } else if (validArgCounts.size === 1) {
      const expected = Array.from(validArgCounts)[0];
      return errors.byId('wrongNumberArgsExact', fn.location, {
        fn: fn.name,
        expected,
        actual: fn.args.length
      });
    } else {
      return errors.byId('wrongNumberArgsVariadic', fn.location, {
        fn: fn.name,
        validArgCounts: Array.from(validArgCounts),
        actual: arity
      });
    }
  },
  changePointWrongFieldType: ({
    location,
    name
  }, type) => errors.byId('changePointWrongFieldType', location, {
    columnName: name,
    givenType: type
  }),
  dropTimestampWarning: ({
    location
  }) => errors.byId('dropTimestampWarning', location, {}),
  forkTooManyBranches: command => errors.byId('forkTooManyBranches', command.location, {}),
  forkTooFewBranches: command => errors.byId('forkTooFewBranches', command.location, {}),
  forkNotAllowedWithSubqueries: command => errors.byId('forkNotAllowedWithSubqueries', command.location, {}),
  inlineStatsNotAllowedAfterLimit: command => errors.byId('inlineStatsNotAllowedAfterLimit', command.location, {})
};
const buildSignatureTypes = sig => sig.params.map(param => {
  let ret = param.type;
  if (sig.minParams) {
    ret = '...' + ret;
  }
  if (param.optional) {
    ret = `[${ret}]`;
  }
  return ret;
}).join(', ');
exports.buildSignatureTypes = buildSignatureTypes;