"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sortModifierSuggestions = exports.rightAfterColumn = exports.getSuggestionsAfterCompleteExpression = exports.getSortPos = exports.getNullsPrefixRange = void 0;
var _helpers = require("../../../definitions/utils/autocomplete/helpers");
var _ = require("../../../..");
var _location = require("../../../ast/location");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getSortPos = (query, command, cursorPosition) => {
  const lastArg = command.args[command.args.length - 1];
  if (!lastArg || /,\s+$/.test(query)) {
    return {
      position: 'empty_expression'
    };
  }
  if (!Array.isArray(lastArg) && lastArg.type !== 'order') {
    const insideFunction = (0, _.isFunctionExpression)(lastArg) && (0, _location.within)(cursorPosition, lastArg);
    return {
      position: 'expression',
      context: {
        insideFunction
      }
    };
  }
  if (/(?:asc|desc)$/i.test(query)) {
    return {
      position: 'order_complete'
    };
  }
  if (/(?:asc|desc)\s+(?:N?U?L?L?S? ?(F?I?R?S?|LA?S?)?)$/i.test(query)) {
    return {
      position: 'after_order'
    };
  }
  if (/(?:nulls\s+first|nulls\s+last)$/i.test(query)) {
    return {
      position: 'nulls_complete'
    };
  }
  if (/(?:nulls\s+first|nulls\s+last)\s+$/i.test(query)) {
    return {
      position: 'after_nulls'
    };
  }
  return {
    position: undefined
  };
};
exports.getSortPos = getSortPos;
const sortModifierSuggestions = exports.sortModifierSuggestions = {
  ASC: (0, _helpers.withAutoSuggest)({
    label: 'ASC',
    text: 'ASC',
    detail: '',
    kind: 'Keyword',
    sortText: '1-ASC'
  }),
  DESC: (0, _helpers.withAutoSuggest)({
    label: 'DESC',
    text: 'DESC',
    detail: '',
    kind: 'Keyword',
    sortText: '1-DESC'
  }),
  NULLS_FIRST: (0, _helpers.withAutoSuggest)({
    label: 'NULLS FIRST',
    text: 'NULLS FIRST',
    detail: '',
    kind: 'Keyword',
    sortText: '2-NULLS FIRST'
  }),
  NULLS_LAST: (0, _helpers.withAutoSuggest)({
    label: 'NULLS LAST',
    text: 'NULLS LAST',
    detail: '',
    kind: 'Keyword',
    sortText: '2-NULLS LAST'
  })
};
const rightAfterColumn = (innerText, expressionRoot, columnExists) => {
  return (0, _.isColumn)(expressionRoot) && columnExists(expressionRoot.parts.join('.')) &&
  // this prevents the branch from being entered for something like "SORT column NULLS LA/"
  // where the "NULLS LA" won't be in the AST so expressionRoot will just be the column
  /(?:sort|,)\s+\S+$/i.test(innerText);
};
exports.rightAfterColumn = rightAfterColumn;
const getSuggestionsAfterCompleteExpression = (innerText, expressionRoot, columnExists) => {
  let sortCommandKeywordSuggestions = [{
    ...sortModifierSuggestions.ASC
  }, {
    ...sortModifierSuggestions.DESC
  }, {
    ...sortModifierSuggestions.NULLS_FIRST
  }, {
    ...sortModifierSuggestions.NULLS_LAST
  }];
  const pipeSuggestion = {
    ..._.pipeCompleteItem
  };
  const commaSuggestion = (0, _helpers.withAutoSuggest)({
    ..._.commaCompleteItem,
    text: ', '
  });

  // does the query end with whitespace?
  if (/\s$/.test(innerText)) {
    // if so, comma needs to be sent back a column to replace the trailing space
    commaSuggestion.rangeToReplace = {
      start: innerText.length - 1,
      end: innerText.length
    };
  }
  // special case: cursor right after a column name
  else if ((0, _.isColumn)(expressionRoot) && rightAfterColumn(innerText, expressionRoot, columnExists)) {
    const {
      fragment,
      rangeToReplace
    } = (0, _helpers.getFragmentData)(innerText);
    sortCommandKeywordSuggestions = sortCommandKeywordSuggestions.map(s => ({
      ...s,
      text: `${fragment} ${s.text}`,
      // add a space after the column name
      filterText: fragment,
      // turn off Monaco's filtering by the suggestion text
      rangeToReplace
    }));
    pipeSuggestion.filterText = fragment;
    pipeSuggestion.text = fragment + ' ' + pipeSuggestion.text;
    pipeSuggestion.rangeToReplace = rangeToReplace;
    commaSuggestion.filterText = fragment;
    commaSuggestion.text = fragment + commaSuggestion.text;
    commaSuggestion.rangeToReplace = rangeToReplace;
  }
  return [...sortCommandKeywordSuggestions, pipeSuggestion, commaSuggestion];
};
exports.getSuggestionsAfterCompleteExpression = getSuggestionsAfterCompleteExpression;
const IS_REGEX = /(?:I+S*\s+(?:N+O*T*\s+)?(N(U(L(L)?)?)?)?)\s*$/i;
const NULLS_REGEX = /(?<nulls>NULLS\s+(FI?R?S?T?|LA?S?T?)?)$/i;

/**
 * The nulls clauses are tricky because they contain whitespace.
 *
 * This function returns the overlap range between the end of the string
 * and the start of any existing NULLS clause.
 *
 * This range needs to be applied to _all_ the suggestions that are returned
 * in any context where the nulls clause is valid because Monaco needs to filter
 * suggestions based on a full prefix.
 *
 * For example, if the user types "SORT column NULLS F", the suggestions
 * will need to be filtered against the full "NULLS F" prefix instead of just "F".
 *
 * Otherwise, invalid suggestions like `FLOOR` could show up leading the user to
 * "SORT column NULLS FLOOR" which is not valid.
 *
 * @param innerText
 * @returns
 */
const getNullsPrefixRange = innerText => {
  var _matchResult$groups;
  if (IS_REGEX.test(innerText)) {
    // we're in an IS NULL or IS NOT NULL context, so we don't need to return a range
    return undefined;
  }
  const matchResult = innerText.match(NULLS_REGEX);
  const nulls = matchResult === null || matchResult === void 0 ? void 0 : (_matchResult$groups = matchResult.groups) === null || _matchResult$groups === void 0 ? void 0 : _matchResult$groups.nulls;
  return nulls ? {
    start: innerText.length - nulls.length,
    end: innerText.length
  } : undefined;
};
exports.getNullsPrefixRange = getNullsPrefixRange;