"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateTabs = exports.setTabs = exports.restoreTab = exports.openSearchSessionInNewTab = exports.openInNewTab = exports.initializeTabs = exports.disconnectTab = exports.clearRecentlyClosedTabs = void 0;
var _lodash = require("lodash");
var _public = require("@kbn/saved-search-plugin/public");
var _i18n = require("@kbn/i18n");
var _esQuery = require("@kbn/es-query");
var _esqlUtils = require("@kbn/esql-utils");
var _utils = require("../../../../../../common/data_sources/utils");
var _selectors = require("../selectors");
var _internal_state = require("../internal_state");
var _runtime_state = require("../runtime_state");
var _constants = require("../../../../../../common/constants");
var _utils2 = require("../utils");
var _breadcrumbs = require("../../../../../utils/breadcrumbs");
var _constants2 = require("../constants");
var _app_locator_get_location = require("../../../../../../common/app_locator_get_location");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const setTabs = params => function setTabsThunkFn(dispatch, getState, {
  runtimeStateManager,
  tabsStorageManager,
  services: {
    profilesManager,
    ebtManager
  }
}) {
  var _previousState$persis;
  const previousState = getState();
  const discoverSessionChanged = params.updatedDiscoverSession && previousState.persistedDiscoverSession && params.updatedDiscoverSession.id !== ((_previousState$persis = previousState.persistedDiscoverSession) === null || _previousState$persis === void 0 ? void 0 : _previousState$persis.id);
  const previousTabs = (0, _selectors.selectAllTabs)(previousState);
  const removedTabs = discoverSessionChanged ? previousTabs : (0, _lodash.differenceBy)(previousTabs, params.allTabs, differenceIterateeByTabId);
  const addedTabs = discoverSessionChanged ? params.allTabs : (0, _lodash.differenceBy)(params.allTabs, previousTabs, differenceIterateeByTabId);
  const justRemovedTabs = [];
  for (const tab of removedTabs) {
    var _selectTabRuntimeInte;
    const newRecentlyClosedTab = {
      ...tab
    };
    // make sure to get the latest internal and app state from runtime state manager before deleting the runtime state
    newRecentlyClosedTab.initialInternalState = (_selectTabRuntimeInte = (0, _runtime_state.selectTabRuntimeInternalState)(runtimeStateManager, tab.id)) !== null && _selectTabRuntimeInte !== void 0 ? _selectTabRuntimeInte : (0, _lodash.cloneDeep)(tab.initialInternalState);
    newRecentlyClosedTab.appState = (0, _lodash.cloneDeep)(tab.appState);
    newRecentlyClosedTab.globalState = (0, _lodash.cloneDeep)(tab.globalState);
    justRemovedTabs.push(newRecentlyClosedTab);
    dispatch(disconnectTab({
      tabId: tab.id
    }));
    delete runtimeStateManager.tabs.byId[tab.id];
  }
  for (const tab of addedTabs) {
    runtimeStateManager.tabs.byId[tab.id] = (0, _runtime_state.createTabRuntimeState)({
      profilesManager,
      ebtManager,
      initialValues: {
        unifiedHistogramLayoutPropsMap: tab.duplicatedFromId ? (0, _runtime_state.selectInitialUnifiedHistogramLayoutPropsMap)(runtimeStateManager, tab.duplicatedFromId) : undefined
      }
    });
  }
  const selectedTabRuntimeState = (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, params.selectedTabId);
  if (selectedTabRuntimeState) {
    selectedTabRuntimeState.scopedEbtManager$.getValue().setAsActiveManager();
  }
  dispatch(_internal_state.internalStateSlice.actions.setTabs({
    ...params,
    recentlyClosedTabs: tabsStorageManager.getNRecentlyClosedTabs({
      previousOpenTabs: previousTabs,
      previousRecentlyClosedTabs: params.recentlyClosedTabs,
      nextOpenTabs: params.allTabs,
      justRemovedTabs
    })
  }));
};
exports.setTabs = setTabs;
const updateTabs = ({
  items,
  selectedItem,
  updatedDiscoverSession
}) => async (dispatch, getState, {
  services,
  runtimeStateManager,
  tabsStorageManager,
  urlStateStorage,
  searchSessionManager
}) => {
  const currentState = getState();
  const currentTab = (0, _selectors.selectTab)(currentState, currentState.tabs.unsafeCurrentId);
  const currentTabRuntimeState = (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, currentTab.id);
  const currentTabStateContainer = currentTabRuntimeState.stateContainer$.getValue();
  const updatedTabs = items.map(item => {
    const existingTab = (0, _selectors.selectTab)(currentState, item.id);
    const tab = {
      ..._constants2.DEFAULT_TAB_STATE,
      ...{
        globalState: {
          timeRange: services.timefilter.getTime(),
          refreshInterval: services.timefilter.getRefreshInterval(),
          filters: services.filterManager.getGlobalFilters()
        }
      },
      ...existingTab,
      ...item
    };
    if (existingTab) {
      return tab;
    }

    // TODO: these lines can likely be removed since `item` is already spread to `tab` above
    // the following assignments for appState, globalState, and dataRequestParams are for supporting `openInNewTab` action
    tab.appState = 'appState' in item ? (0, _lodash.cloneDeep)(item.appState) : tab.appState;
    tab.globalState = 'globalState' in item ? (0, _lodash.cloneDeep)(item.globalState) : tab.globalState;
    tab.dataRequestParams = 'dataRequestParams' in item ? item.dataRequestParams : tab.dataRequestParams;
    if (item.duplicatedFromId) {
      var _selectTabRuntimeInte2;
      // the new tab was created by duplicating an existing tab
      const existingTabToDuplicateFrom = (0, _selectors.selectTab)(currentState, item.duplicatedFromId);
      if (!existingTabToDuplicateFrom) {
        return tab;
      }
      tab.initialInternalState = (_selectTabRuntimeInte2 = (0, _runtime_state.selectTabRuntimeInternalState)(runtimeStateManager, item.duplicatedFromId)) !== null && _selectTabRuntimeInte2 !== void 0 ? _selectTabRuntimeInte2 : (0, _lodash.cloneDeep)(existingTabToDuplicateFrom.initialInternalState);
      tab.appState = (0, _lodash.cloneDeep)(existingTabToDuplicateFrom.appState);
      tab.globalState = (0, _lodash.cloneDeep)(existingTabToDuplicateFrom.globalState);
      tab.uiState = (0, _lodash.cloneDeep)(existingTabToDuplicateFrom.uiState);
    } else if (item.restoredFromId) {
      // the new tab was created by restoring a recently closed tab
      const recentlyClosedTabToRestore = (0, _selectors.selectRecentlyClosedTabs)(currentState).find(t => t.id === item.restoredFromId);
      if (!recentlyClosedTabToRestore) {
        return tab;
      }
      tab.initialInternalState = (0, _lodash.cloneDeep)(recentlyClosedTabToRestore.initialInternalState);
      tab.appState = (0, _lodash.cloneDeep)(recentlyClosedTabToRestore.appState);
      tab.globalState = (0, _lodash.cloneDeep)(recentlyClosedTabToRestore.globalState);
    } else if (!('appState' in item)) {
      // the new tab is a fresh one
      const currentQuery = currentTab.appState.query;
      const currentDataView = currentTabRuntimeState.currentDataView$.getValue();
      if (!currentQuery || !currentDataView) {
        return tab;
      }
      tab.appState = {
        ...((0, _esQuery.isOfAggregateQueryType)(currentQuery) ? {
          query: {
            esql: (0, _esqlUtils.getInitialESQLQuery)(currentDataView, true)
          }
        } : {}),
        dataSource: (0, _utils.createDataSource)({
          dataView: currentDataView,
          query: currentQuery
        })
      };
    }
    return tab;
  });
  const selectedTab = selectedItem !== null && selectedItem !== void 0 ? selectedItem : currentTab;

  // Push the selected tab ID to the URL, which creates a new browser history entry.
  // This must be done before setting other URL state, which replace the history entry
  // in order to avoid creating multiple browser history entries when switching tabs.
  await tabsStorageManager.pushSelectedTabIdToUrl(selectedTab.id);
  if (selectedTab.id !== currentTab.id) {
    currentTabStateContainer === null || currentTabStateContainer === void 0 ? void 0 : currentTabStateContainer.actions.stopSyncing();
    const nextTab = updatedTabs.find(tab => tab.id === selectedTab.id);
    const nextTabRuntimeState = (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, selectedTab.id);
    const nextTabStateContainer = nextTabRuntimeState === null || nextTabRuntimeState === void 0 ? void 0 : nextTabRuntimeState.stateContainer$.getValue();
    if (nextTab && nextTabStateContainer) {
      const {
        timeRange,
        refreshInterval,
        filters: globalFilters
      } = nextTab.globalState;
      const appState = nextTabStateContainer.appState.get();
      const {
        filters: appFilters,
        query
      } = appState;
      await urlStateStorage.set(_constants.GLOBAL_STATE_URL_KEY, {
        time: timeRange,
        refreshInterval,
        filters: globalFilters
      }, {
        replace: true
      });
      await urlStateStorage.set(_constants.APP_STATE_URL_KEY, appState, {
        replace: true
      });
      services.timefilter.setTime(timeRange !== null && timeRange !== void 0 ? timeRange : services.timefilter.getTimeDefaults());
      services.timefilter.setRefreshInterval(refreshInterval !== null && refreshInterval !== void 0 ? refreshInterval : services.timefilter.getRefreshIntervalDefaults());
      services.filterManager.setGlobalFilters((0, _lodash.cloneDeep)(globalFilters !== null && globalFilters !== void 0 ? globalFilters : []));
      services.filterManager.setAppFilters((0, _lodash.cloneDeep)(appFilters !== null && appFilters !== void 0 ? appFilters : []));
      services.data.query.queryString.setQuery(query !== null && query !== void 0 ? query : services.data.query.queryString.getDefaultQuery());
      if (nextTab.dataRequestParams.searchSessionId) {
        services.data.search.session.continue(nextTab.dataRequestParams.searchSessionId, true);
        if (nextTab.dataRequestParams.isSearchSessionRestored) {
          searchSessionManager.pushSearchSessionIdToURL(nextTab.dataRequestParams.searchSessionId, {
            replace: true
          });
        }
      }
      if (!nextTab.dataRequestParams.isSearchSessionRestored) {
        searchSessionManager.removeSearchSessionIdFromURL({
          replace: true
        });
      }
      nextTabStateContainer.actions.initializeAndSync();
      if (nextTab.forceFetchOnSelect) {
        nextTabStateContainer.dataState.reset();
        nextTabStateContainer.actions.fetchData();
      }
    } else {
      await urlStateStorage.set(_constants.GLOBAL_STATE_URL_KEY, null, {
        replace: true
      });
      await urlStateStorage.set(_constants.APP_STATE_URL_KEY, null, {
        replace: true
      });
      searchSessionManager.removeSearchSessionIdFromURL({
        replace: true
      });
      services.data.search.session.reset();
    }
    dispatch(_internal_state.internalStateSlice.actions.discardFlyoutsOnTabChange());
  }
  dispatch(setTabs({
    allTabs: updatedTabs,
    selectedTabId: selectedTab.id,
    recentlyClosedTabs: (0, _selectors.selectRecentlyClosedTabs)(currentState),
    updatedDiscoverSession
  }));
};
exports.updateTabs = updateTabs;
const initializeTabs = exports.initializeTabs = (0, _utils2.createInternalStateAsyncThunk)('internalState/initializeTabs', async function initializeTabsThunkFn({
  discoverSessionId,
  shouldClearAllTabs
}, {
  dispatch,
  getState,
  extra: {
    services,
    tabsStorageManager,
    customizationContext
  }
}) {
  const {
    userId: existingUserId,
    spaceId: existingSpaceId
  } = getState();
  const getUserId = async () => {
    try {
      var _await$services$core$, _services$core$securi;
      return (_await$services$core$ = (await ((_services$core$securi = services.core.security) === null || _services$core$securi === void 0 ? void 0 : _services$core$securi.authc.getCurrentUser())).profile_uid) !== null && _await$services$core$ !== void 0 ? _await$services$core$ : '';
    } catch {
      // ignore as user id might be unavailable for some deployments
      return '';
    }
  };
  const getSpaceId = async () => {
    try {
      var _await$services$space, _await$services$space2, _services$spaces;
      return (_await$services$space = (_await$services$space2 = await ((_services$spaces = services.spaces) === null || _services$spaces === void 0 ? void 0 : _services$spaces.getActiveSpace())) === null || _await$services$space2 === void 0 ? void 0 : _await$services$space2.id) !== null && _await$services$space !== void 0 ? _await$services$space : '';
    } catch {
      // ignore
      return '';
    }
  };
  const [userId, spaceId, persistedDiscoverSession] = await Promise.all([existingUserId === undefined ? getUserId() : existingUserId, existingSpaceId === undefined ? getSpaceId() : existingSpaceId, discoverSessionId ? services.savedSearch.getDiscoverSession(discoverSessionId) : undefined]);
  if (customizationContext.displayMode === 'standalone' && persistedDiscoverSession) {
    var _persistedDiscoverSes;
    services.chrome.recentlyAccessed.add((0, _public.getSavedSearchFullPathUrl)(persistedDiscoverSession.id), (_persistedDiscoverSes = persistedDiscoverSession.title) !== null && _persistedDiscoverSes !== void 0 ? _persistedDiscoverSes : _i18n.i18n.translate('discover.defaultDiscoverSessionTitle', {
      defaultMessage: 'Untitled Discover session'
    }), persistedDiscoverSession.id);
    (0, _breadcrumbs.setBreadcrumbs)({
      services,
      titleBreadcrumbText: persistedDiscoverSession.title
    });
  }
  const initialTabsState = tabsStorageManager.loadLocally({
    userId,
    spaceId,
    persistedDiscoverSession,
    shouldClearAllTabs,
    defaultTabState: _constants2.DEFAULT_TAB_STATE
  });
  const history = services.getScopedHistory();
  const locationState = history === null || history === void 0 ? void 0 : history.location.state;

  // Replace instead of push the tab ID to the URL on initialization in order to
  // avoid capturing a browser history entry with a potentially empty _tab state
  await tabsStorageManager.pushSelectedTabIdToUrl(initialTabsState.selectedTabId, {
    replace: true
  });

  // Manually restore the previous location state since pushing the tab ID
  // to the URL clears it, but initial location state must be passed on,
  // e.g. ad hoc data views specs
  if (locationState) {
    history.replace({
      ...history.location,
      state: locationState
    });
  }
  dispatch(setTabs({
    ...initialTabsState,
    updatedDiscoverSession: persistedDiscoverSession
  }));
  return {
    userId,
    spaceId,
    persistedDiscoverSession
  };
});
const restoreTab = ({
  restoreTabId
}) => function restoreTabThunkFn(dispatch, getState) {
  const currentState = getState();

  // Restoring the 'new' tab ID is a no-op because it represents a placeholder for creating new tabs,
  // not an actual tab that can be restored.
  if (restoreTabId === currentState.tabs.unsafeCurrentId || restoreTabId === _constants.NEW_TAB_ID) {
    return;
  }
  const currentTabs = (0, _selectors.selectAllTabs)(currentState);
  const currentTab = (0, _selectors.selectTab)(currentState, currentState.tabs.unsafeCurrentId);
  let items = currentTabs;
  // search among open tabs
  let selectedItem = items.find(tab => tab.id === restoreTabId);
  if (!selectedItem) {
    // search among recently closed tabs
    const recentlyClosedTabs = (0, _selectors.selectRecentlyClosedTabs)(currentState);
    const closedTab = recentlyClosedTabs.find(tab => tab.id === restoreTabId);
    if (closedTab) {
      // reopening one of the closed tabs
      selectedItem = (0, _lodash.omit)(closedTab, 'closedAt');
      items = [...items, closedTab];
    }
  }
  return dispatch(updateTabs({
    items,
    // TODO: should we even call updateTabs if selectedItem is not found or just return?
    selectedItem: selectedItem || currentTab
  }));
};
exports.restoreTab = restoreTab;
const openInNewTab = ({
  tabLabel,
  appState,
  globalState,
  searchSessionId,
  dataViewSpec
}) => function openInNewTabThunkFn(dispatch, getState) {
  const initialAppState = appState ? (0, _lodash.cloneDeep)(appState) : {};
  const initialGlobalState = globalState ? (0, _lodash.cloneDeep)(globalState) : {};
  const currentState = getState();
  const currentTabs = (0, _selectors.selectAllTabs)(currentState);
  const newDefaultTab = {
    ..._constants2.DEFAULT_TAB_STATE,
    ...(0, _utils2.createTabItem)(currentTabs),
    appState: initialAppState,
    globalState: initialGlobalState
  };
  if (tabLabel) {
    newDefaultTab.label = tabLabel;
  }
  if (searchSessionId) {
    newDefaultTab.initialInternalState = {
      ...newDefaultTab.initialInternalState,
      searchSessionId
    };
  }
  if (dataViewSpec) {
    newDefaultTab.initialInternalState = {
      ...newDefaultTab.initialInternalState,
      serializedSearchSource: {
        index: dataViewSpec
      }
    };
  }
  return dispatch(updateTabs({
    items: [...currentTabs, newDefaultTab],
    selectedItem: newDefaultTab
  }));
};
exports.openInNewTab = openInNewTab;
const openSearchSessionInNewTab = ({
  searchSession
}) => async function openSearchSessionInNewTabThunkFn(dispatch) {
  const restoreState = searchSession.restoreState;
  if (!restoreState.searchSessionId) {
    return;
  }
  const {
    appState,
    globalState: originalGlobalState,
    state: {
      dataViewSpec
    }
  } = (0, _app_locator_get_location.parseAppLocatorParams)(restoreState);
  const globalState = {};
  if (originalGlobalState !== null && originalGlobalState !== void 0 && originalGlobalState.time) {
    globalState.timeRange = originalGlobalState.time;
  }
  if (originalGlobalState !== null && originalGlobalState !== void 0 && originalGlobalState.refreshInterval) {
    globalState.refreshInterval = originalGlobalState.refreshInterval;
  }
  if (originalGlobalState !== null && originalGlobalState !== void 0 && originalGlobalState.filters) {
    globalState.filters = originalGlobalState.filters;
  }
  return dispatch(openInNewTab({
    tabLabel: searchSession.name,
    searchSessionId: restoreState.searchSessionId,
    appState,
    globalState,
    dataViewSpec
  }));
};
exports.openSearchSessionInNewTab = openSearchSessionInNewTab;
const clearRecentlyClosedTabs = () => function clearRecentlyClosedTabsThunkFn(dispatch, getState) {
  const currentState = getState();
  return dispatch(setTabs({
    allTabs: (0, _selectors.selectAllTabs)(currentState),
    selectedTabId: currentState.tabs.unsafeCurrentId,
    recentlyClosedTabs: []
  }));
};
exports.clearRecentlyClosedTabs = clearRecentlyClosedTabs;
const disconnectTab = ({
  tabId
}) => function disconnectTabThunkFn(_, __, {
  runtimeStateManager
}) {
  var _tabRuntimeState$cust;
  const tabRuntimeState = (0, _runtime_state.selectTabRuntimeState)(runtimeStateManager, tabId);
  const stateContainer = tabRuntimeState.stateContainer$.getValue();
  stateContainer === null || stateContainer === void 0 ? void 0 : stateContainer.dataState.cancel();
  stateContainer === null || stateContainer === void 0 ? void 0 : stateContainer.actions.stopSyncing();
  (_tabRuntimeState$cust = tabRuntimeState.customizationService$.getValue()) === null || _tabRuntimeState$cust === void 0 ? void 0 : _tabRuntimeState$cust.cleanup();
};
exports.disconnectTab = disconnectTab;
function differenceIterateeByTabId(tab) {
  return tab.id;
}