"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.THRESHOLD_MET_GROUP_ID = exports.THRESHOLD_MET_GROUP = exports.RULE_TYPES_CONFIG = exports.AggregationType = exports.APM_SERVER_FEATURE_ID = exports.ANOMALY_DETECTOR_SELECTOR_OPTIONS = exports.ANOMALY_ALERT_SEVERITY_TYPES = void 0;
exports.formatAnomalyReason = formatAnomalyReason;
exports.formatErrorCountReason = formatErrorCountReason;
exports.formatTransactionDurationReason = formatTransactionDurationReason;
exports.formatTransactionErrorRateReason = formatTransactionErrorRateReason;
exports.getApmMlDetectorLabel = getApmMlDetectorLabel;
exports.getFieldValueLabel = void 0;
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/observability-plugin/common");
var _anomaly_severity = require("@kbn/ml-anomaly-utils/anomaly_severity");
var _anomaly_threshold = require("@kbn/ml-anomaly-utils/anomaly_threshold");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _apm = require("../es_fields/apm");
var _environment_filter_values = require("../environment_filter_values");
var _apm_ml_detectors = require("../anomaly_detection/apm_ml_detectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const APM_SERVER_FEATURE_ID = exports.APM_SERVER_FEATURE_ID = 'apm';
let AggregationType = exports.AggregationType = /*#__PURE__*/function (AggregationType) {
  AggregationType["Avg"] = "avg";
  AggregationType["P95"] = "95th";
  AggregationType["P99"] = "99th";
  return AggregationType;
}({});
const THRESHOLD_MET_GROUP_ID = exports.THRESHOLD_MET_GROUP_ID = 'threshold_met';
const THRESHOLD_MET_GROUP = exports.THRESHOLD_MET_GROUP = {
  id: THRESHOLD_MET_GROUP_ID,
  name: _i18n.i18n.translate('xpack.apm.a.thresholdMet', {
    defaultMessage: 'Threshold met'
  })
};
const getFieldNameLabel = field => {
  switch (field) {
    case _apm.SERVICE_NAME:
      return 'service';
    case _apm.SERVICE_ENVIRONMENT:
      return 'env';
    case _apm.TRANSACTION_TYPE:
      return 'type';
    case _apm.TRANSACTION_NAME:
      return 'name';
    case _apm.ERROR_GROUP_ID:
      return 'error key';
    case _apm.ERROR_GROUP_NAME:
      return 'error name';
    default:
      return field;
  }
};
const getFieldValueLabel = (field, fieldValue) => {
  return field === _apm.SERVICE_ENVIRONMENT ? (0, _environment_filter_values.getEnvironmentLabel)(fieldValue) : fieldValue;
};
exports.getFieldValueLabel = getFieldValueLabel;
const formatGroupByFields = groupByFields => {
  const groupByFieldLabels = Object.keys(groupByFields).map(field => `${getFieldNameLabel(field)}: ${getFieldValueLabel(field, groupByFields[field])}`);
  return groupByFieldLabels.join(', ');
};
function formatErrorCountReason({
  threshold,
  measured,
  windowSize,
  windowUnit,
  groupByFields
}) {
  return _i18n.i18n.translate('xpack.apm.alertTypes.errorCount.reason', {
    defaultMessage: `Error count is {measured} in the last {interval} for {group}. Alert when > {threshold}.`,
    values: {
      threshold,
      measured,
      interval: (0, _common.formatDurationFromTimeUnitChar)(windowSize, windowUnit),
      group: formatGroupByFields(groupByFields)
    }
  });
}
function formatTransactionDurationReason({
  threshold,
  measured,
  asDuration,
  aggregationType,
  windowSize,
  windowUnit,
  groupByFields
}) {
  let aggregationTypeFormatted = aggregationType.charAt(0).toUpperCase() + aggregationType.slice(1);
  if (aggregationTypeFormatted === 'Avg') aggregationTypeFormatted = aggregationTypeFormatted + '.';
  return _i18n.i18n.translate('xpack.apm.alertTypes.transactionDuration.reason', {
    defaultMessage: `{aggregationType} latency is {measured} in the last {interval} for {group}. Alert when > {threshold}.`,
    values: {
      threshold: asDuration(threshold),
      measured: asDuration(measured),
      aggregationType: aggregationTypeFormatted,
      interval: (0, _common.formatDurationFromTimeUnitChar)(windowSize, windowUnit),
      group: formatGroupByFields(groupByFields)
    }
  });
}
function formatTransactionErrorRateReason({
  threshold,
  measured,
  asPercent,
  windowSize,
  windowUnit,
  groupByFields
}) {
  return _i18n.i18n.translate('xpack.apm.alertTypes.transactionErrorRate.reason', {
    defaultMessage: `Failed transactions is {measured} in the last {interval} for {group}. Alert when > {threshold}.`,
    values: {
      threshold: asPercent(threshold, 100),
      measured: asPercent(measured, 100),
      interval: (0, _common.formatDurationFromTimeUnitChar)(windowSize, windowUnit),
      group: formatGroupByFields(groupByFields)
    }
  });
}
function formatAnomalyReason({
  serviceName,
  severityLevel,
  anomalyScore,
  windowSize,
  windowUnit,
  detectorType
}) {
  return _i18n.i18n.translate('xpack.apm.alertTypes.transactionDurationAnomaly.reason', {
    defaultMessage: `{severityLevel} {detectorTypeLabel} anomaly with a score of {anomalyScore}, was detected in the last {interval} for {serviceName}.`,
    values: {
      serviceName,
      severityLevel,
      detectorTypeLabel: getApmMlDetectorLabel(detectorType),
      anomalyScore,
      interval: (0, _common.formatDurationFromTimeUnitChar)(windowSize, windowUnit)
    }
  });
}
const RULE_TYPES_CONFIG = exports.RULE_TYPES_CONFIG = {
  [_ruleDataUtils.ApmRuleType.ErrorCount]: {
    name: _i18n.i18n.translate('xpack.apm.errorCountAlert.name', {
      defaultMessage: 'Error count threshold'
    }),
    actionGroups: [THRESHOLD_MET_GROUP],
    defaultActionGroupId: THRESHOLD_MET_GROUP_ID,
    minimumLicenseRequired: 'basic',
    producer: APM_SERVER_FEATURE_ID,
    isExportable: true
  },
  [_ruleDataUtils.ApmRuleType.TransactionDuration]: {
    name: _i18n.i18n.translate('xpack.apm.transactionDurationAlert.name', {
      defaultMessage: 'Latency threshold'
    }),
    actionGroups: [THRESHOLD_MET_GROUP],
    defaultActionGroupId: THRESHOLD_MET_GROUP_ID,
    minimumLicenseRequired: 'basic',
    producer: APM_SERVER_FEATURE_ID,
    isExportable: true
  },
  [_ruleDataUtils.ApmRuleType.Anomaly]: {
    name: _i18n.i18n.translate('xpack.apm.anomalyAlert.name', {
      defaultMessage: 'APM Anomaly'
    }),
    actionGroups: [THRESHOLD_MET_GROUP],
    defaultActionGroupId: THRESHOLD_MET_GROUP_ID,
    minimumLicenseRequired: 'basic',
    producer: APM_SERVER_FEATURE_ID,
    isExportable: true
  },
  [_ruleDataUtils.ApmRuleType.TransactionErrorRate]: {
    name: _i18n.i18n.translate('xpack.apm.transactionErrorRateAlert.name', {
      defaultMessage: 'Failed transaction rate threshold'
    }),
    actionGroups: [THRESHOLD_MET_GROUP],
    defaultActionGroupId: THRESHOLD_MET_GROUP_ID,
    minimumLicenseRequired: 'basic',
    producer: APM_SERVER_FEATURE_ID,
    isExportable: true
  }
};
const ANOMALY_ALERT_SEVERITY_TYPES = exports.ANOMALY_ALERT_SEVERITY_TYPES = [{
  type: _anomaly_severity.ML_ANOMALY_SEVERITY.CRITICAL,
  label: _i18n.i18n.translate('xpack.apm.alerts.anomalySeverity.criticalLabel', {
    defaultMessage: 'critical'
  }),
  threshold: _anomaly_threshold.ML_ANOMALY_THRESHOLD.CRITICAL
}, {
  type: _anomaly_severity.ML_ANOMALY_SEVERITY.MAJOR,
  label: _i18n.i18n.translate('xpack.apm.alerts.anomalySeverity.majorLabel', {
    defaultMessage: 'major'
  }),
  threshold: _anomaly_threshold.ML_ANOMALY_THRESHOLD.MAJOR
}, {
  type: _anomaly_severity.ML_ANOMALY_SEVERITY.MINOR,
  label: _i18n.i18n.translate('xpack.apm.alerts.anomalySeverity.minor', {
    defaultMessage: 'minor'
  }),
  threshold: _anomaly_threshold.ML_ANOMALY_THRESHOLD.MINOR
}, {
  type: _anomaly_severity.ML_ANOMALY_SEVERITY.WARNING,
  label: _i18n.i18n.translate('xpack.apm.alerts.anomalySeverity.warningLabel', {
    defaultMessage: 'warning'
  }),
  threshold: _anomaly_threshold.ML_ANOMALY_THRESHOLD.WARNING
}];
function getApmMlDetectorLabel(type) {
  switch (type) {
    case _apm_ml_detectors.AnomalyDetectorType.txLatency:
      return _i18n.i18n.translate('xpack.apm.alerts.anomalyDetector.latencyLabel', {
        defaultMessage: 'latency'
      });
    case _apm_ml_detectors.AnomalyDetectorType.txThroughput:
      return _i18n.i18n.translate('xpack.apm.alerts.anomalyDetector.throughputLabel', {
        defaultMessage: 'throughput'
      });
    case _apm_ml_detectors.AnomalyDetectorType.txFailureRate:
      return _i18n.i18n.translate('xpack.apm.alerts.anomalyDetector.failedTransactionRateLabel', {
        defaultMessage: 'failed transaction rate'
      });
  }
}
const ANOMALY_DETECTOR_SELECTOR_OPTIONS = exports.ANOMALY_DETECTOR_SELECTOR_OPTIONS = [_apm_ml_detectors.AnomalyDetectorType.txLatency, _apm_ml_detectors.AnomalyDetectorType.txThroughput, _apm_ml_detectors.AnomalyDetectorType.txFailureRate].map(type => ({
  type,
  label: getApmMlDetectorLabel(type)
}));

// Server side registrations
// x-pack/solutions/observability/plugins/apm/server/lib/alerts/<alert>.ts
// x-pack/solutions/observability/plugins/apm/server/lib/alerts/register_apm_alerts.ts

// Client side registrations:
// x-pack/solutions/observability/plugins/apm/public/components/alerting/<alert>/index.tsx
// x-pack/solutions/observability/plugins/apm/public/components/alerting/register_apm_alerts