"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAPIRoutes = registerAPIRoutes;
var _configSchema = require("@kbn/config-schema");
var _content_management = require("../../common/content_management");
var _constants = require("./constants");
var _v = require("../content_management/v1");
var _create = require("./create");
var _update = require("./update");
var _delete = require("./delete");
var _search = require("./search");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const formatResult = item => {
  const {
    id,
    type,
    attributes,
    createdAt,
    updatedAt,
    createdBy,
    updatedBy,
    error,
    managed,
    version,
    // TODO rest contains spaces and namespaces
    // These should not be spread into data and instead be moved to meta
    ...rest
  } = item;
  return {
    id,
    type,
    data: {
      ...attributes,
      ...rest
    },
    meta: {
      createdAt,
      updatedAt,
      createdBy,
      updatedBy,
      error,
      managed,
      version
    }
  };
};
function registerAPIRoutes({
  http,
  contentManagement,
  restCounter,
  logger
}) {
  const {
    versioned: versionedRouter
  } = http.createRouter();
  (0, _create.registerCreateRoute)(versionedRouter);
  (0, _update.registerUpdateRoute)(versionedRouter);
  (0, _delete.registerDeleteRoute)(versionedRouter);
  (0, _search.registerSearchRoute)(versionedRouter);

  // Get API route
  const getRoute = versionedRouter.get({
    path: `${_constants.PUBLIC_API_PATH}/{id}`,
    summary: `Get a dashboard`,
    ..._constants.commonRouteConfig
  });
  getRoute.addVersion({
    version: _constants.INTERNAL_API_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'A unique identifier for the dashboard.'
            }
          })
        })
      },
      response: {
        200: {
          body: _v.getDashboardAPIGetResultSchema
        }
      }
    }
  }, async (ctx, req, res) => {
    const client = contentManagement.contentClient.getForRequest({
      request: req,
      requestHandlerContext: ctx
    }).for(_content_management.CONTENT_ID, _content_management.LATEST_VERSION);
    let result;
    try {
      ({
        result
      } = await client.get(req.params.id));
    } catch (e) {
      if (e.isBoom && e.output.statusCode === 404) {
        return res.notFound({
          body: {
            message: `A dashboard with saved object ID ${req.params.id}] was not found.`
          }
        });
      }
      if (e.isBoom && e.output.statusCode === 403) {
        return res.forbidden();
      }
      return res.badRequest(e.message);
    }
    const formattedResult = formatResult(result.item);
    return res.ok({
      body: {
        ...formattedResult,
        meta: {
          ...formattedResult.meta,
          ...result.meta
        }
      }
    });
  });
}