"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTypeHashes = exports.getMigrationHash = void 0;
var _lodash = require("lodash");
var _crypto = require("crypto");
var _extract_migration_info = require("./extract_migration_info");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getTypeHashes = soType => {
  const migrationInfo = (0, _extract_migration_info.extractMigrationInfo)(soType);
  const migrationHashes = soType.migrations ? getMigrationsHashes(soType) : [];
  const modelVersionsHashes = soType.modelVersions ? getModelVersionsHashes(soType) : [];
  const warning = [];
  const hasMigrationsWrapper = (0, _lodash.uniq)(migrationHashes.map(entry => entry.split(': ').pop())).length < migrationHashes.length;
  const hasModelVersionsWrapper = (0, _lodash.uniq)(modelVersionsHashes.map(entry => entry.split(': ').pop())).length < modelVersionsHashes.length;
  if (hasMigrationsWrapper || hasModelVersionsWrapper) {
    warning.push(`${soType.name}|warning: The SO type owner should ensure these transform functions DO NOT mutate after they are defined.`);
    if (hasMigrationsWrapper) {
      warning.push(`${soType.name}|warning: This type uses 'migrations:' WRAPPER functions that prevent detecting changes in the implementation.`);
    }
    if (hasModelVersionsWrapper) {
      warning.push(`${soType.name}|warning: This type uses 'modelVersions:' WRAPPER functions that prevent detecting changes in the implementation.`);
    }
  }
  return [...warning, ...migrationHashes, ...modelVersionsHashes, `${soType.name}|schemas: ${getTypeSchemasHash(migrationInfo)}`, `${soType.name}|mappings: ${getTypeMappingsHash(migrationInfo)}`, `${soType.name}|global: ${getTypeGlobalHash(migrationInfo)}`].reverse();
};
exports.getTypeHashes = getTypeHashes;
const getTypeGlobalHash = migInfo => {
  const hash = (0, _crypto.createHash)('sha1'); // eslint-disable-line @kbn/eslint/no_unsafe_hash
  const globalData = [migInfo.name, migInfo.namespaceType, migInfo.convertToAliasScript, migInfo.hasExcludeOnUpgrade, migInfo.convertToMultiNamespaceTypeVersion, migInfo.schemaVersions].join('|');
  return hash.update(globalData).digest('hex');
};
const getTypeMappingsHash = migInfo => {
  const hash = (0, _crypto.createHash)('sha1'); // eslint-disable-line @kbn/eslint/no_unsafe_hash
  const mappingData = JSON.stringify(migInfo.mappings, Object.keys(migInfo.mappings).sort());
  return hash.update(mappingData).digest('hex');
};
const getTypeSchemasHash = migInfo => {
  const hash = (0, _crypto.createHash)('sha1'); // eslint-disable-line @kbn/eslint/no_unsafe_hash
  const schemaData = migInfo.schemaVersions.join('|');
  return hash.update(schemaData).digest('hex');
};
const getMigrationsHashes = soType => {
  const migrations = typeof soType.migrations === 'function' ? soType.migrations() : soType.migrations;
  return Object.entries(migrations).map(([version, migration]) => {
    const hash = (0, _crypto.createHash)('sha1'); // eslint-disable-line @kbn/eslint/no_unsafe_hash
    const migrationData = typeof migration === 'function' ? migration.toString() : [!!migration.deferred, migration.transform.toString()].join('|');
    return `${soType.name}|${version}: ${hash.update(migrationData).digest('hex')}`;
  });
};
const getModelVersionsHashes = soType => {
  const modelVersions = typeof soType.modelVersions === 'function' ? soType.modelVersions() : soType.modelVersions;
  return Object.entries(modelVersions).map(([version, modelVersion], index) => {
    if (`${++index}` !== version) {
      throw new Error(`Missing model version '${index}' (10.${index}.0) for SO type ${soType.name}. Please define versions in order and without skipping any version numbers.`);
    }
    const modelVersionHash = getModelVersionHash(modelVersion);
    return `${soType.name}|10.${version}.0: ${modelVersionHash}`;
  });
};
const getModelVersionHash = modelVersion => {
  const hash = (0, _crypto.createHash)('sha256');
  const modelVersionData = JSON.stringify(modelVersion);
  return `${hash.update(modelVersionData).digest('hex')}`;
};
const getMigrationHash = soType => {
  var _migInfo$convertToAli, _migInfo$convertToMul, _migInfo$switchToMode;
  const migInfo = (0, _extract_migration_info.extractMigrationInfo)(soType);
  const hash = (0, _crypto.createHash)('sha256');
  const hashParts = [migInfo.name, migInfo.namespaceType, (_migInfo$convertToAli = migInfo.convertToAliasScript) !== null && _migInfo$convertToAli !== void 0 ? _migInfo$convertToAli : 'none', migInfo.hasExcludeOnUpgrade, (_migInfo$convertToMul = migInfo.convertToMultiNamespaceTypeVersion) !== null && _migInfo$convertToMul !== void 0 ? _migInfo$convertToMul : 'none', migInfo.migrationVersions.join(','), migInfo.schemaVersions.join(','), JSON.stringify(migInfo.mappings, Object.keys(migInfo.mappings).sort()), (_migInfo$switchToMode = migInfo.switchToModelVersionAt) !== null && _migInfo$switchToMode !== void 0 ? _migInfo$switchToMode : 'none', migInfo.modelVersions.map(serializeModelVersion).join(',')];
  const hashFeed = hashParts.join('-');
  return hash.update(hashFeed).digest('hex');
};
exports.getMigrationHash = getMigrationHash;
const serializeModelVersion = modelVersion => {
  const schemas = [modelVersion.schemas.forwardCompatibility, modelVersion.schemas.create];
  return [modelVersion.version, modelVersion.modelVersionHash, modelVersion.changeTypes.join(','), modelVersion.hasTransformation, schemas.join(','), ...modelVersion.newMappings].join('|');
};