"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mergeUserQueryWithNamespacesBool = mergeUserQueryWithNamespacesBool;
exports.performSearch = performSearch;
var _lodash = require("lodash");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _search = require("../search");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function createEmptySearchResponse() {
  return {
    hits: {
      hits: []
    },
    took: 0,
    timed_out: false,
    _shards: {
      total: 0,
      successful: 0,
      failed: 0
    }
  };
}
async function performSearch({
  options
}, {
  registry,
  helpers,
  serializer,
  allowedTypes,
  client,
  extensions = {}
}) {
  const {
    common: commonHelper,
    encryption: encryptionHelper,
    migration: migrationHelper,
    serializer: serializerHelper
  } = helpers;
  const {
    securityExtension,
    spacesExtension
  } = extensions;
  const {
    namespaces: requestedNamespaces,
    type,
    ...esOptions
  } = options;
  if (requestedNamespaces.length === 0) throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('options.namespaces cannot be an empty array');
  if (!type.length) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('options.type must be a string or an array of strings');
  }
  const types = (0, _lodash.castArray)(type).filter(t => allowedTypes.includes(t));
  if (types.length === 0) {
    return createEmptySearchResponse();
  }
  let namespaces;
  try {
    var _await$spacesExtensio;
    namespaces = (_await$spacesExtensio = await (spacesExtension === null || spacesExtension === void 0 ? void 0 : spacesExtension.getSearchableNamespaces(requestedNamespaces))) !== null && _await$spacesExtensio !== void 0 ? _await$spacesExtensio : requestedNamespaces;
  } catch (error) {
    if (_boom.default.isBoom(error) && error.output.payload.statusCode === 403) {
      // The user is not authorized to access any space, return an empty response.
      return createEmptySearchResponse();
    }
    throw error;
  }
  if (namespaces.length === 0) {
    // The user is not authorized to access any of the requested spaces, return an empty response.
    return createEmptySearchResponse();
  }

  // We have to first perform an initial authorization check so that we can construct the search DSL accordingly
  const spacesToAuthorize = new Set(namespaces);
  const typesToAuthorize = new Set(types);
  const authorizationResult = await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeFind({
    namespaces: spacesToAuthorize,
    types: typesToAuthorize
  }));
  if ((authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.status) === 'unauthorized') {
    // If the user is unauthorized to find *anything* they requested, return an empty response
    return createEmptySearchResponse();
  }
  let typeToNamespacesMap;
  if ((authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.status) === 'partially_authorized') {
    typeToNamespacesMap = new Map();
    for (const [objType, entry] of authorizationResult.typeMap) {
      if (!entry.find) continue;
      // This ensures that the query DSL can filter only for object types that the user is authorized to access for a given space
      const {
        authorizedSpaces,
        isGloballyAuthorized
      } = entry.find;
      typeToNamespacesMap.set(objType, isGloballyAuthorized ? namespaces : authorizedSpaces);
    }
  }
  const query = mergeUserQueryWithNamespacesBool(esOptions.query, (0, _search.getNamespacesBoolFilter)({
    namespaces,
    registry,
    types,
    typeToNamespacesMap
  }));
  const result = await client.search({
    ...esOptions,
    // If `pit` is provided, we drop the `index`, otherwise ES returns 400.
    index: esOptions.pit ? undefined : commonHelper.getIndicesForTypes(types),
    query
  }, {
    ignore: [404],
    meta: true
  });
  if (result.statusCode === 404) {
    if (!(0, _coreElasticsearchServerInternal.isSupportedEsServer)(result.headers)) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
    }
    // 404 is only possible here if the index is missing, which
    // we don't want to leak, see "404s from missing index" above
    return createEmptySearchResponse();
  }
  const redactTypeMapParams = {
    previouslyCheckedNamespaces: spacesToAuthorize,
    objects: []
  };
  for (const hit of result.body.hits.hits) {
    if (serializer.isRawSavedObject(hit)) {
      var _namespaces;
      redactTypeMapParams.objects.push({
        type: hit._source.type,
        id: hit._id,
        existingNamespaces: (_namespaces = hit._source.namespaces) !== null && _namespaces !== void 0 ? _namespaces : []
      });
    }
  }
  const redactTypeMap = redactTypeMapParams ? await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.getFindRedactTypeMap(redactTypeMapParams)) : undefined;

  // Migrations and encryption don't work on raw documents. To process them we have
  // to serialize raw documents to saved objects and then deserialize them back to
  // raw documents again.
  const processHit = async hit => {
    var _options$fields;
    if (!serializer.isRawSavedObject(hit)) {
      return hit;
    }
    const savedObject = serializerHelper.rawToSavedObject(hit);
    if ((_options$fields = options.fields) !== null && _options$fields !== void 0 && _options$fields.length) {
      // If the fields argument is set, don't migrate.
      // This document may only contains a subset of it's fields meaning the migration
      // (transform and forwardCompatibilitySchema) is not guaranteed to succeed. We
      // still try to decrypt/redact the fields that are present in the document.
      const decrypted = await encryptionHelper.optionallyDecryptAndRedactSingleResult(savedObject, redactTypeMap);
      return {
        ...hit,
        ...serializer.savedObjectToRaw(decrypted)
      };
    }
    const migratedSavedObject = await migrationHelper.migrateAndDecryptStorageDocument({
      document: savedObject,
      typeMap: redactTypeMap
    });
    return {
      ...hit,
      ...serializer.savedObjectToRaw(migratedSavedObject)
    };
  };
  const processedHits = [];
  for (const hit of result.body.hits.hits) {
    processedHits.push(await processHit(hit));
  }
  result.body.hits.hits = processedHits;
  return result.body;
}
function mergeUserQueryWithNamespacesBool(userQuery, namespacesBoolFilter) {
  const must = [namespacesBoolFilter];
  if (userQuery) {
    must.push(userQuery);
  }
  return {
    bool: {
      must
    }
  };
}