"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.URLVoidConnector = void 0;
var _zod = require("@kbn/zod");
var _connector_spec_ui = require("../connector_spec_ui");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Example: URLVoid Connector
 *
 * This demonstrates a domain reputation connector with:
 * - Domain reputation scanning
 * - URL safety checking
 * - Domain information retrieval
 * - API usage statistics
 *
 * MVP implementation focusing on core domain reputation actions.
 */

const URLVoidConnector = exports.URLVoidConnector = {
  metadata: {
    id: '.urlvoid',
    displayName: 'URLVoid',
    description: 'Domain and URL reputation checking via multi-engine scanning',
    minimumLicense: 'gold',
    supportedFeatureIds: ['alerting', 'siem']
  },
  schema: _zod.z.discriminatedUnion('method', [_zod.z.object({
    method: _zod.z.literal('headers'),
    headers: _zod.z.object({
      'X-Api-Key': _connector_spec_ui.UISchemas.secret().describe('API Key')
    })
  })]),
  actions: {
    scanDomain: {
      isTool: true,
      input: _zod.z.object({
        domain: _zod.z.string().describe('Domain name to scan')
      }),
      handler: async (ctx, input) => {
        const typedInput = input;
        const apiKey = ctx.auth.method === 'headers' ? ctx.auth.headers['X-Api-Key'] : '';
        const response = await ctx.client.get(`https://api.urlvoid.com/api1000/${apiKey}/host/${typedInput.domain}`);
        return {
          domain: typedInput.domain,
          reputation: response.data.reputation,
          detections: response.data.detections,
          engines: response.data.engines
        };
      }
    },
    checkUrl: {
      isTool: true,
      input: _zod.z.object({
        url: _zod.z.string().url().describe('URL to check')
      }),
      handler: async (ctx, input) => {
        const typedInput = input;
        const domain = new URL(typedInput.url).hostname;
        const apiKey = ctx.auth.method === 'headers' ? ctx.auth.headers['X-Api-Key'] : '';
        const response = await ctx.client.get(`https://api.urlvoid.com/api1000/${apiKey}/host/${domain}`);
        return {
          url: typedInput.url,
          domain,
          reputation: response.data.reputation,
          detections: response.data.detections,
          engines: response.data.engines
        };
      }
    },
    getDomainInfo: {
      isTool: true,
      input: _zod.z.object({
        domain: _zod.z.string().describe('Domain name')
      }),
      handler: async (ctx, input) => {
        const typedInput = input;
        const apiKey = ctx.auth.method === 'headers' ? ctx.auth.headers['X-Api-Key'] : '';
        const response = await ctx.client.get(`https://api.urlvoid.com/api1000/${apiKey}/host/${typedInput.domain}`);
        return {
          domain: typedInput.domain,
          reputation: response.data.reputation,
          ip: response.data.ip,
          country: response.data.country,
          registrar: response.data.registrar,
          created: response.data.domain_age,
          detections: response.data.detections
        };
      }
    },
    scanDomainStats: {
      isTool: true,
      input: _zod.z.object({}),
      handler: async ctx => {
        const apiKey = ctx.auth.method === 'headers' ? ctx.auth.headers['X-Api-Key'] : '';
        const response = await ctx.client.get(`https://api.urlvoid.com/api1000/${apiKey}/stats/remained`);
        return {
          queriesRemaining: response.data.queries_remaining,
          queriesUsed: response.data.queries_used,
          plan: response.data.plan
        };
      }
    }
  },
  test: {
    handler: async ctx => {
      try {
        const apiKey = ctx.auth.method === 'headers' ? ctx.auth.headers['X-Api-Key'] : '';
        await ctx.client.get(`https://api.urlvoid.com/api1000/${apiKey}/stats/remained`);
        return {
          ok: true,
          message: 'Successfully connected to URLVoid API'
        };
      } catch (error) {
        return {
          ok: false,
          message: `Failed to connect: ${error}`
        };
      }
    },
    description: 'Verifies URLVoid API key'
  }
};