"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.update = update;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _preconfigured_action_disabled_modification = require("../../../../lib/errors/preconfigured_action_disabled_modification");
var _audit_events = require("../../../../lib/audit_events");
var _lib = require("../../../../lib");
var _lib2 = require("../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function update({
  context,
  id,
  action
}) {
  var _actionType$validate, _context$auditLogger3;
  try {
    await context.authorization.ensureAuthorized({
      operation: 'update'
    });
    const foundInMemoryConnector = context.inMemoryConnectors.find(connector => connector.id === id);
    if (foundInMemoryConnector !== null && foundInMemoryConnector !== void 0 && foundInMemoryConnector.isSystemAction) {
      throw _boom.default.badRequest(_i18n.i18n.translate('xpack.actions.serverSideErrors.systemActionUpdateForbidden', {
        defaultMessage: 'System action {id} can not be updated.',
        values: {
          id
        }
      }));
    }
    if (foundInMemoryConnector !== null && foundInMemoryConnector !== void 0 && foundInMemoryConnector.isPreconfigured) {
      throw new _preconfigured_action_disabled_modification.PreconfiguredActionDisabledModificationError(_i18n.i18n.translate('xpack.actions.serverSideErrors.predefinedActionUpdateDisabled', {
        defaultMessage: 'Preconfigured action {id} can not be updated.',
        values: {
          id
        }
      }), 'update');
    }
  } catch (error) {
    var _context$auditLogger;
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.connectorAuditEvent)({
      action: _audit_events.ConnectorAuditAction.UPDATE,
      savedObject: {
        type: 'action',
        id
      },
      error
    }));
    throw error;
  }
  const {
    attributes,
    references,
    version
  } = await context.unsecuredSavedObjectsClient.get('action', id);
  const {
    actionTypeId
  } = attributes;
  const {
    name,
    config,
    secrets
  } = action;
  const actionType = context.actionTypeRegistry.get(actionTypeId);
  const configurationUtilities = context.actionTypeRegistry.getUtils();
  const validatedActionTypeConfig = (0, _lib.validateConfig)(actionType, config, {
    configurationUtilities
  });
  const validatedActionTypeSecrets = (0, _lib.validateSecrets)(actionType, secrets, {
    configurationUtilities
  });
  if ((_actionType$validate = actionType.validate) !== null && _actionType$validate !== void 0 && _actionType$validate.connector) {
    (0, _lib.validateConnector)(actionType, {
      config,
      secrets
    });
  }
  context.actionTypeRegistry.ensureActionTypeEnabled(actionTypeId);
  const hookServices = {
    scopedClusterClient: context.scopedClusterClient
  };
  if (actionType.preSaveHook) {
    try {
      await actionType.preSaveHook({
        connectorId: id,
        config,
        secrets,
        logger: context.logger,
        request: context.request,
        services: hookServices,
        isUpdate: true
      });
    } catch (error) {
      var _context$auditLogger2;
      (_context$auditLogger2 = context.auditLogger) === null || _context$auditLogger2 === void 0 ? void 0 : _context$auditLogger2.log((0, _audit_events.connectorAuditEvent)({
        action: _audit_events.ConnectorAuditAction.UPDATE,
        savedObject: {
          type: 'action',
          id
        },
        error
      }));
      throw error;
    }
  }
  (_context$auditLogger3 = context.auditLogger) === null || _context$auditLogger3 === void 0 ? void 0 : _context$auditLogger3.log((0, _audit_events.connectorAuditEvent)({
    action: _audit_events.ConnectorAuditAction.UPDATE,
    savedObject: {
      type: 'action',
      id
    },
    outcome: 'unknown'
  }));
  const result = await (0, _lib.tryCatch)(async () => await context.unsecuredSavedObjectsClient.create('action', {
    ...attributes,
    actionTypeId,
    name,
    isMissingSecrets: false,
    config: validatedActionTypeConfig,
    secrets: validatedActionTypeSecrets
  }, (0, _lodash.omitBy)({
    id,
    overwrite: true,
    references,
    version
  }, _lodash.isUndefined)));
  const wasSuccessful = !(result instanceof Error);
  const label = `connectorId: "${id}"; type: ${actionTypeId}`;
  const tags = ['post-save-hook', id];
  if (actionType.postSaveHook) {
    try {
      await actionType.postSaveHook({
        connectorId: id,
        config,
        secrets,
        logger: context.logger,
        request: context.request,
        services: hookServices,
        isUpdate: true,
        wasSuccessful
      });
    } catch (err) {
      context.logger.error(`postSaveHook update error for ${label}: ${err.message}`, {
        tags
      });
    }
  }
  if (!wasSuccessful) {
    throw result;
  }
  try {
    await context.connectorTokenClient.deleteConnectorTokens({
      connectorId: id
    });
  } catch (e) {
    context.logger.error(`Failed to delete auth tokens for connector "${id}" after update: ${e.message}`);
  }
  return {
    id,
    actionTypeId: result.attributes.actionTypeId,
    isMissingSecrets: result.attributes.isMissingSecrets,
    name: result.attributes.name,
    config: result.attributes.config,
    isPreconfigured: false,
    isSystemAction: false,
    isDeprecated: (0, _lib2.isConnectorDeprecated)(result.attributes),
    isConnectorTypeDeprecated: context.actionTypeRegistry.isDeprecated(actionTypeId)
  };
}