"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineInvalidateSessionsRoutes = defineInvalidateSessionsRoutes;
var _configSchema = require("@kbn/config-schema");
var _coreHttpServer = require("@kbn/core-http-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Defines routes required for session invalidation.
 */
function defineInvalidateSessionsRoutes({
  router,
  getSession,
  buildFlavor
}) {
  router.post({
    path: '/api/security/session/_invalidate',
    validate: {
      body: _configSchema.schema.object({
        match: _configSchema.schema.oneOf([_configSchema.schema.literal('all'), _configSchema.schema.literal('query')]),
        query: _configSchema.schema.conditional(_configSchema.schema.siblingRef('match'), _configSchema.schema.literal('query'), _configSchema.schema.object({
          provider: _configSchema.schema.object({
            type: _configSchema.schema.string(),
            name: _configSchema.schema.maybe(_configSchema.schema.string())
          }),
          username: _configSchema.schema.maybe(_configSchema.schema.string())
        }), _configSchema.schema.never())
      })
    },
    security: {
      authz: {
        requiredPrivileges: [_coreHttpServer.ReservedPrivilegesSet.superuser]
      }
    },
    options: {
      // The invalidate session API was introduced to address situations where the session index
      // could grow rapidly - when session timeouts are disabled, or with anonymous access.
      // In the serverless environment, sessions timeouts are always be enabled, and there is no
      // anonymous access. However, keeping this endpoint available internally in serverless would
      // be useful in situations where we need to batch-invalidate user sessions.
      access: buildFlavor === 'serverless' ? 'internal' : 'public',
      summary: `Invalidate user sessions`
    }
  }, async (_context, request, response) => {
    return response.ok({
      body: {
        total: await getSession().invalidate(request, {
          match: request.body.match,
          query: request.body.query
        })
      }
    });
  });
}