"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OBSERVABILITY_SEARCH_KNOWLEDGE_BASE_TOOL_ID = void 0;
exports.createSearchKnowledgeBaseTool = createSearchKnowledgeBaseTool;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _gptTokenizer = require("gpt-tokenizer");
var _lodash = require("lodash");
var _common = require("@kbn/spaces-plugin/common");
var _get_access_query = require("./get_access_query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const OBSERVABILITY_SEARCH_KNOWLEDGE_BASE_TOOL_ID = exports.OBSERVABILITY_SEARCH_KNOWLEDGE_BASE_TOOL_ID = 'observability.search_knowledge_base';

// Knowledge base constants based on observability AI assistant
const OBSERVABILITY_KB_INDEX_ALIAS = '.kibana-observability-ai-assistant-kb';
var KnowledgeBaseType = /*#__PURE__*/function (KnowledgeBaseType) {
  KnowledgeBaseType["UserInstruction"] = "user_instruction";
  KnowledgeBaseType["Contextual"] = "contextual";
  return KnowledgeBaseType;
}(KnowledgeBaseType || {});
const searchKnowledgeBaseSchema = _zod.z.object({
  query: _zod.z.string().min(1).describe('One focused semantic search question that restates the latest user request together with essential conversation context, preserving exact identifiers and key entities.')
});
function createSearchKnowledgeBaseTool({
  core,
  logger
}) {
  const toolDefinition = {
    id: OBSERVABILITY_SEARCH_KNOWLEDGE_BASE_TOOL_ID,
    type: _onechatCommon.ToolType.builtin,
    description: `Search the observability knowledge base for custom organizational knowledge, and user-specific information. This contains specialized content not accessible through other search tools, including personal user details, preferences, and context from previous interactions. Use when built-in tools don't have the needed information or when queries involve custom organizational policies, procedures, domain-specific knowledge, or personal user information not available in standard indices.`,
    schema: searchKnowledgeBaseSchema,
    tags: ['observability', 'knowledge', 'documentation', 'context'],
    handler: async ({
      query
    }, {
      esClient,
      request
    }) => {
      const [, plugins] = await core.getStartServices();
      const user = plugins.security.authc.getCurrentUser(request);
      try {
        const normalizedQuery = query.trim();
        logger.debug(`Searching knowledge base with query: "${normalizedQuery}"`);
        const namespace = await getNamespaceForRequest(request, core);
        logger.debug(`Using namespace: ${namespace}`);
        const entries = await searchKnowledgeBase({
          user,
          esClient,
          query: normalizedQuery,
          namespace,
          logger
        });
        if (entries.length === 0) {
          logger.debug('No knowledge base entries found');
          return {
            results: [{
              type: _tool_result.ToolResultType.other,
              data: {
                message: 'No knowledge base entries found for the given query.',
                entries: []
              }
            }]
          };
        }
        logger.debug(`Found ${entries.length} knowledge base entries`);
        const filteredEntries = filterEntriesByTokenLimit(entries, logger);
        return {
          results: [{
            type: _tool_result.ToolResultType.other,
            data: {
              query: normalizedQuery,
              entries: filteredEntries,
              total: entries.length,
              noOfDroppedEntries: entries.length - filteredEntries.length
            }
          }]
        };
      } catch (error) {
        logger.error(`Error searching from knowledge base: ${error.message}`);
        logger.debug(error);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to searching from knowledge base: ${error.message}`,
              stack: error.stack
            }
          }]
        };
      }
    }
  };
  return toolDefinition;
}

/**
 * Query the knowledge base using semantic search
 */
async function searchKnowledgeBase({
  user,
  esClient,
  query,
  namespace,
  logger
}) {
  try {
    const response = await esClient.asInternalUser.search({
      index: [OBSERVABILITY_KB_INDEX_ALIAS],
      query: {
        bool: {
          should: [{
            semantic: {
              field: 'semantic_text',
              query
            }
          }],
          filter: [...(0, _get_access_query.getAccessQuery)({
            user,
            namespace
          }), {
            bool: {
              must_not: {
                term: {
                  type: KnowledgeBaseType.UserInstruction
                }
              }
            }
          } // exclude user instructions
          ]
        }
      },
      size: 10,
      _source: {
        includes: ['text', 'labels', 'doc_id', 'title']
      }
    });
    return response.hits.hits.map(hit => {
      var _hit$_source$text, _hit$_source, _hit$_source$title, _hit$_source2, _hit$_score, _hit$_id;
      return {
        text: (_hit$_source$text = (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : _hit$_source.text) !== null && _hit$_source$text !== void 0 ? _hit$_source$text : '',
        title: (_hit$_source$title = (_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : _hit$_source2.title) !== null && _hit$_source$title !== void 0 ? _hit$_source$title : '',
        esScore: (_hit$_score = hit._score) !== null && _hit$_score !== void 0 ? _hit$_score : 0,
        id: (_hit$_id = hit._id) !== null && _hit$_id !== void 0 ? _hit$_id : ''
      };
    });
  } catch (error) {
    var _error$meta;
    // return empty array if index doesn't exist
    if (((_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : _error$meta.statusCode) === 404) {
      logger.debug('Knowledge base index does not exist');
      return [];
    }
    throw error;
  }
}

/**
 * Filter entries by token limit to avoid exceeding context window
 */
const MAX_TOKENS = 4000;
function filterEntriesByTokenLimit(entries, logger) {
  const sortedEntries = (0, _lodash.orderBy)(entries, 'esScore', 'desc');
  let tokenCount = 0;
  const returnedEntries = [];
  for (const entry of sortedEntries) {
    const entryTokens = (0, _gptTokenizer.encode)(entry.text).length;
    const isAboveTokenLimit = tokenCount + entryTokens > MAX_TOKENS;
    if (isAboveTokenLimit) {
      break;
    }
    returnedEntries.push(entry);
    tokenCount += entryTokens;
  }
  const droppedCount = sortedEntries.length - returnedEntries.length;
  if (droppedCount > 0) {
    logger.debug(`Dropped ${droppedCount} entries due to token limit (${MAX_TOKENS} tokens)`);
  }
  return returnedEntries;
}

// Get the current namespace from the request
async function getNamespaceForRequest(request, core) {
  const [coreStart] = await core.getStartServices();
  const {
    serverBasePath
  } = coreStart.http.basePath;
  const basePath = coreStart.http.basePath.get(request);
  const {
    spaceId
  } = (0, _common.getSpaceIdFromPath)(basePath, serverBasePath);
  return spaceId;
}